/***************************************************************************
 *   Copyright (C) 2001-2005 AVM GmbH. All rights reserved.                *
 *   Copyright (C) 1998-2000 Markus Dahlweid                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kimoninfoline.h"
#include "kimoninfoline.moc"

#include "kimon.h"
#include <stdio.h>
#include <qfile.h>
#undef Unsorted
#include <qdir.h>
#include <qtextstream.h>
#include <qmessagebox.h>
#include <klocale.h>
#include <kglobal.h>

// layout
int  KIsdnInfoLine::m_iX             = 10;
int  KIsdnInfoLine::m_iControllerCX  = 60;
int  KIsdnInfoLine::m_iChannelCX     = 60;
int  KIsdnInfoLine::m_iStatusCX      = 60;
int  KIsdnInfoLine::m_iTimeCX        = 60;
const int  KIsdnInfoLine::m_iLedCX   = 14;
const int  KIsdnInfoLine::m_iSpaceCX = 6;
const int  KIsdnInfoLine::m_iY       = 3;
int  KIsdnInfoLine::m_iCY            = 20;
const int  KIsdnInfoLine::m_iLedCY   = 14;

bool KIsdnInfoLine::m_bShowLogfileError = true;

const unsigned long MAX_LOGFILESIZE = 1024 * 1024;

#ifndef max
#define max(a,b) (((a) > (b)) ? (a) : (b))
#endif

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
KIsdnInfoLine::KIsdnInfoLine (QWidget* parent, bool panel, const char* name)
						 : QWidget(parent, name)
{
    _upsince = QDateTime::currentDateTime ();
    
    _panel = panel;
    _onl = false;
    
    m_pLabelController = new QLabel (this);
    if ( _panel ) m_pLabelController->setFrameStyle (QFrame::Panel|QFrame::Sunken);
    m_pLabelController->setAlignment (AlignHCenter | AlignVCenter);
    m_pLabelController->setText ("");
    m_pLabelController->show ();

    m_pLabelChannel = new QLabel (this);
    if ( _panel ) m_pLabelChannel->setFrameStyle (QFrame::Panel|QFrame::Sunken);
    m_pLabelChannel->setAlignment (AlignHCenter | AlignVCenter);
    m_pLabelChannel->setText ("");
    m_pLabelChannel->show ();
    
    m_pLabelStatus = new QLabel (this);
    if ( _panel ) m_pLabelStatus->setFrameStyle (QFrame::Panel|QFrame::Sunken);
    m_pLabelStatus->setAlignment (AlignHCenter|AlignVCenter);
    m_pLabelStatus->setText (i18n ("Offline"));
    m_pLabelStatus->show ();
    
    m_pLabelTime = new QLabel (this);
    if ( _panel ) m_pLabelTime->setFrameStyle( QFrame::Panel|QFrame::Sunken);
    m_pLabelTime->setAlignment (AlignHCenter|AlignVCenter);
    m_pLabelTime->setText ("00:00:00");
    m_pLabelTime->show ();

    if ( _panel )
    {
      	// This is necessary because of a bug in kled.h
      	m_pWidgetLed = new QWidget (this);
      	m_pWidgetLed->setGeometry (m_iX + m_iControllerCX + m_iChannelCX + m_iStatusCX +
                                   m_iTimeCX + 4 * m_iSpaceCX,
                                   m_iY + 3, m_iLedCX, m_iLedCY);
      	m_pLed = new KIsdnInfoLineStatus (m_pWidgetLed);
      	m_pLed->setFixedSize (m_iLedCX, m_iLedCY);
      	m_pLed->show ();
      	m_pLed->on ();
      	m_pWidgetLed->show ();
    }
    else
    {
    	m_pLabelController->setText (i18n ("Controller"));
    	m_pLabelChannel->setText (i18n ("B Channel"));
    	m_pLabelStatus->setText (i18n ("Status"));
    	m_pLabelTime->setText (i18n ("Time"));
    		
        m_pWidgetLed = NULL;
        m_pLed = NULL;
    }
    UpdateLayout ();
}

KIsdnInfoLine::~KIsdnInfoLine()
{
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfoLine::Log (ELogEvent LogEvent)
{
    QDateTime timeUpto = QDateTime::currentDateTime ();

    QString strLogfile = QDir::homeDirPath ();
    if ( strLogfile.right (1)  == "/" )
        strLogfile += "kisdnwatch.log";
    else
        strLogfile += "/kisdnwatch.log";

    // open/create logfile
    QFile Logfile (strLogfile);
    if ( Logfile.size () >= MAX_LOGFILESIZE )
    {
        QString strBakfile = strLogfile + ".bak";
        remove (strBakfile.latin1 ());
        rename (strLogfile.latin1 (), strBakfile.latin1 ());
    }
    if ( !Logfile.open (IO_WriteOnly | IO_Append) )
    {
        if ( m_bShowLogfileError )
        {
            m_bShowLogfileError = false;
            QMessageBox::warning (this, i18n (ErrTitle), i18n ("Cannot open/create %1").arg (strLogfile));
        }
        return;
    }
    m_bShowLogfileError = true;
    QTextStream LogStream (&Logfile);
    QString strUpto = KGlobal::locale ()->formatDateTime (timeUpto, true, true);

    // append log entry
    switch ( LogEvent )
    {
        case log_onstart:
            
            LogStream << strUpto << i18n (" Starting %1 with controller %2 channel %3 online!")
                                            .arg (i18n (AppTitle))
                                            .arg (m_pLabelController->text ())
                                            .arg (m_pLabelChannel->text ()) << endl;
            break;

        case log_onexit:
            LogStream << strUpto << i18n (" Closing %1 with controller %2 channel %3 online!")
                                            .arg (i18n (AppTitle))
                                            .arg (m_pLabelController->text ())
                                            .arg (m_pLabelChannel->text ()) << endl;

        case log_ondisc:
            {
                QString strUpsince = KGlobal::locale ()->formatDateTime (_upsince, true, true);
                int Duration = _upsince.secsTo (timeUpto);
                int Sec = Duration % 60;
                int Min = Duration / 60;
                int Hour = Min / 60;
                Min = Min % 60;
                QString strTime;
                strTime.sprintf ("%02d:%02d:%02d", Hour, Min, Sec);

                LogStream << i18n ("%1 to %2 online %3 controller %4 channel %5")
                                .arg (strUpsince)
                                .arg (strUpto)
                                .arg (strTime)
                                .arg (m_pLabelController->text ())
                                .arg (m_pLabelChannel->text ()) << endl;
            }
            break;
    }; // switch LogEvent

    Logfile.close ();
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfoLine::UpdateLayout ()
{
    int iX = m_iX;

    QSize size = m_pLabelController->sizeHint ();
    m_iControllerCX = max (m_iControllerCX, size.width ());
    
    m_iCY = max (m_iCY, size.height ());
    size = m_pLabelChannel->sizeHint ();
    m_iChannelCX = max (m_iChannelCX, size.width ());
    m_iCY = max (m_iCY, size.height ());
    size = m_pLabelStatus->sizeHint ();
    m_iStatusCX = max (m_iStatusCX, size.width ());
    m_iCY = max (m_iCY, size.height ());
    size = m_pLabelTime->sizeHint ();
    m_iTimeCX = max (m_iTimeCX, size.width ());
    m_iCY = max (m_iCY, size.height ());

    m_pLabelController->setFixedSize (m_iControllerCX, m_iCY);
    m_pLabelController->move (iX, m_iY);
    iX += m_iControllerCX + m_iSpaceCX;

    m_pLabelChannel->setFixedSize (m_iChannelCX, m_iCY);
    m_pLabelChannel->move (iX, m_iY);
    iX += m_iChannelCX + m_iSpaceCX;

    m_pLabelStatus->setFixedSize(m_iStatusCX, m_iCY);
    m_pLabelStatus->move (iX, m_iY);
    iX += m_iStatusCX + m_iSpaceCX;

    m_pLabelTime->setFixedSize (m_iTimeCX, m_iCY);
    m_pLabelTime->move (iX, m_iY);
    iX += m_iTimeCX + m_iSpaceCX;

    if ( m_pWidgetLed != NULL )
        m_pWidgetLed->setGeometry (iX, m_iY + max (3, (m_iCY - m_iLedCY) / 2), m_iLedCX, m_iLedCY);
    iX += m_iLedCX + m_iSpaceCX;

    setFixedSize (iX + 5, m_iCY + 3);
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfoLine::set_controller (int iController)
{
    m_pLabelController->setNum (iController);
}


/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfoLine::set_channel (int iChannel)
{
    m_pLabelChannel->setNum (iChannel);
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfoLine::set_online (int dir, bool bFirst)
{
    if ( dir != _onl )
    {
        if ( dir == 0 )
        {
	    m_pLed->setColor(KLed::darkGray);
            m_pLabelStatus->setText (i18n ("Offline"));
            _onl = dir;
            Log (log_ondisc);
	}
        else
        {
            reset_time ();
            m_pLed->setColor(KLed::green);
            m_pLabelStatus->setText (i18n ("Online"));
            _onl = dir;

            if ( bFirst ) Log (log_onstart);
        }
    }
    recalc_time ();
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfoLine::set_time (QString strTime)
{
    m_pLabelTime->setText (strTime);
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfoLine::reset_time ()
{
    _upsince = QDateTime::currentDateTime ();
    set_time (QString ("00:00:00"));
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfoLine::recalc_time ()
{
    if ( _onl != 0 )
    {
        int iDuration = _upsince.secsTo (QDateTime::currentDateTime ());
        int iSec = iDuration % 60;
        int iMin = iDuration / 60;
        int iHou = iMin / 60;
        iMin = iMin % 60;
        QString strTime;
        strTime.sprintf ("%02d:%02d:%02d", iHou, iMin, iSec);
        set_time (strTime);
    }
    else set_time (QString ("00:00:00"));
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfoLine::showEvent (QShowEvent*)
{
    UpdateLayout ();
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfoLine::styleChange (QStyle& style)
{
    QWidget::styleChange (style);
    UpdateLayout ();
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfoLine::destroy (bool destroyWindow, bool destroySubWindows)
{
    if ( _onl != 0 )
    {
        _onl = 0;
        Log (log_onexit);
    }
    QWidget::destroy (destroyWindow, destroySubWindows);
}


/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
QString KIsdnInfoLine::GetInfoTip () const
{
    QString strInfo;
    if ( _onl != 0 )
        strInfo = i18n ("%1/%2: %3").arg (m_pLabelController->text ())
                                    .arg (m_pLabelChannel->text ())
                                    .arg (m_pLabelTime->text ());
    return strInfo;
}
