/***************************************************************************
 *   Copyright (C) 2001-2005 AVM GmbH. All rights reserved.                *
 *   Copyright (C) 1998-2000 Markus Dahlweid                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kimon.h"
#include "config.h"
#include "kimoninfo.h"
#include "kimoninfoline.h"
#include "kimoninfostatus.h"
#include "kimonprefs.h"
#include "kcontrollerprofile.h"
#include "kavmmontabdialog.h"
#include "kimon.moc"

#include <kmainwindow.h>
#include <kwin.h>
#include <klocale.h>
#include <kcmdlineargs.h>
#include <kaboutdata.h>
#include <kconfig.h>
#include <kurllabel.h>

#include <qpainter.h>
#include <qwidget.h>
#include <qmessagebox.h>
#include <qtooltip.h>
#include <qlayout.h>
#include <qcursor.h>
#include <netwm.h>

#include <stdio.h>
#include <stdlib.h>

#include "../img/out_of_order.xpm"
#include "../img/b_00.xpm"
#include "../img/b_01.xpm"
#include "../img/b_10.xpm"
#include "../img/b_11.xpm"


KConfig  *kimonConf;
KIsdnMonApplication* pApp = NULL;
bool     sticky;
const char* AppTitle = I18N_NOOP ("K ISDN Watch");
const char* ErrTitle = I18N_NOOP ("K ISDN Watch error");

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
KIsdnMon::KIsdnMon (QWidget *parent, const char *name, WFlags f)
        : QWidget (parent, name, f)
{
    m_pIInfo = NULL;
    retry = true;

    // retrieve the config object
    kimonConf = KGlobal::config();
    kimonConf->setGroup ("Behaviour");
    autohideTimeout = -1;
    enableTimeout = false;
    m_iOpenOnChange = CCapiInfo::line_nop;
    sticky = true;

    tid = startTimer (1000);	/* this is default (1 sec) */

    m_pMenu = new KPopupMenu;
    m_pMenu->insertTitle (i18n (AppTitle));
    m_pMenu->insertItem (i18n ("Open"), this, SLOT (open_notimeout ()));
    m_pMenu->insertItem (i18n ("CAPI Info..."), this, SLOT (profiles ()));
    m_pMenu->insertItem (i18n ("Preferences..."), this, SLOT (prefs ()));
    m_pMenu->insertSeparator ();
    m_pMenu->insertItem (i18n ("Quit"), this, SLOT (quit ()));
    autohideTimer = 0;
    closed = true;
#ifdef CPROT  	
  	m_CapiInfo.OpenProtocol ("/tmp/kisdnwatch.debug");
#endif
}

KIsdnMon::~KIsdnMon ()
{
#ifdef CPROT  	
  	m_CapiInfo.CloseProtocol ();
#endif
    killTimer (tid);
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnMon::SetToolTip (QString strToolTip)
{
    QToolTip::add (this, (strToolTip.isEmpty ()) ? i18n (AppTitle) : strToolTip);
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnMon::SetSystemTrayWindow ()
{
    KWin::setSystemTrayWindowFor (winId (), 0);
    setBackgroundMode (X11ParentRelative);
    setBackgroundOrigin (WindowOrigin);
    show ();
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnMon::parseConfig ()
{
    m_iOpenOnChange = kimonConf->readNumEntry ("OpenOnCall", CCapiInfo::line_up);
    autohideTimeout = kimonConf->readNumEntry ("AutohideTimeout", 30);
    if ( 2 > autohideTimeout ) autohideTimeout = 2;
    enableTimeout = kimonConf->readBoolEntry ("EnableAutohide", true);
    sticky = kimonConf->readBoolEntry ("BeSticky", true);
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnMon::paintEvent (QPaintEvent *)
{
    QPainter p;
    p.begin (this);
    p.setWindow (0, 0, 24, 24);

    const char** ppPixmap = out_of_order;
		if ( m_CapiInfo.IsAvmCapi () && !m_CapiInfo.IsDslOnly () )
		{
				if ( (m_CapiInfo.ControllerCount () > 1) || (m_CapiInfo.GetChannelCount (1) > 2) )
				{
						// more than two b channels
						switch ( m_CapiInfo.GetChannelCount (true, 0) )
						{
								case 0:
										ppPixmap = b_00;
										break;
								case 1:
										ppPixmap = b_10;
										break;
								default:
										ppPixmap = b_11;
						}; // switch
				} // if ControllerCount
				else
				{
						// up to two b channels
						if ( !m_CapiInfo.IsBChannelUp (1, 0) )
						{
								if ( !m_CapiInfo.IsBChannelUp (1, 1) )
										ppPixmap = b_00;
								else
										ppPixmap = b_01;
						}
						else
						{
								if ( !m_CapiInfo.IsBChannelUp (1, 1) )
										ppPixmap = b_10;
								else
										ppPixmap = b_11;
						}
				} // if/else ControllerCount
		} // if IsAvmCapi

    p.drawPixmap (4, 0, QPixmap (ppPixmap));
    p.end();
}

/*--------------------------------------------------------------------------*\
		this function reads the necessary information from CAPI
\*--------------------------------------------------------------------------*/
void KIsdnMon::readInfo ()
{
		// the first time
		if ( autohideTimeout < 0 ) parseConfig ();

    /* as long as we don't need to retry */
    if ( !retry ) return;

		if ( !m_CapiInfo.IsInit () )
    {
    		// the first time - say hello
    		QString strError;
    		unsigned int nLastError = m_CapiInfo.GetLastError ();
    		if ( !m_CapiInfo.Register (strError) &&									// error
    				 (nLastError != m_CapiInfo.GetLastError ()) )				// new error
        		fprintf(stderr, "%s\n", strError.latin1());
    }

    // it only works on AVM's CAPI: get new data
    if ( m_CapiInfo.IsAvmCapi () ) m_CapiInfo.Update ();

    /* create IInfo. This is done only once, but it has to be done
       here, cause we have to know the number of channels */
    if ( m_CapiInfo.IsInit () &&  (m_pIInfo == NULL)  ) m_pIInfo = new KIsdnInfo (m_CapiInfo);

    if ( m_pIInfo != NULL )
    {
   	    m_pIInfo->Update (m_CapiInfo);
        SetToolTip  (m_pIInfo->GetInfoTip ());
      	if ( (m_CapiInfo.TestLineEvent () & m_iOpenOnChange) != 0 ) open ();
    }

    repaint ();
}
    
/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnMon::mousePressEvent (QMouseEvent* e)
{
    if ( e->button () == RightButton )
    {
      	m_pMenu->popup (QCursor::pos (), 6);
      	m_pMenu->exec ();
    }
    else if ( e->button () == LeftButton )
    {
        if ( (m_pIInfo != NULL) && m_pIInfo->isVisible () )
            close ();
        else
            open (true);
    }
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnMon::timerEvent (QTimerEvent *)
{
    readInfo ();

    if ( enableTimeout && !closed )
    {
       	autohideTimer++;
       	if ( m_pIInfo != NULL ) m_pIInfo->setCaption (i18n ("%1 (Autohide in: %2 sec)").arg (i18n (AppTitle)).arg (autohideTimeout - autohideTimer));
    }
    else
    {
       	if ( m_pIInfo != NULL ) m_pIInfo->setCaption (i18n ("%1 (Autohide disabled)").arg (i18n (AppTitle)));
    }
    if ( enableTimeout && !closed && (autohideTimer >= autohideTimeout) )
    {
       	close ();
       	closed = true;
       	autohideTimer = 0;
    }
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnMon::open (bool bNoTimeout)
{
    if ( m_CapiInfo.IsInit () )
    {
        if ( m_CapiInfo.IsAvmCapi () )
        {
            if ( (m_pIInfo != NULL) && !m_pIInfo->isVisible () )
            {
                // dont grab keyboard focus if autopopup
                XWMHints xwmHints;
                xwmHints.flags = InputHint;
                xwmHints.input = bNoTimeout;
                XSetWMHints (qt_xdisplay (),m_pIInfo->winId (),&xwmHints);
          			
                m_pIInfo->show ();
                m_pIInfo->raise ();
                autohideTimer = 0;
                if ( !bNoTimeout )
                {
                    closed = false;
                    if ( sticky )
                        KWin::setState (m_pIInfo->winId (), NET::Sticky);
                    else
                        KWin::setState (m_pIInfo->winId (), 0);
                }
                else closed = true;
            }
        }
        else QMessageBox::warning (this, i18n (ErrTitle), i18n ("Sorry, this only works with AVM's CAPI driver!"));
    }
    else
    {
    		// show error again
        QString strError;
        if ( m_CapiInfo.GetLastError (strError) != 0 )
            QMessageBox::warning (this, i18n (ErrTitle), strError);
    }
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnMon::close ()
{
    if ( m_pIInfo != NULL ) m_pIInfo->hide ();
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnMon::quit ()
{
    kimonConf->sync ();
    kapp->quit ();
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnMon::apply ()
{
    /* init the DataSource, cause it could have changed */
    kimonConf->sync ();
    parseConfig ();
    repaint ();

    /* config has changed -> retry to open device */
    retry = true;
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnMon::prefs ()
{
    QString kimon_version;
    KAvmmonTabDialog *pTabDlg = new KAvmmonTabDialog;

    // info page
    QWidget *pPage = new QWidget (pTabDlg);
    QGridLayout* g = new QGridLayout (pPage, 6, 1, 10);
    g->setColStretch (0, 0 );
    g->setRowStretch (0, 1);
    QLabel* pLabel = new QLabel (pPage);
    pLabel->setMinimumSize (100,25);
    pLabel->setFont (QFont ("Helvetica", 20, QFont::Bold));
    pLabel->setText (i18n (AppTitle));
    g->addWidget (pLabel, 1, 0);
    pLabel = new QLabel (pPage);
    pLabel->setAutoResize (true);
    kimon_version = i18n ("Version %1").arg (VERSION);
    pLabel->setText (kimon_version);
    g->addWidget (pLabel, 2, 0);
    pLabel = new QLabel (pPage);
    pLabel->setAutoResize (true);
    pLabel->setText (i18n ("an AVM-CAPI-monitor for the K Desktop Environment 3.2"));
    g->addWidget (pLabel, 3, 0);
    pLabel = new QLabel (pPage);
    pLabel->setAutoResize (true);
    pLabel->setText (i18n ("by AVM Berlin (info@avm.de) based on"));
    g->addWidget (pLabel, 4, 0);
    pLabel = new QLabel (pPage);
    pLabel->setAutoResize (true);
    pLabel->setText (i18n ("kimon by Markus Dahlweid (markus@dahlweid.com)"));
    g->addWidget (pLabel, 5, 0);
    KURLLabel* pUrlLabel = new KURLLabel (pPage);
    pUrlLabel->setAutoResize (true);
    pUrlLabel->setText ("http://www.avm.de");
    pUrlLabel->setURL (QString ("http://www.avm.de"));
    g->addWidget (pUrlLabel, 6, 0);
    pTabDlg->addTab (pPage, i18n ("&Info"));

    // options page
    pPage = (QWidget*)new KIsdnPathPrefs (pTabDlg);
    pTabDlg->addTab (pPage, i18n ("&Behaviour"));

    pTabDlg->setOkButton ();
    pTabDlg->setCancelButton ();
    connect (pTabDlg, SIGNAL (applyButtonPressed ()), pPage, SLOT (saveValues ()));
    connect (pTabDlg, SIGNAL (applyButtonPressed ()), this, SLOT (apply ()));

    pTabDlg->show ();
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnMon::profiles ()
{
   if ( m_CapiInfo.IsInit () )
   {
				// reload controller info
				m_CapiInfo.ReadProfile ();

				if ( m_CapiInfo.ControllerCount () > 0 )
				{
    				KAvmmonTabDialog* pTab = new KAvmmonTabDialog;

						for (_cbyte byController = 1; byController <= m_CapiInfo.ControllerCount (); byController++)
						{
								CCapiController* pController = m_CapiInfo.GetController (byController);
								if ( pController != NULL )
								{
                		// don't show DSL controller
             				if ( ((pController->GetB1 () & CAPI_B1PROTOCOL_ATM) != 0) &&
             						 ((pController->GetB2 () & CAPI_B2PROTOCOL_PPPOE) != 0) &&
             						 ((pController->GetB3 () & CAPI_B3PROTOCOL_PPPOE) != 0) )
                    		continue;
    								
            				QWidget* pPage = (QWidget*)new KControllerProfile (pController, pTab);
    								pPage->setMinimumSize (350,300);

    								QString strController;
    								strController.sprintf ("Controller %u", (unsigned int)byController);
    								pTab->addTab (pPage, strController);
								} // if pController
						} // for byController

    				pTab->setOkButton ();
    				pTab->show ();
    				return;
    		}
    }

    // show error again
    QString strError;
    if ( m_CapiInfo.GetLastError (strError) != 0 ) QMessageBox::warning (this, i18n (ErrTitle), strError);
}



/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
int main ( int argc, char **argv )
{
    KAboutData aboutData (I18N_NOOP ("kisdnwatch"), AppTitle, VERSION,
                          I18N_NOOP ("AVM-CAPI-Monitor for the K Desktop Environment"),
			  KAboutData::License_GPL, I18N_NOOP ("(c) 2001-2005 AVM Berlin"));
    aboutData.addAuthor (I18N_NOOP ("AVM"),0, I18N_NOOP ("info@avm.de"));

    KCmdLineArgs::init ( argc, argv, &aboutData );

    // unique instance of this application!
    if ( !KUniqueApplication::start () )
    {
        // could not be localized before KApp contruction...
        //fprintf (stderr, (const char*)(i18n ("%1 is already running!\n").arg (i18n (AppTitle)).local8Bit ()));
        fprintf (stderr, "%s is already running!\n", AppTitle);
        exit (0);
    }

    KIsdnMonApplication app;
    pApp = &app;
    KIsdnMon IMon;
    
    IMon.SetToolTip ();
    IMon.SetSystemTrayWindow ();

    app.setTopWidget (&IMon);

    return app.exec ();
}
