/***************************************************************************
 *   Copyright (C) 2001-2003 AVM GmbH. All rights reserved.                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "ccapiinfo.h"
#include "msg.h"
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <qtextstream.h>


/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
CCapiController::CCapiController (_cbyte byController)
							 : m_wChanCount (0),
								 m_dwB1 (0),
								 m_dwB2 (0),
								 m_dwB3 (0),
								 m_iAvmCapi (-1),
                                 m_bValidResponse (false),
								 m_wDProtocol (0),
								 m_byLine (0),
								 m_byController (byController)
{
		for (int ii = 0; ii < CAPIINFO_MAX_BCHANNELS; ii++) m_bChannelInfo[ii] = false;
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void CCapiController::SetProfile (_cbyte* pProfile)
{
		if ( pProfile != NULL )
		{
				m_wChanCount = *((_cword*)(pProfile + 2));
				m_dwOptions = *((_cdword*)(pProfile + 4));
				m_dwB1 = *((_cdword*)(pProfile + 8));
				m_dwB2 = *((_cdword*)(pProfile + 12));
				m_dwB3 = *((_cdword*)(pProfile + 16));
								
  			// controller's manufacturer specific profile information
  			// subject to change!!
  			if ( *((_cbyte*)(pProfile + 44)) == 1 )
  			{
						m_wDProtocol = *((_cdword*)(pProfile + 44 + 3));
						m_byLine = *((_cbyte*)(pProfile + 44 + 5));
  			}
  			else
  			{
						m_wDProtocol = 0;
						m_byLine = 0;
				}
				
#ifdef CPROT				
				CAPI_PROTOCOL_TEXT ("\ncontroller=%hu, B channels=%hu, options=0x%08lX, b1=0x%08lX, b2=0x%08lX, b3=0x%08lX\n", (_cword)m_byController, m_wChanCount,
														m_dwOptions, m_dwB1, m_dwB2, m_dwB3);
#endif						
		}
		else
		{
				m_wChanCount = 0;
				for (int ii = 0; ii < CAPIINFO_MAX_BCHANNELS; ii++) m_bChannelInfo[ii] = false;
				m_dwOptions = 0;
				m_dwB1 = 0;
				m_dwB2 = 0;
				m_dwB3 = 0;
		}
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
_cword CCapiController::GetChannelCount (bool bChannelUp) const
{
		_cword wReturn = 0;
		for (int ii = 0; ii < CAPIINFO_MAX_BCHANNELS; ii++) if ( m_bChannelInfo[ii] == bChannelUp ) wReturn++;
		return wReturn;
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
QString CCapiController::GetManufacturer ()
{
		_cbyte* pBuffer = new _cbyte[64];
		pBuffer[0] = '\0';
		QString strManufacturer = (char*)capi20_get_manufacturer (m_byController, pBuffer);
#ifdef CPROT				
		CAPI_PROTOCOL_TEXT ("\ncapi20_get_manufacturer (Ctrl=%u) returned \"%s\"\n", m_byController, pBuffer);
#endif						
		delete pBuffer;

		// test if AVM's CAPI (needed in GetManufacturerVersion)
		m_iAvmCapi = ( strManufacturer.find ("avm", 0, false) > -1 ) ? 1 : 0;
		
		return strManufacturer;
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
QString CCapiController::GetVersion () const
{
		QString strVersion;

		unsigned long Buffer[4];
  	memset (Buffer, 0, sizeof (Buffer));
		if ( capi20_get_version (m_byController, (unsigned char*)Buffer) != NULL )
				strVersion.sprintf ("%lu.%lu", Buffer[0], Buffer[1]);
#ifdef CPROT				
		CAPI_PROTOCOL_TEXT ("\ncapi20_get_version (Ctrl=%u) returned version=%lu.%lu, manufacturer-spec.=%lu.%lu\n", m_byController,
												Buffer[0], Buffer[1], Buffer[2], Buffer[3]);
#endif						
		
		return strVersion;
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
QString CCapiController::GetManufacturerVersion ()
{
		QString strVersion;
		
		if ( m_iAvmCapi < 0 ) GetManufacturer (); // test if AVM's CAPI
		
		unsigned long Buffer[4];
  	memset (Buffer, 0, sizeof (Buffer));
		if ( capi20_get_version (m_byController, (unsigned char*)Buffer) != NULL )
		{
				if ( m_iAvmCapi > 0 )
						strVersion.sprintf ("%lu.%lu%lu.%02lu", (Buffer[2] & 0x000000F0) >> 4,
                                Buffer[2] & 0x0000000F, (Buffer[3] & 0x000000F0) >> 4,
                                Buffer[3] & 0x0000000F);
				else
						strVersion.sprintf ("%lu.%lu", Buffer[2], Buffer[3]);
		}
#ifdef CPROT				
		CAPI_PROTOCOL_TEXT ("\ncapi20_get_version (Ctrl=%u) returned version=%lu.%lu, manufacturer-spec.=%lu.%lu\n", m_byController,
												Buffer[0], Buffer[1], Buffer[2], Buffer[3]);
#endif						
		
		return strVersion;
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
QString CCapiController::GetSerialNumber () const
{
		_cbyte* pBuffer = new _cbyte[8];
		pBuffer[0] = '\0';
		QString strSerialNumber = (char*)capi20_get_serial_number (m_byController, pBuffer);
#ifdef CPROT				
		CAPI_PROTOCOL_TEXT ("\ncapi20_get_serial_number (Ctrl=%u) returned \"%s\"\n", m_byController, pBuffer);
#endif						
		delete pBuffer;
		
		return strSerialNumber;
}

#ifdef CPROT
QFile Logfile;
char pCapiProtocolBuffer[CAPI_PROTOCOL_INIT_BUF_SIZE];
bool bProtInit = false;

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void CAPI_PROTOCOL_CALLBACK (char* pOutput, CAPI_PROTOCOL_TYP /*type*/, CAPI_MESSAGE /*pMessage*/)
{
		if ( Logfile.handle () > -1 )
		{
				QTextStream stream (&Logfile);
				stream << pOutput;
		}
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
/*static*/ bool CCapiInfo::OpenProtocol (const char* szFileName)
{
		CloseProtocol (); // reopen?
		Logfile.setName (szFileName);
		bool bOk = Logfile.open (IO_Append);
		if ( !bOk ) bOk = Logfile.open (IO_WriteOnly);
		
		// write header in logfile
		if ( bOk ) CAPI_PROTOCOL_INIT (pCapiProtocolBuffer, CAPI_PROTOCOL_CALLBACK);
		
		return bOk;
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
/*static*/ void CCapiInfo::CloseProtocol ()
{
		if ( Logfile.handle () > -1 ) Logfile.close ();
}

#endif // CPROT

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
CCapiInfo::CCapiInfo()
				 : m_bInit (false),
				   m_wApplId (0),
				   m_wLastMsgNumber (0),
				   m_nLastError (CapiNoError),
           m_iLastErrno (0),
				   m_iLineEvent (line_nop),
				   m_wApplCount (0),
				   m_bAvmCapi (false)
{
		m_pManuData = new unsigned char[GetManuDataSize ()];
    m_Controller.setAutoDelete (true);
#ifdef CPROT
    if ( !bProtInit )
    {
    		CAPI_PROTOCOL_INIT (pCapiProtocolBuffer, CAPI_PROTOCOL_CALLBACK);
    		bProtInit = true;
    }
#endif
}


CCapiInfo::~CCapiInfo()
{
		Release ();
		if ( m_pManuData != NULL ) delete m_pManuData;
}

/*--------------------------------------------------------------------------*\
		Initialize before using it
		
		strMsg				to return "human readable" error
									(language depends on CAPI library)
		Return Value: true if succeded, otherwise false
\*--------------------------------------------------------------------------*/
bool CCapiInfo::Register (QString& strMsg)
{
		if ( !m_bInit )
		{
				
				// capi available?
				m_nLastError = capi20_isinstalled ();
#ifdef CPROT				
				CAPI_PROTOCOL_TEXT ("\ncapi20_isinstalled () returned 0x%hX\n%s\n", m_nLastError, capi_info2str (m_nLastError));
#endif						
				// access to /dev/capi20 failed?
        m_iLastErrno = ( (m_nLastError == CapiRegNotInstalled) || (m_nLastError == CapiMsgNotInstalled) ) ?
        							 errno : 0;

        switch ( m_nLastError )
        {
            case CapiRegNotInstalled:
            case CapiMsgNotInstalled:			// access to /dev/capi20 failed?
              	m_iLastErrno = errno;
            		break;

            case CapiNoError:							// register this app.
            		{
          					m_nLastError = capi20_register (1, 2, 128, (unsigned int*)&m_wApplId);
          					if ( m_nLastError == CapiNoError )
          					{
#ifdef CPROT				
          							CAPI_PROTOCOL_TEXT ("\ncapi20_register ()  ApplID=%u\n", m_wApplId);
#endif						
          							m_bInit = true;
          							if ( !ReadProfile () )
          							{
          									unsigned int nError = m_nLastError;
          									Release ();
          									m_nLastError = nError; // return error of ReadProfile () rather than Release ()
          							}
          					}
#ifdef CPROT				
          					else CAPI_PROTOCOL_TEXT ("\ncapi20_register () returned 0x%hX\n%s\n", m_nLastError, capi_info2str (m_nLastError));
#endif						
                } // case CapiNoError
            		break;
        }; // switch m_nLastError
				
				// error message
				if ( m_nLastError != CapiNoError ) GetLastError (strMsg);
		}
		
		return m_bInit;
}

/*--------------------------------------------------------------------------*\
		release when finished
\*--------------------------------------------------------------------------*/
bool CCapiInfo::Release ()
{
		if ( m_bInit )
		{
				// bye, bye
				if ( (m_nLastError = capi20_release (m_wApplId)) != CapiNoError )
				{
#ifdef CPROT				
						CAPI_PROTOCOL_TEXT ("\ncapi20_release () returned 0x%hX\n%s\n", m_nLastError, capi_info2str (m_nLastError));
#endif						
						return false;
				}
				
				// cleaning up
 			 	m_bInit = false;
 			  m_wLastMsgNumber = 0;
 			  m_nLastError = CapiNoError;
 			  m_wApplCount = 0;
 			  m_Controller.clear ();
		}
		
		return true;
}

/*--------------------------------------------------------------------------*\
		poll CAPI for current b channel info, updates members
\*--------------------------------------------------------------------------*/
bool CCapiInfo::Update()
{
		if ( m_bInit )
		{
				// querey proprietary data from AVM's CAPI implementation for all controllers
				_cword nError = PutRequestMessages ();
				switch ( nError )
				{
						// no problem
    				case CapiMsgBusy:
    				case CapiNoError:
    						break;
    				
						// fatal -> release appl.
						case CapiIllAppNr:
    				case CapiIllCmdOrSubcmdOrMsgToSmall:
    				case CapiReceiveOverflow:
    				case CapiSendQueueFull:
    				case CapiMsgOSResourceErr:
    				case CapiMsgNotInstalled:
    				case CapiRegNotInstalled:
    				case CapiMsgCtrlerNotSupportExtEquip:
    				case CapiMsgCtrlerOnlySupportExtEquip:
    						Release ();
    						m_nLastError = nError;
    						return false;
    				    				
    				// only report error
    				default:
    						m_nLastError = nError;
    						break;
				}; // switch nError
				
				// receive an process messages
				bool bDone = false;
				while ( !bDone )
				{
						nError = GetAndProcessMessage ();
						switch ( nError )
						{
								// no problem
    						case CapiReceiveQueueEmpty:
    								bDone = true;
    						case CapiNoError:
    								break;
    				
								// fatal -> release appl.
								case CapiIllAppNr:
    						case CapiIllCmdOrSubcmdOrMsgToSmall:
    						case CapiReceiveOverflow:
    						case CapiSendQueueFull:
    						case CapiMsgOSResourceErr:
    						case CapiMsgNotInstalled:
    						case CapiRegNotInstalled:
    						case CapiMsgCtrlerNotSupportExtEquip:
    						case CapiMsgCtrlerOnlySupportExtEquip:
    								Release ();
    								// fall throu!!
    				    				
    						// only report error
    						default:
    								m_nLastError = nError; // return error of CAPI_GET_CMSG rather than MANUFACTURER_REQ
    								bDone = true;
    								break;
						}; // switch nError
				} // while !bDone
		} // if m_bInit
		
		return (m_nLastError == CapiNoError);
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
_cword CCapiInfo::ManufacturerRequest (_cbyte byController, _cdword dwFunction)
{
#ifdef CPROT
		static unsigned char Message[2048];
 		capi_fill_MANUFACTURER_REQ (&m_cmsg, m_wApplId, GetNewMessageNumber (),
 																byController, CAPI_MANUID_AVM, CAPI_MANUCLASS,
 																dwFunction, m_pManuData);
 		capi_cmsg2message (&m_cmsg, Message);
 		CAPI_PROTOCOL_MESSAGE (Message);
 		_cword nError = capi20_put_message (m_wApplId, Message);
 		if ( nError != CapiNoError )
 				CAPI_PROTOCOL_TEXT ("\ncapi20_put_message () returned 0x%hX\n%s\n", nError, capi_info2str (nError));
 		return nError;
#else
 		return MANUFACTURER_REQ (&m_cmsg, m_wApplId, GetNewMessageNumber (),
 														 byController, CAPI_MANUID_AVM, CAPI_MANUCLASS,
 														 dwFunction, m_pManuData);
#endif // CPROT																			
}

/*--------------------------------------------------------------------------*\
		querey proprietary data from AVM's CAPI implementation for all controllers
\*--------------------------------------------------------------------------*/
_cword CCapiInfo::PutRequestMessages ()
{
		_cword wReturn = CapiNoError;
				
		// querey proprietary data from AVM's CAPI implementation for all controllers
		_cword nError = CapiNoError;
		_cbyte nControllerOk = 0;
		CCapiController* pController = m_Controller.first ();
		while ( pController != NULL )
		{
				m_pManuData[0] = 63;
				nError = ManufacturerRequest (pController->GetController (), CAPI_MANUFUNCTION_GETBCHANNELINFO);
				if ( nError == CapiNoError )
						nControllerOk++;
				else
						wReturn = nError;	// report recent error
				pController = m_Controller.next ();
		} // while
				
		if ( nControllerOk > 0 ) wReturn = CapiNoError; // don't report error if (at least) one ok

		return wReturn;
}

/*--------------------------------------------------------------------------*\
		receive an process messages
\*--------------------------------------------------------------------------*/
_cword CCapiInfo::GetAndProcessMessage ()
{
		// get messages from capi
#ifdef CPROT
		CAPI_MESSAGE pMessage;
		unsigned int nError = capi20_get_message (m_wApplId, &pMessage);
		if ( nError != CapiNoError )
		{
				if ( nError != CapiReceiveQueueEmpty )
						CAPI_PROTOCOL_TEXT ("\ncapi20_get_message () returned 0x%hX\n%s\n", nError, capi_info2str (nError));
				return nError;
		}
		CAPI_PROTOCOL_MESSAGE (pMessage);
		capi_message2cmsg (&m_cmsg, pMessage);
#else
		unsigned int nError = CAPI_GET_CMSG (&m_cmsg, m_wApplId);
		if ( nError != CapiNoError ) return nError;
#endif // CPROT
    		
    switch ( m_cmsg.Subcommand )
    {
				case CAPI_CONF:	/*----- Confirmation -----*/
						{
    						// the answer we are waiting for, ignore others
								if ( (m_cmsg.Command == CAPI_MANUFACTURER) &&
										 (MANUFACTURER_CONF_MANUID (&m_cmsg) == CAPI_MANUID_AVM) &&
										 (MANUFACTURER_CONF_CLASS (&m_cmsg) == 0) &&
										 (MANUFACTURER_CONF_FUNCTION (&m_cmsg) == CAPI_MANUFUNCTION_GETBCHANNELINFO) )
 								{
 										_cstruct pManuData = MANUFACTURER_CONF_MANUDATA (&m_cmsg);
 										_cword wManuDataLen = 0;
 										if ( *((_cbyte*)m_cmsg.ManuData) == 0xFF )
 										{
 												// data size >= 255
     										wManuDataLen = *((_cword*)(pManuData + 1));
     										pManuData += 3;
 										}
 										else
 										{
 												// data size < 255
     										wManuDataLen = *((_cbyte*)pManuData);
     										pManuData += 1;
 										}
 														
 										// how many appl. are registered (including me!)
 										if ( wManuDataLen > 1 )
 										{
 												m_wApplCount = *((_cword*)pManuData);
#ifdef CPROT
												CAPI_PROTOCOL_TEXT ("%hu ApplIDs registered (including me!)\n", m_wApplCount);
#endif												
										}
 										else m_wApplCount = 0;
 			
 										// select controller
 										CCapiController* pController = GetController ((_cbyte)MANUFACTURER_CONF_CONTROLLER (&m_cmsg));
 										if ( pController != NULL )
 										{
#ifdef CPROT
												CAPI_PROTOCOL_TEXT ("controller %hu B channel usage: ", (_cword)pController->GetController ());
#endif												
 												// state of channels
 												for (unsigned int nn = 0; nn < pController->GetChannelCount (); nn++)
 												{
 														bool bOld = pController->m_bChannelInfo[nn];
 														if ( wManuDataLen > (nn + 5) )
 														{
 																pController->m_bChannelInfo[nn] = ( *((_cbyte*)(pManuData + nn + 5)) != 0 );
#ifdef CPROT
																CAPI_PROTOCOL_TEXT ("%s", (pController->m_bChannelInfo[nn]) ? "1" : "0");
#endif												
 														}
 														else
 														{
 																pController->m_bChannelInfo[nn] = false;
#ifdef CPROT
																CAPI_PROTOCOL_TEXT ("-");
#endif												
 														}
 														if ( bOld != pController->m_bChannelInfo[nn] )
 																m_iLineEvent |= (bOld) ? line_down : line_up;
 												} // for nn
                                                pController->m_bValidResponse = true;
#ifdef CPROT
												CAPI_PROTOCOL_TEXT ("\n");
#endif												
 										} // if pController
 								} // if m_cmsg.Command
						}
						break; // case CAPI_CONF
										
				case CAPI_IND: /*----- Indication -----*/
						{
								// should send response
								capi_cmsg_answer (&m_cmsg);
								m_cmsg.Messagenumber = GetNewMessageNumber ();
								capi_put_cmsg (&m_cmsg);
						}
						break;
		} // switch m_cmsg.Subcommand
		
		return CapiNoError;
}

/*--------------------------------------------------------------------------*\
		is b channel connected? true if so
\*--------------------------------------------------------------------------*/
bool CCapiInfo::IsBChannelUp (_cbyte byController, _cbyte byBChannel)
{
		CCapiController* pController = NULL;
		if ( m_bInit && ((pController = GetController (byController)) != NULL) )
				return pController->GetChannelInfo (byBChannel);
		return false;
}

/*--------------------------------------------------------------------------*\
		get info about CAPI and installed controllers
\*--------------------------------------------------------------------------*/
bool CCapiInfo::ReadProfile ()
{
		if ( m_bInit )
		{
				_cbyte* pBuffer = new _cbyte[64];
				pBuffer[0] = '\0';
				
				// manufacturer of CAPI device
				QString strManufacturer = (char*)capi20_get_manufacturer (0, pBuffer);
#ifdef CPROT				
				CAPI_PROTOCOL_TEXT ("\ncapi20_get_manufacturer (Ctrl=0) returned \"%s\"\n", pBuffer);
#endif						
				m_bAvmCapi = (strManufacturer.find ("avm", 0, false) > -1);
				    				
 				// how many controllers are installed (we need one at least)
 				_cword wConCount = 0;
 				if ( (m_nLastError = capi20_get_profile (0, pBuffer)) == CapiNoError )
 				{
 						wConCount = *((_cword*)pBuffer);
 						if ( wConCount == 0 ) m_nLastError = CapiMsgNotInstalled;
#ifdef CPROT				
						CAPI_PROTOCOL_TEXT ("\ncapi20_get_profile (Ctrl=0) returned %hu controllers\n", wConCount);
#endif						
 						
 						for (_cword ww = 0; (ww < wConCount) && (m_nLastError == CapiNoError); ww++)
 						{
								CCapiController* pController = GetController (ww + 1);
    						
    						// how many B-Channels are available on this controller
         				m_nLastError = capi20_get_profile (ww + 1, pBuffer);
#ifdef CPROT				
								CAPI_PROTOCOL_TEXT ("\ncapi20_get_profile (Ctrl=%hu) returned 0x%hX\n%s\n", ww + 1, m_nLastError, capi_info2str (m_nLastError));
#endif						
         				switch ( m_nLastError )
         				{
         						case CapiNoError:
         						{
         								if ( pController == NULL )
         								{
         										pController = new CCapiController (ww + 1);
         										m_Controller.append (pController);
         								}
         								pController->SetProfile (pBuffer);
         								break;
         						}
         						
         						case CapiMsgNotInstalled:
         						{
         								if ( pController == NULL )
         								{
         										pController = new CCapiController (ww + 1);
         										m_Controller.append (pController);
         								}
         								pController->SetProfile (NULL);
         								m_nLastError = CapiNoError;	// controller not present
         								break;
         						}
         				}; // switch m_nLastError
 						} // for ww
    		} // if capi20_get_profile
#ifdef CPROT				
    		else CAPI_PROTOCOL_TEXT ("\ncapi20_get_profile (Ctrl=0) returned 0x%hX\n%s\n", m_nLastError, capi_info2str (m_nLastError));
#endif						
    		
    		delete pBuffer;
    		return (m_nLastError == CapiNoError);
    }

    return false;
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
CCapiController* CCapiInfo::GetController (_cbyte byController)
{
		CCapiController findController (byController);
		m_Controller.find (&findController);
		return m_Controller.current ();	// NULL if not found
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
unsigned int CCapiInfo::GetChannelCount (_cbyte byController)
{
		unsigned int nReturn = 0;
		
		if ( byController == 0 )
		{
				// all controller
				CCapiController* pController = m_Controller.first ();
				while ( pController != NULL )
				{
						nReturn += pController->GetChannelCount ();
						pController = m_Controller.next ();
				} // while
		}
		else
		{
				// specified controller
				CCapiController* pController = GetController (byController);
				if ( pController != NULL ) nReturn = pController->GetChannelCount ();
		}
		
		return nReturn;
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
bool CCapiInfo::IsDslOnly ()
{
    bool bDslFound = false;
    bool bNonDslFound = false;
				
    CCapiController* pController = m_Controller.first ();
	while ( (pController != NULL) && !bNonDslFound )
	{
        if ( pController->IsDsl () )
            bDslFound = true;
        else
            bNonDslFound = true;
		pController = m_Controller.next ();
	} // while

    return (bDslFound && !bNonDslFound);
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
unsigned int CCapiInfo::GetChannelCount (bool bChannelUp, _cbyte byController)
{
		unsigned int nReturn = 0;
		
		if ( byController == 0 )
		{
				// all controller
				CCapiController* pController = m_Controller.first ();
				while ( pController != NULL )
				{
						nReturn += pController->GetChannelCount (bChannelUp);
						pController = m_Controller.next ();
				} // while
		}
		else
		{
				// specified controller
				CCapiController* pController = GetController (byController);
				if ( pController != NULL ) nReturn = pController->GetChannelCount (bChannelUp);
		}
		
		return nReturn;
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
int CCapiInfo::TestLineEvent ()
{
		int iLineEvent = m_iLineEvent;
		m_iLineEvent = line_nop;
		return iLineEvent;
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
unsigned int CCapiInfo::GetLastError (QString& strMsg) const
{
		switch ( m_nLastError )
    {
        case CapiRegNotInstalled:
        case CapiMsgNotInstalled:			// access to /dev/capi20 failed?
          	if ( m_iLastErrno != 0 )
            {
                strMsg.sprintf ("/dev/capi20: %s", strerror (m_iLastErrno));
        				break;
            }

        default:
   					strMsg = capi_info2str (m_nLastError);
        		break;
    } // switch m_nLastError
  	
    return m_nLastError;
}
