/***************************************************************************
 *   Copyright (C) 2004 AVM GmbH. All rights reserved.                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "netdevstatistics.h"

#include <qfileinfo.h>
#include <qfile.h>


const QString DEVNAME_PROC = "ppp0:";
const char* DEVNAME_SYS[] = { "dsl0", "ppp0", NULL };


/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
CNetdevStatistics::CNetdevStatistics ()
                 : m_iDevicenameSys (-1),
                   m_bUseProc (false),
                   m_nRxBytes (0),
                   m_nTxBytes (0)
{
    QFileInfo FileInfo ("/sys/class/net");
    m_bUseProc = !FileInfo.exists ();
}

CNetdevStatistics::~CNetdevStatistics ()
{
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
bool CNetdevStatistics::Update ()
{
    if ( m_bUseProc ) return UpdateFromProc ();
    return UpdateFromSys ();
}

/*--------------------------------------------------------------------------*\
    pre-2.6 kernel
\*--------------------------------------------------------------------------*/
bool CNetdevStatistics::UpdateFromProc ()
{
    QFile File (QString ("/proc/net/dev"));
    if ( File.open (IO_ReadOnly) )
    {
        // skip header
        QString strLine;
        if ( File.readLine (strLine, 1000) == -1 ) return false;
        if ( File.readLine (strLine, 1000) == -1 ) return false;
        strLine.truncate (0);
        
        // search for interface line
        while ( File.readLine (strLine, 1000) > 0 )
        {
            strLine = strLine.stripWhiteSpace ();
            if ( strLine.left (DEVNAME_PROC.length ()) == DEVNAME_PROC )
            {
                strLine = strLine.mid (DEVNAME_PROC.length ())
                                 .stripWhiteSpace ().simplifyWhiteSpace ();
                break;
            }
            strLine.truncate (0);
        }
        
        if ( !strLine.isEmpty () )
        {
            // grab info
            bool bOk = false;
            unsigned int nTx, nRx = strLine.section (' ', 0, 0).toUInt (&bOk);
            if ( bOk ) nTx = strLine.section (' ', 8, 8).toUInt (&bOk);
            if ( bOk )
            {
                m_nRxBytes = nRx;
                m_nTxBytes = nTx;
                return true;
            }
        }
    }
    
    return false;
}

/*--------------------------------------------------------------------------*\
    2.6 kernel or later
\*--------------------------------------------------------------------------*/
bool CNetdevStatistics::UpdateFromSys ()
{
    unsigned int nRx = 0, nTx = 0;
    
    if ( GetNumFromSys ("rx_bytes", nRx) && GetNumFromSys ("tx_bytes", nTx) )
    {
        m_nRxBytes = nRx;
        m_nTxBytes = nTx;
        return true;
    }
    
    return false;
}

/*--------------------------------------------------------------------------*\
    2.6 kernel or later
\*--------------------------------------------------------------------------*/
bool CNetdevStatistics::GetNumFromSys (const char* pszName, unsigned int& nNum)
{
    bool bOk = false;
    QString str;
    
    for (int ii = 0; (m_iDevicenameSys == -1) && (DEVNAME_SYS[ii] != NULL); ii++)
    {
        str.sprintf ("/sys/class/net/%s", DEVNAME_SYS[ii]);
        QFileInfo FileInfo (str);
        if ( FileInfo.exists () ) m_iDevicenameSys = ii;
    }
    
    if ( m_iDevicenameSys > -1 )
    {
        str.sprintf ("/sys/class/net/%s/statistics/%s", DEVNAME_SYS[m_iDevicenameSys], pszName);
        
        QFile File (str);
        if ( File.open (IO_ReadOnly) )
        {
            str.truncate (0);
            File.readLine (str, 20);
            nNum = str.toUInt (&bOk);
        }
    }
    
    return bOk;
}
