/***************************************************************************
 *   Copyright (C) 2001-2005 AVM GmbH. All rights reserved.                *
 *   Copyright (C) 1998-2000 Markus Dahlweid                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kimon.h"
#include "kimoninfo.h"
#include "kimoninfo.moc"

#include <qwidget.h>
#include <qdatetm.h>
#include <qfile.h>
#undef Unsorted
#include <qdir.h>
#include <qtextstream.h>
#include <qmessagebox.h>
#include <qdatetime.h>
#include <kglobal.h>
#include <klocale.h>

extern QObject *IMon;

// layout
int KIsdnInfo::m_iX                = 10;
int KIsdnInfo::m_iStatusCX         = 80;
int KIsdnInfo::m_iTimeCX           = 80;
int KIsdnInfo::m_iRateCX           = 120;
int KIsdnInfo::m_iAdslAtmPPPCX     = 30;
const int KIsdnInfo::m_iLedCX      = 14;
const int KIsdnInfo::m_iSpaceCX    = 6;
const int KIsdnInfo::m_iSpaceCY    = 3;
const int KIsdnInfo::m_iY          = 3;
int KIsdnInfo::m_iCY               = 20;
const int KIsdnInfo::m_iLedCY      = 14;
const int KIsdnInfo::m_iCompactCX  = 14;
const int KIsdnInfo::m_iCompactCY  = 15;
const int KIsdnInfo::m_iCompactSeparatorCY    = 1;

const unsigned long MAX_LOGFILESIZE = 1024 * 1024;

#ifndef max
#define max(a,b) (((a) > (b)) ? (a) : (b))
#endif

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
KIsdnInfo::KIsdnInfo (QWidget* parent, const char* name)
		 : QWidget (parent, name),
           m_OnlineResetSecs (0),
    	   m_bOnline (false),
           m_bShowLogfileError (true),
           m_bFirstUpdate (true),
    	   m_nRxBytes (0),
    	   m_nTxBytes (0)
{
    setIcon (pApp->icon ());
		
    m_pLabelTime = new QLabel (this);
    m_pLabelTime->setAlignment (AlignHCenter | AlignVCenter);
    m_pLabelTime->setText (i18n ("Time"));
    m_pLabelTime->show ();
    m_pTime = new QLabel (this);
    m_pTime->setFrameStyle (QFrame::Panel | QFrame::Sunken);
    m_pTime->setAlignment (AlignHCenter | AlignVCenter);
    m_pTime->setText ("00:00:00");
    m_pTime->show ();

    m_pLabelRate = new QLabel (this);
    m_pLabelRate->setAlignment (AlignHCenter | AlignVCenter);
    m_pLabelRate->setText (i18n ("Data rx/tx (kBit/s)"));
    m_pLabelRate->show ();
    m_pRate = new QLabel (this);
    m_pRate->setFrameStyle (QFrame::Panel | QFrame::Sunken);
    m_pRate->setAlignment (AlignHCenter | AlignVCenter);
    m_pRate->setText ("");
    m_pRate->show ();
		
    m_pLabelStatus = new QLabel (this);
    m_pLabelStatus->setAlignment (AlignHCenter | AlignVCenter);
    m_pLabelStatus->setText (i18n ("Status"));
    m_pLabelStatus->show ();
    m_pStatus = new QLabel (this);
    m_pStatus->setFrameStyle (QFrame::Panel | QFrame::Sunken);
    m_pStatus->setAlignment (AlignHCenter | AlignVCenter);
    m_pStatus->setText (CDslInfo::GetStatusString (CDslInfo::offline));
    m_pStatus->show ();
		
    int iLedsX = m_iX + m_iStatusCX + m_iTimeCX + m_iRateCX + 3 * m_iSpaceCX;
    int iLedsY = m_iY + m_iSpaceCY + max (3, (m_iCY - m_iLedCY) / 2);
    m_pLabelAdsl = new QLabel (this);
    m_pLabelAdsl->setAlignment (AlignHCenter | AlignVCenter);
    m_pLabelAdsl->setText (i18n ("ADSL"));
    m_pWidgetAdsl = new QWidget (this);
    m_pWidgetAdsl->setGeometry (iLedsX + (m_iAdslAtmPPPCX - m_iLedCX) / 2, iLedsY, m_iLedCX, m_iLedCY);
    m_pLedAdsl = new KIsdnInfoLineStatus (m_pWidgetAdsl);
    m_pLedAdsl->setFixedSize (m_iLedCX, m_iLedCY);
    m_pLedAdsl->show ();
    m_pLedAdsl->on ();
		
    m_pLabelAtm = new QLabel (this);
    m_pLabelAtm->setAlignment (AlignHCenter | AlignVCenter);
    m_pLabelAtm->setText (i18n ("ATM"));
    m_pWidgetAtm = new QWidget (this);
    m_pWidgetAtm->setGeometry (iLedsX + m_iAdslAtmPPPCX + m_iSpaceCX + (m_iAdslAtmPPPCX - m_iLedCX) / 2, iLedsY, m_iLedCX, m_iLedCY);
    m_pLedAtm = new KIsdnInfoLineStatus (m_pWidgetAtm);
    m_pLedAtm->setFixedSize (m_iLedCX, m_iLedCY);
    m_pLedAtm->show ();
    m_pLedAtm->on ();
		
    m_pLabelPPP = new QLabel (this);
    m_pLabelPPP->setAlignment (AlignHCenter | AlignVCenter);
    m_pLabelPPP->setText (i18n ("PPP"));
    m_pWidgetPPP = new QWidget (this);
    m_pWidgetPPP->setGeometry (iLedsX + 2 * m_iAdslAtmPPPCX + 2 * m_iSpaceCX + (m_iAdslAtmPPPCX - m_iLedCX) / 2, iLedsY, m_iLedCX, m_iLedCY);
    m_pLedPPP = new KIsdnInfoLineStatus (m_pWidgetPPP);
    m_pLedPPP->setFixedSize (m_iLedCX, m_iLedCY);
    m_pLedPPP->show ();
    m_pLedPPP->on ();
		
    m_pLabelFP = new QLabel (this);
    m_pLabelFP->setAlignment (AlignHCenter | AlignVCenter);
    m_pLabelFP->setText (i18n ("Fast"));
    m_pWidgetFP = new QWidget (this);
    m_pWidgetFP->setGeometry (iLedsX + 2 * m_iAdslAtmPPPCX + 2 * m_iSpaceCX + (m_iAdslAtmPPPCX - m_iLedCX) / 2, iLedsY, m_iLedCX, m_iLedCY);
    m_pLedFP = new KIsdnInfoLineStatus (m_pWidgetFP);
    m_pLedFP->setFixedSize (m_iLedCX, m_iLedCY);
    m_pLedFP->show ();
    m_pLedFP->on ();

    m_pCompactButton = new KArrowButton (this);
    connect (m_pCompactButton, SIGNAL (clicked ()), this, SLOT (ToggleCompact ()));
    m_pCompactButton->show ();
    m_pSeparator = new QLabel (this);
    m_pSeparator->setFrameStyle (QFrame::Panel | QFrame::Raised);
    m_pSeparator->show ();
        
    m_pLabelTotalTime = new QLabel (this);
    m_pLabelTotalTime->setAlignment (AlignHCenter | AlignVCenter);
    m_pLabelTotalTime->setText (i18n ("Time (total)"));
    m_pTotalTime = new QLabel (this);
    m_pTotalTime->setFrameStyle (QFrame::Panel | QFrame::Sunken);
    m_pTotalTime->setAlignment (AlignHCenter | AlignVCenter);
    int iSecsTo = pIMon->GetOnlineSecs ();
    int iSec = iSecsTo % 60;
    int iMin = iSecsTo / 60;
    int iHou = iMin / 60;
    iMin = iMin % 60;
    QString strTime;
    strTime.sprintf ("%02d:%02d:%02d", iHou, iMin, iSec);
    m_pTotalTime->setText (strTime);
		
    m_pLabelTransfer = new QLabel (this);
    m_pLabelTransfer->setAlignment (AlignHCenter | AlignVCenter);
    m_pLabelTransfer->setText (i18n ("Data rx/tx (total)"));
    m_pTransfer = new QLabel (this);
    m_pTransfer->setFrameStyle (QFrame::Panel | QFrame::Sunken);
    m_pTransfer->setAlignment (AlignHCenter | AlignVCenter);
    m_pTransfer->setText ("");

    m_pButtonReset = new QPushButton (this);
    m_pButtonReset->setText (i18n ("&Reset"));
    connect (m_pButtonReset, SIGNAL (clicked ()), this, SLOT (ResetTransfer ()));

    UpdateLayout ();
}

KIsdnInfo::~KIsdnInfo ()
{
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfo::Log (ELogEvent LogEvent)
{
    QDateTime timeUpto = QDateTime::currentDateTime ();

    QString strLogfile = QDir::homeDirPath ();
    if ( strLogfile.right (1)  == "/" )
        strLogfile += "kadslwatch.log";
    else
        strLogfile += "/kadslwatch.log";

    // open/create logfile
    QFile Logfile (strLogfile);
    if ( Logfile.size () >= MAX_LOGFILESIZE )
    {
        QString strBakfile = strLogfile + ".bak";
        remove (strBakfile.latin1 ());
        rename (strLogfile.latin1 (), strBakfile.latin1 ());
    }
    if ( !Logfile.open (IO_WriteOnly | IO_Append) )
    {
        if ( m_bShowLogfileError )
        {
            m_bShowLogfileError = false;
            QMessageBox::warning (this, i18n (ErrTitle), i18n ("Cannot open/create %1").arg (strLogfile));
        }
        return;
    }
    m_bShowLogfileError = true;
    QTextStream LogStream (&Logfile);
    
    QString strUpto = KGlobal::locale ()->formatDateTime (timeUpto, true, true);

    // append log entry
    switch ( LogEvent )
    {
        case log_onstart:
            LogStream << strUpto << i18n (" Starting %1 with PPP online!").arg (i18n (AppTitle)) << endl;
            break;

        case log_onexit:
            LogStream << strUpto << i18n (" Closing %1 with PPP online!").arg (i18n (AppTitle)) << endl;

        case log_ondisc:
            {
                QString strUpsince = KGlobal::locale ()->formatDateTime (m_tmStartOnline, true, true);
                int Duration = m_tmStartOnline.secsTo (timeUpto);
	        int Sec = Duration % 60;
	        int Min = Duration / 60;
	        int Hour = Min / 60;
	        Min = Min % 60;
                QString strDuration;
                strDuration.sprintf ("%02d:%02d:%02d", Hour, Min, Sec);
                Duration = pIMon->GetOnlineSecs ();
	        Sec = Duration % 60;
	        Min = Duration / 60;
	        Hour = Min / 60;
	        Min = Min % 60;
                QString strTotalOnline;
                strTotalOnline.sprintf ("%02d:%02d:%02d", Hour, Min, Sec);

                LogStream << i18n ("%1 to %2 PPP online %3 download %4 upload %5 / totals: time %6 download %7 upload %8")
                                .arg (strUpsince)
                                .arg (strUpto)
                                .arg (strDuration)
                                .arg (FormatBytes ((double)m_nRxBytes))
                                .arg (FormatBytes ((double)m_nTxBytes))
                                .arg (strTotalOnline)
                                .arg (FormatBytes (pIMon->GetRxBytes ()))
                                .arg (FormatBytes (pIMon->GetTxBytes ())) << endl;
            }
            break;

        case log_reset:
            {
                int Duration = pIMon->GetOnlineSecs () + m_OnlineResetSecs;
	        int Sec = Duration % 60;
	        int Min = Duration / 60;
	        int Hour = Min / 60;
	        Min = Min % 60;
                QString strTotalOnline;       	
                strTotalOnline.sprintf ("%02d:%02d:%02d", Hour, Min, Sec);
                LogStream << i18n ("%1 Reset totals: time %2 download %3 upload %4")
                                .arg (strUpto)
                                .arg (strTotalOnline)
                                .arg (FormatBytes (pIMon->GetRxBytes ()))
                                .arg (FormatBytes (pIMon->GetTxBytes ())) << endl;
            }
            break;
    }; // switch LogEvent

    Logfile.close ();
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfo::Update (CDslInfo& DslInfo)
{
    char szRate[32] = {"0.0 / 0.0"};
    
    // Status
    m_pStatus->setText (CDslInfo::GetStatusString (DslInfo.GetStatus ()));
                            
    switch ( DslInfo.GetStatus () )
    {
        case CDslInfo::adsl_start:
        case CDslInfo::adsl_unplugged:
            SetOnline (false);
            m_pLedAdsl->setColor (KLed::red);
            m_pLedAtm->setColor (KLed::darkGray);
            m_pLedPPP->setColor (KLed::darkGray);
            m_pLedFP->setColor (KLed::darkGray);
            DslInfo.SetInfoLED (false);
            break;
        case CDslInfo::adsl_train:
        case CDslInfo::adsl_retrain:
            SetOnline (false);
            m_pLedAdsl->setColor (KLed::yellow);
            m_pLedAtm->setColor (KLed::darkGray);
            m_pLedPPP->setColor (KLed::darkGray);
            m_pLedFP->setColor (KLed::darkGray);
            DslInfo.SetInfoLED (false);
            break;
        case CDslInfo::adsl_online:
            SetOnline (false);
            m_pLedAdsl->setColor (KLed::green);
            m_pLedAtm->setColor (KLed::darkGray);
            m_pLedPPP->setColor (KLed::darkGray);
            m_pLedFP->setColor (KLed::darkGray);
            m_pLedFP->setColor ((DslInfo.IsFastPath ()) ? KLed::green : KLed::darkGray);
            DslInfo.SetInfoLED (DslInfo.IsFastPath ());
            break;
        case CDslInfo::atm_online:
            {
                SetOnline (false);
                m_pLedAdsl->setColor (KLed::green);
                m_pLedAtm->setColor (KLed::green);
                m_pLedPPP->setColor (KLed::darkGray);
                m_pLedFP->setColor ((DslInfo.IsFastPath ()) ? KLed::green : KLed::darkGray);
                DslInfo.SetInfoLED (DslInfo.IsFastPath ());
            }
            break;
        case CDslInfo::ppp_online:
            {
                SetOnline (true);
                m_pLedAdsl->setColor (KLed::green);
                m_pLedAtm->setColor (KLed::green);
                m_pLedPPP->setColor (KLed::green);
                m_pLedFP->setColor ((DslInfo.IsFastPath ()) ? KLed::green : KLed::darkGray);
                DslInfo.SetInfoLED (DslInfo.IsFastPath ());
                    
                // rate
                sprintf (szRate, "%.1f / %.1f", DslInfo.GetRxRate (), DslInfo.GetTxRate ());
                            
                // add transfer
                unsigned int nBytes = DslInfo.GetRxBytes ();
                if ( nBytes > m_nRxBytes )
                {
                    pIMon->SetRxBytes (pIMon->GetRxBytes () + (double)(nBytes - m_nRxBytes));
                    m_nRxBytes = nBytes;
                }
                nBytes = DslInfo.GetTxBytes ();
                if ( nBytes > m_nTxBytes )
                {
                    pIMon->SetTxBytes (pIMon->GetTxBytes () + (double)(nBytes - m_nTxBytes));
                    m_nTxBytes = nBytes;
                }
            }
            break;
        default:
            SetOnline (false);
            m_pLedAdsl->setColor (KLed::darkGray);
            m_pLedAtm->setColor (KLed::darkGray);
            m_pLedPPP->setColor (KLed::darkGray);
            m_pLedFP->setColor (KLed::darkGray);
            DslInfo.SetInfoLED (DslInfo.IsFastPath ());
    };
    m_pRate->setText (szRate);
				
    // transfer
    m_pTransfer->setText (FormatBytes (pIMon->GetRxBytes ()) + " / " + FormatBytes (pIMon->GetTxBytes ()));

    if ( DslInfo.GotValidResponse () ) m_bFirstUpdate = false;
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfo::SetOnline (bool bOnline)
{
    if ( bOnline )
    {
        if ( !m_bOnline )
        {
            m_tmStartOnline = QDateTime::currentDateTime ();
            m_OnlineResetSecs = 0;
            m_bOnline = true;
            m_nRxBytes = 0;
            m_nTxBytes = 0;
        }
        int iSecsTo = m_tmStartOnline.secsTo (QDateTime::currentDateTime ());
        int iSec = iSecsTo % 60;
        int iMin = iSecsTo / 60;
        int iHou = iMin / 60;
        iMin = iMin % 60;
        QString strTime;
       	strTime.sprintf ("%02d:%02d:%02d", iHou, iMin, iSec);
       	m_pTime->setText (strTime);
        if ( m_bFirstUpdate ) Log (log_onstart);

        // total time
        iSecsTo += pIMon->GetOnlineSecs () - m_OnlineResetSecs;
        iSec = iSecsTo % 60;
        iMin = iSecsTo / 60;
        iHou = iMin / 60;
        iMin = iMin % 60;
       	strTime.sprintf ("%02d:%02d:%02d", iHou, iMin, iSec);
       	m_pTotalTime->setText (strTime);
    }
    else
    {
        if ( m_bOnline )
        {
            m_bOnline = false;
            m_pTime->setText ("00:00:00");
            QDateTime timeUpto = QDateTime::currentDateTime ();
            pIMon->SetOnlineSecs (pIMon->GetOnlineSecs () + m_tmStartOnline.secsTo (timeUpto) - m_OnlineResetSecs);
            m_OnlineResetSecs = 0;
            Log (log_ondisc);
        }
  		
        int iSecsTo = pIMon->GetOnlineSecs ();
        int iSec = iSecsTo % 60;
        int iMin = iSecsTo / 60;
        int iHou = iMin / 60;
        iMin = iMin % 60;
        QString strTime;
       	strTime.sprintf ("%02d:%02d:%02d", iHou, iMin, iSec);
       	m_pTotalTime->setText (strTime);
    }
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfo::ResetTransfer ()
{
    if ( m_bOnline ||
         (pIMon->GetOnlineSecs () > 0) ||
         (pIMon->GetRxBytes () != 0.0) || (pIMon->GetTxBytes () != 0.0) )
    {
        if ( m_bOnline )
        {
            QDateTime timeUpto = QDateTime::currentDateTime ();
            m_OnlineResetSecs = m_tmStartOnline.secsTo (timeUpto);
        }
        Log (log_reset);
        pIMon->SetRxBytes (0.0);
        pIMon->SetTxBytes (0.0);
        pIMon->SetOnlineSecs (0);
        SetOnline (m_bOnline); // refresh total time display
    }
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
QString KIsdnInfo::FormatBytes (double f)
{
    bool bNeg = false;
    if ( f < 0.0 )
    {
            f *= -1.0;
            bNeg = true;
    }

    const char* pPostfix[] = {"kB", "MB", "GB", "TB"};
    int iPos;
    for (iPos = -1; (iPos < 3) && (f >= 10240.0); iPos++ ) f /= 1024.0;

    if ( bNeg ) f *= -1.0;
    QString str;
    if ( iPos > -1 )
        str.sprintf ("%.0f%s", f, pPostfix[iPos]);
    else
        str.sprintf ("%.0fB", f);
    return str;
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfo::UpdateLayout ()
{
    int iX = m_iX;
    int iY = m_iY;

    QSize size = m_pLabelTime->sizeHint ();
    m_iTimeCX = max (m_iTimeCX, size.width ());
    m_iCY = max (m_iCY, size.height ());
    size = m_pTime->sizeHint ();
    m_iTimeCX = max (m_iTimeCX, size.width ());
    m_iCY = max (m_iCY, size.height ());
    
    size = m_pLabelRate->sizeHint ();
    m_iRateCX = max (m_iRateCX, size.width ());
    m_iCY = max (m_iCY, size.height ());
    size = m_pRate->sizeHint ();
    m_iRateCX = max (m_iRateCX, size.width ());
    m_iCY = max (m_iCY, size.height ());
    
    size = m_pLabelStatus->sizeHint ();
    m_iStatusCX = max (m_iStatusCX, size.width ());
    m_iCY = max (m_iCY, size.height ());
    size = m_pStatus->sizeHint ();
    m_iStatusCX = max (m_iStatusCX, size.width ());
    m_iCY = max (m_iCY, size.height ());
    
    if ( !pIMon->isCompactView () )
    {
        size = m_pLabelAdsl->sizeHint ();
        m_iAdslAtmPPPCX = max (m_iAdslAtmPPPCX, size.width ());
        m_iCY = max (m_iCY, size.height ());
        size = m_pLabelAtm->sizeHint ();
        m_iAdslAtmPPPCX = max (m_iAdslAtmPPPCX, size.width ());
        m_iCY = max (m_iCY, size.height ());
        size = m_pLabelPPP->sizeHint ();
        m_iAdslAtmPPPCX = max (m_iAdslAtmPPPCX, size.width ());
        m_iCY = max (m_iCY, size.height ());
        size = m_pLabelFP->sizeHint ();
        m_iAdslAtmPPPCX = max (m_iAdslAtmPPPCX, size.width ());
        m_iCY = max (m_iCY, size.height ());
        
        size = m_pLabelTotalTime->sizeHint ();
        m_iTimeCX = max (m_iTimeCX, size.width ());
        m_iCY = max (m_iCY, size.height ());
        size = m_pTotalTime->sizeHint ();
        m_iTimeCX = max (m_iTimeCX, size.width ());
        m_iCY = max (m_iCY, size.height ());
    
        size = m_pLabelTransfer->sizeHint ();
        m_iRateCX = max (m_iRateCX, size.width ());
        m_iCY = max (m_iCY, size.height ());
        size = m_pTransfer->sizeHint ();
        m_iRateCX = max (m_iRateCX, size.width ());
        m_iCY = max (m_iCY, size.height ());
        
        size = m_pButtonReset->sizeHint ();
        m_iStatusCX = max (m_iStatusCX, size.width ());
    }
        
    m_pLabelTime->setFixedSize (m_iTimeCX, m_iCY);
    m_pLabelTime->move (iX, iY);
    m_pTime->setFixedSize (m_iTimeCX, m_iCY);
    m_pTime->move (iX, iY + m_iCY + m_iSpaceCY);
    iX += m_iTimeCX + m_iSpaceCX;

    m_pLabelRate->setFixedSize (m_iRateCX, m_iCY);
    m_pLabelRate->move (iX, iY);
    m_pRate->setFixedSize (m_iRateCX, m_iCY);
    m_pRate->move (iX, iY + m_iCY + m_iSpaceCY);
    iX += m_iRateCX + m_iSpaceCX;

    m_pLabelStatus->setFixedSize (m_iStatusCX, m_iCY);
    m_pLabelStatus->move (iX, m_iY);
    m_pStatus->setFixedSize (m_iStatusCX, m_iCY);
    m_pStatus->move (iX, m_iY + m_iCY + m_iSpaceCY);
    
    m_pCompactButton->setFixedSize (m_iCompactCX, m_iCompactCY);
        
    if ( pIMon->isCompactView () )
    {
        m_pCompactButton->setArrowType (Qt::RightArrow);
        
        m_pCompactButton->move (iX + m_iStatusCX - m_iCompactCX, iY + 2 * m_iCY + 2 * m_iSpaceCY);
        m_pSeparator->setFixedSize (iX + m_iStatusCX - m_iCompactCX - m_iSpaceCX - m_iX, m_iCompactSeparatorCY);
        m_pSeparator->move (m_iX, iY + 2 * m_iCY + 2 * m_iSpaceCY + m_iCompactCY / 2);
        
        setFixedSize (iX + m_iStatusCX + 5, m_iY + 2 * m_iCY + m_iCompactCY + 2 * m_iSpaceCY + 3);

        m_pLabelAdsl->hide ();
        m_pWidgetAdsl->hide ();
        m_pLabelAtm->hide ();
        m_pWidgetAtm->hide ();
        m_pLabelPPP->hide ();
        m_pWidgetPPP->hide ();
        m_pLabelFP->hide ();
        m_pWidgetFP->hide ();
                    
        m_pLabelTotalTime->hide ();
        m_pTotalTime->hide ();
        m_pLabelTransfer->hide ();
        m_pTransfer->hide ();
        m_pButtonReset->hide ();
    }
    else
    {
        m_pCompactButton->setArrowType (Qt::LeftArrow);
    
        iX += m_iStatusCX + m_iSpaceCX;
        int iLedsY = iY + m_iCY + m_iSpaceCY + max (3, (m_iCY - m_iLedCY) / 2);
        m_pLabelAdsl->setFixedSize (m_iAdslAtmPPPCX, m_iCY);
        m_pLabelAdsl->move (iX, iY);
        m_pLabelAdsl->show ();
        m_pWidgetAdsl->setGeometry (iX + (m_iAdslAtmPPPCX - m_iLedCX) / 2, iLedsY, m_iLedCX, m_iLedCY);
        m_pWidgetAdsl->show ();
        iX += m_iAdslAtmPPPCX + m_iSpaceCX;
    
        m_pLabelAtm->setFixedSize (m_iAdslAtmPPPCX, m_iCY);
        m_pLabelAtm->move (iX, iY);
        m_pLabelAtm->show ();
        m_pWidgetAtm->setGeometry (iX + (m_iAdslAtmPPPCX - m_iLedCX) / 2, iLedsY, m_iLedCX, m_iLedCY);
        m_pWidgetAtm->show ();
        iX += m_iAdslAtmPPPCX + m_iSpaceCX;
    
        m_pLabelPPP->setFixedSize (m_iAdslAtmPPPCX, m_iCY);
        m_pLabelPPP->move (iX, iY);
        m_pLabelPPP->show ();
        m_pWidgetPPP->setGeometry (iX + (m_iAdslAtmPPPCX - m_iLedCX) / 2, iLedsY, m_iLedCX, m_iLedCY);
        m_pWidgetPPP->show ();
        iX += m_iAdslAtmPPPCX + m_iSpaceCX;
    
        m_pLabelFP->setFixedSize (m_iAdslAtmPPPCX, m_iCY);
        m_pLabelFP->move (iX, iY);
        m_pLabelFP->show ();
        m_pWidgetFP->setGeometry (iX + (m_iAdslAtmPPPCX - m_iLedCX) / 2, iLedsY, m_iLedCX, m_iLedCY);
        m_pWidgetFP->show ();

        m_pCompactButton->move (iX + m_iAdslAtmPPPCX - m_iCompactCX, iY + 2 * m_iCY + 2 * m_iSpaceCY);
        m_pSeparator->setFixedSize (iX + m_iAdslAtmPPPCX - m_iCompactCX - m_iSpaceCX - m_iX, m_iCompactSeparatorCY);
        m_pSeparator->move (m_iX, iY + 2 * m_iCY + 2 * m_iSpaceCY + m_iCompactCY / 2);
        
        setFixedSize (iX + m_iAdslAtmPPPCX + 5,
                      m_iY + 4 * m_iCY + 4 * m_iSpaceCY + m_iCompactSeparatorCY + m_iCompactCY / 2 + 3);
                
        iX = m_iX;
        iY += 2 * m_iCY + 3 * m_iSpaceCY + m_iCompactSeparatorCY + m_iCompactCY / 2;
        m_pLabelTotalTime->setFixedSize (m_iTimeCX, m_iCY);
        m_pLabelTotalTime->move (iX, iY);
        m_pLabelTotalTime->show ();
        m_pTotalTime->setFixedSize (m_iTimeCX, m_iCY);
        m_pTotalTime->move (iX, iY + m_iCY + m_iSpaceCY);
        m_pTotalTime->show ();
        iX += m_iTimeCX + m_iSpaceCX;
    
        m_pLabelTransfer->setFixedSize (m_iRateCX, m_iCY);
        m_pLabelTransfer->move (iX, iY);
        m_pLabelTransfer->show ();
        m_pTransfer->setFixedSize (m_iRateCX, m_iCY);
        m_pTransfer->move (iX, iY + m_iCY + m_iSpaceCY);
        m_pTransfer->show ();
        iX += m_iRateCX + m_iSpaceCX;
    
        size = m_pButtonReset->sizeHint ();
        m_pButtonReset->setFixedSize (m_iStatusCX, size.height ());
        m_pButtonReset->move (iX, iY + 2 * m_iCY + m_iSpaceCY - size.height ());
        m_pButtonReset->show ();
    }
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfo::showEvent (QShowEvent*)
{
    UpdateLayout ();
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfo::styleChange (QStyle& style)
{
    QWidget::styleChange (style);
    UpdateLayout ();
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfo::destroy (bool destroyWindow, bool destroySubWindows)
{
    AboutToClose ();
    QWidget::destroy (destroyWindow, destroySubWindows);
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfo::AboutToClose ()
{
    if ( m_bOnline )
    {
        m_bOnline = false;
        QDateTime timeUpto = QDateTime::currentDateTime ();
        pIMon->SetOnlineSecs (pIMon->GetOnlineSecs () + m_tmStartOnline.secsTo (timeUpto) - m_OnlineResetSecs);
        m_OnlineResetSecs = 0;
        Log (log_onexit);
    }
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void KIsdnInfo::ToggleCompact ()
{
    pIMon->setCompactView (!pIMon->isCompactView ());
    UpdateLayout ();
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
QString KIsdnInfo::GetInfoTip () const
{
    QString strInfo;
    if ( m_bOnline )
        strInfo = i18n ("Time: %1 Total %2").arg (m_pTime->text ())
                                            .arg (m_pTotalTime->text ());
    return strInfo;
}
