/***************************************************************************
 *   Copyright (C) 2001-2005 AVM GmbH. All rights reserved.                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "dslinfo.h"
#ifdef CPROT
    #include "msg.h"
#endif

#include <klocale.h>

#define __min(a, b) (((a) < (b)) ? (a) : (b))

const char* lpszDslStatus[] =
{
    I18N_NOOP ("Offline"),
    I18N_NOOP ("No Answer"),
    I18N_NOOP ("Unplugged"),
    I18N_NOOP ("Train"),
    I18N_NOOP ("Retrain"),
    I18N_NOOP ("ADSL Online"),
    I18N_NOOP ("ATM Online"),
    I18N_NOOP ("PPP Online")
};

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
CIdent::CIdent ()
{
    Set (NULL);
}

/*--------------------------------------------------------------------------*\
    pLmIdent == NULL to reset members
\*--------------------------------------------------------------------------*/
_cword CIdent::Set (unsigned char* pLmIdent)
{
    _cword wStructLen = 0;
    
    if ( pLmIdent != NULL )
    {
        // struct lenght
        int iIdx = 0;
        _cword wLen = pLmIdent[iIdx++];
        if ( wLen > 0 )
        {
            if ( wLen == 0xFF )
            {
                wLen = *((_cword*)(pLmIdent + iIdx));
                iIdx += sizeof (_cword);
            }
            wStructLen = wLen + iIdx;
    				
            // read 5 parameters if present
            for (int iParam = 0; (iParam < 5) && (iIdx < wLen); iParam++)
            {
                // parameters are structs except 1st parameter
                _cword wParamLen = (iParam == 0) ? sizeof (_cword) : pLmIdent[iIdx++];
                if ( (wParamLen + iIdx) <= wLen )
                {
                    switch (iParam)
                    {
                        case 0:
                            m_wStd = *((_cword*)(pLmIdent + iIdx));
                            break;
                        case 1:
                            memset (&m_RVI, 0, sizeof (m_RVI));
                            memcpy (&m_RVI, pLmIdent + iIdx, __min (sizeof (m_RVI), wParamLen));
                            break;
                        case 2:
                            memset (&m_RVN, 0, sizeof (m_RVN));
                            memcpy (&m_RVN, pLmIdent + iIdx, __min (sizeof (m_RVN), wParamLen));
                            break;
                        case 3:
                            memset (&m_LVI, 0, sizeof (m_LVI));
                            memcpy (&m_LVI, pLmIdent + iIdx, __min (sizeof (m_LVI), wParamLen));
                            break;
                        case 4:
                            memset (&m_LVN, 0, sizeof (m_LVN));
                            memcpy (&m_LVN, pLmIdent + iIdx, __min (sizeof (m_LVN), wParamLen));
                            break;
                    }; // switch
                } // if
                iIdx += wParamLen;
            }; // for
        } // if wLen > 0
        else wStructLen = 1;
    } // pLmIdent != NULL
    else
    {
        // reset
        m_wStd = 0;
        memset (&m_RVI, 0, sizeof (m_RVI));
        memset (&m_RVN, 0, sizeof (m_RVN));
        memset (&m_LVI, 0, sizeof (m_LVI));
        memset (&m_LVN, 0, sizeof (m_LVN));
    }
		
    return wStructLen;
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
CDslInfo::CDslInfo ()
        : CCapiInfo (),
          m_byDslController (0),
          m_Status (offline),
          m_bValidResponse (true),
          m_bFastPath (false),
          m_nRxBytes (0),
          m_fRxRate (0),
          m_nTxBytes (0),
          m_fTxRate (0)
{
    m_pManuAtmData = new unsigned char[GetManuDataSize ()];
    m_bManuAtmPending = false;
}


CDslInfo::~CDslInfo ()
{
    if ( m_pManuAtmData != NULL ) delete m_pManuAtmData;
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
/*static*/ QString CDslInfo::GetStatusString (EDslStatus DslStatus)
{
    return i18n (lpszDslStatus[DslStatus]);
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
unsigned int CDslInfo::GetLastError (QString& strMsg) const
{
    if ( GetLastError () == CapiNoDslcontroller )
    {
        strMsg = i18n ("DSL controller not installed.");
        return CapiNoDslcontroller;
    }
  	
    return CCapiInfo::GetLastError (strMsg);
}

/*--------------------------------------------------------------------------*\
    version number of bin file (fdslbase.bin)
\*--------------------------------------------------------------------------*/
QString CDslInfo::GetDslBinVersion () const
{
    QString str = i18n ("-");
    if ( *((_cdword*)&m_Ident.m_LVN) != 0 )
        str.sprintf ("%d.%02d.%02d-%02d", m_Ident.m_LVN[0], m_Ident.m_LVN[1],
                     m_Ident.m_LVN[2], m_Ident.m_LVN[3]);
		
    return str;
}

/*--------------------------------------------------------------------------*\
    vendor of remote point
\*--------------------------------------------------------------------------*/
QString CDslInfo::GetRemoteVendor () const
{
    QString str = i18n ("-");
    if ( (m_Ident.m_wStd != 0) &&
         (*((_cdword*)&m_Ident.m_RVI) != *((_cdword*)&m_Ident.m_LVI)) )
    {
        if ( (m_Ident.m_wStd == 3) || (m_Ident.m_wStd == 4) )
        {
            str.truncate (0);
            switch ( *((_cword*)&m_Ident.m_RVI) )
            {
                case 0x00B5:
                case 0xB500:
                    switch ( *((_cdword*)(&m_Ident.m_RVI + sizeof (_cword))) )
                    {
                        case 0x43545354:
                            str = i18n ("Texas Instruments");
                            break;
                        case 0x4E505347:
                            str = i18n ("Globespan");
                            break;
                    }; // switch
                    break;
                case 0x0000:
                    if ( *((_cdword*)(&m_Ident.m_RVI + sizeof (_cword))) == 0x56444E41 )
                        str = i18n ("ADI");
                    break;
            }; // switch
								
            if ( str.isEmpty () )
                str = i18n ("G.DMT (%1-%2-%3-%4-%5-%6)").arg (m_Ident.m_RVI[0], 2, 16)
                                                        .arg (m_Ident.m_RVI[1], 2, 16)
                                                        .arg (m_Ident.m_RVI[2], 2, 16)
                                                        .arg (m_Ident.m_RVI[3], 2, 16)
                                                        .arg (m_Ident.m_RVI[4], 2, 16)
                                                        .arg (m_Ident.m_RVI[5], 2, 16);
        } // if 3 || 4
        else
        {
            str = GetFromVendorTable (*((_cdword*)&m_Ident.m_RVI));
            if ( str.isEmpty () )
                str = i18n ("unknown (0x%1)").arg (*((unsigned long*)&m_Ident.m_RVI), 4, 16);
        } // if/else 3 || 4
    } // if m_Ident.wStd
		
    return str;
}

/*--------------------------------------------------------------------------*\
    vendor's version number of remote point
\*--------------------------------------------------------------------------*/
QString CDslInfo::GetRemoteVersion () const
{
    QString str = i18n ("-");
    if ( (m_Ident.m_wStd != 0) &&
         (*((_cword*)&m_Ident.m_RVN) != *((_cdword*)&m_Ident.m_LVN)) &&
         (*((_cdword*)&m_Ident.m_RVI) != 0) )
    {
        if ( (m_Ident.m_wStd == 3) || (m_Ident.m_wStd == 4) )
            str.sprintf ("%d.%d", m_Ident.m_RVN[0], m_Ident.m_RVN[1]);
        else
            str.sprintf ("%d", *((_cword*)&m_Ident.m_RVN));
    } // if m_Ident.wStd
		
    return str;
}

/*--------------------------------------------------------------------------*\
    Initialize before using it
		
    strMsg        to return "human readable" error (language depends on CAPI library)
    Return Value: true if succeded, otherwise false
\*--------------------------------------------------------------------------*/
bool CDslInfo::Register (QString& strMsg)
{
    if ( !IsInit () )
    {
        m_byDslController = 0;
        memset (&m_Ident, 0, sizeof (m_Ident));
        m_bValidResponse = false;
        m_bManuAtmPending =  false;
    }
    return CCapiInfo::Register (strMsg);
}

/*--------------------------------------------------------------------------*\
    release
\*--------------------------------------------------------------------------*/
bool CDslInfo::Release ()
{
    SetInfoLED (false);
    return CCapiInfo::Release ();
}

/*--------------------------------------------------------------------------*\
    get info about CAPI and installed controllers
\*--------------------------------------------------------------------------*/
void CDslInfo::SetInfoLED (bool bOn)
{
    if ( IsInit () && (m_byDslController > 0) )
    {
        *((_cbyte*)m_pManuData) = 3 * sizeof (_cdword);
        *((_cdword*)(m_pManuData + sizeof (_cbyte))) = 1;
        *((_cdword*)(m_pManuData + sizeof (_cbyte) + sizeof (_cdword))) = 8;
        *((_cdword*)(m_pManuData + sizeof (_cbyte) + 2 * sizeof (_cdword))) = (bOn) ? 8 : 0;
        ManufacturerRequest (m_byDslController, CAPI_MANUFUNCTION_LEDS);
    }
}

/*--------------------------------------------------------------------------*\
    get info about CAPI and installed controllers
\*--------------------------------------------------------------------------*/
bool CDslInfo::ReadProfile ()
{
    if ( IsInit () && CCapiInfo::ReadProfile () )
    {
        m_byDslController = 0;
				
        // serch for DSL controller
        if ( ControllerCount () > 0 )
        {
            for (_cbyte byController = 1; byController <= ControllerCount (); byController++)
            {
                CCapiController* pController = GetController (byController);
                if ( (pController != NULL) && pController->IsDsl() )
                {
                    m_byDslController = pController->GetController ();
                    // only one per system!
                    break;
                } // if pController
            } // for byController
        } // if ControllerCount
				
        if ( m_byDslController < 1 ) m_nLastError = CapiNoDslcontroller;
    		
        return (m_nLastError == CapiNoError);
    } // if // m_bInit
		
    return false;
}

/*--------------------------------------------------------------------------*\
    poll CAPI for current info, updates members
\*--------------------------------------------------------------------------*/
bool CDslInfo::Update()
{
    if ( IsInit () && (m_byDslController > 0) ) return CCapiInfo::Update();
		
    return (GetLastError () == CapiNoError);
}

/*--------------------------------------------------------------------------*\
    querey proprietary data from AVM's CAPI implementation for all controllers
\*--------------------------------------------------------------------------*/
_cword CDslInfo::PutRequestMessages ()
{
    _cword wReturn = CapiNoError;
				
    // querey proprietary data from AVM's CAPI implementation for all controllers
    *((_cdword*)m_pManuData) = 0x00000703;
    wReturn = ManufacturerRequest (m_byDslController, CAPI_MANUFUNCTION_ADSLLM);
	
    if ( (wReturn == CapiNoError) && !m_bManuAtmPending )
    {
        *((_cbyte*)m_pManuAtmData) = 0xFF;
        *((_cword*)(m_pManuAtmData + sizeof (_cbyte))) = GetManuDataSize () - 3;
        *((_cdword*)(m_pManuAtmData + sizeof (_cbyte) + sizeof (_cword))) = 9;
        wReturn = ManufacturerRequest (m_byDslController, CAPI_MANUFUNCTION_ATM);
        // don't reuse buffer until CONF
        if ( wReturn == CapiNoError ) m_bManuAtmPending = true;
    }
  		
    return wReturn;
}

/*--------------------------------------------------------------------------*\
    use seperate buffer for CAPI_MANUFUNCTION_ATM requests
\*--------------------------------------------------------------------------*/
_cword CDslInfo::ManufacturerRequest (_cbyte byController, _cdword dwFunction)
{
    if ( dwFunction == CAPI_MANUFUNCTION_ATM )
    {
#ifdef CPROT
        static unsigned char Message[2048];
        capi_fill_MANUFACTURER_REQ (&m_cmsg, m_wApplId, GetNewMessageNumber (),
                                    byController, CAPI_MANUID_AVM, CAPI_MANUCLASS,
                                    CAPI_MANUFUNCTION_ATM, m_pManuAtmData);
        capi_cmsg2message (&m_cmsg, Message);
        CAPI_PROTOCOL_MESSAGE (Message);
        _cword nError = capi20_put_message (m_wApplId, Message);
        if ( nError != CapiNoError )
            CAPI_PROTOCOL_TEXT ("\ncapi20_put_message () returned 0x%hX\n%s\n", nError,
                                capi_info2str (nError));
        return nError;
#else
        return MANUFACTURER_REQ (&m_cmsg, m_wApplId, GetNewMessageNumber (),
                                 byController, CAPI_MANUID_AVM, CAPI_MANUCLASS,
                                 CAPI_MANUFUNCTION_ATM, m_pManuAtmData);
#endif // CPROT																			
    }

    return CCapiInfo::ManufacturerRequest (byController, dwFunction);
}

/*--------------------------------------------------------------------------*\
    receive an process messages
\*--------------------------------------------------------------------------*/
_cword CDslInfo::GetAndProcessMessage ()
{
    // get messages from capi
#ifdef CPROT
    CAPI_MESSAGE pMessage;
    unsigned int nError = capi20_get_message (m_wApplId, &pMessage);
    if ( nError != CapiNoError )
    {
        if ( nError != CapiReceiveQueueEmpty )
            CAPI_PROTOCOL_TEXT ("\ncapi20_get_message () returned 0x%hX\n%s\n", nError,
                                capi_info2str (nError));
        return nError;
    }
    CAPI_PROTOCOL_MESSAGE (pMessage);
    capi_message2cmsg (&m_cmsg, pMessage);
#else
    unsigned int nError = CAPI_GET_CMSG (&m_cmsg, m_wApplId);
    if ( nError != CapiNoError ) return nError;
#endif // CPROT
    		
    switch ( m_cmsg.Subcommand )
    {
        case CAPI_CONF:	/*----- Confirmation -----*/
            {
                // the answer we are waiting for, ignore others
                if ( (MANUFACTURER_CONF_CONTROLLER (&m_cmsg) == m_byDslController) &&
                     (m_cmsg.Command == CAPI_MANUFACTURER) &&
                     (MANUFACTURER_CONF_MANUID (&m_cmsg) == CAPI_MANUID_AVM) &&
                     (MANUFACTURER_CONF_CLASS (&m_cmsg) == 0) &&
                     (MANUFACTURER_CONF_FUNCTION (&m_cmsg) == CAPI_MANUFUNCTION_ATM) )
                    ProcessAtm (MANUFACTURER_CONF_MANUDATA (&m_cmsg));
            }
            break; // case CAPI_CONF
			
        case CAPI_IND: /*----- Indication -----*/
            {
                // the answer we are waiting for, ignore others
                if ( (MANUFACTURER_IND_CONTROLLER (&m_cmsg) == m_byDslController) &&
                     (m_cmsg.Command == CAPI_MANUFACTURER) &&
                     (MANUFACTURER_IND_MANUID (&m_cmsg) == CAPI_MANUID_AVM) &&
                     (MANUFACTURER_IND_CLASS (&m_cmsg) == 0) &&
                     (MANUFACTURER_IND_FUNCTION (&m_cmsg) == CAPI_MANUFUNCTION_ADSLLM) )
                {
                    ProcessAdslLm (MANUFACTURER_IND_MANUDATA (&m_cmsg));
                    m_bValidResponse = true;
                }
                                            
                // should send response
                capi_cmsg_answer (&m_cmsg);
                m_cmsg.Messagenumber = GetNewMessageNumber ();
                capi_put_cmsg (&m_cmsg);
            }
            break;
    } // switch m_cmsg.Subcommand
	
    return CapiNoError;
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void CDslInfo::ProcessAtm (_cstruct pManuData)
{
		// struct length
    _cword wManuDataLen = (pManuData++)[0];
    if ( wManuDataLen == 0xFF )
    {
       	wManuDataLen = *((_cword*)pManuData);
       	pManuData += sizeof (_cword);
    }

    bool bPPPOnline = false;
    if ( (wManuDataLen >= sizeof (_cdword)) && (*((_cdword*)pManuData) == 9) )
    {
        if ( wManuDataLen >= (sizeof (_cdword) + 2 * sizeof (unsigned short) + 12 * sizeof (unsigned long)) )
        {
            if ( m_NetdevStatistics.Update () )
            {
                int iElapsed = 0;
                if ( m_tmSnapshot.isNull () )
                    m_tmSnapshot.start ();
                else
                    iElapsed = m_tmSnapshot.restart ();
    
                // total transfer in current connection
                unsigned int nRxBytes = m_NetdevStatistics.GetRxBytes ();
                unsigned int nTxBytes = m_NetdevStatistics.GetTxBytes ();
    				
                // transfer since recent snapshot
                if ( iElapsed > 0 )
                {
                    if ( nRxBytes > m_nRxBytes )
                        m_fRxRate = ((nRxBytes - m_nRxBytes) * 8) / iElapsed;
                    else
                        m_fRxRate = 0;
                    if ( nTxBytes > m_nTxBytes )
                        m_fTxRate = ((nTxBytes - m_nTxBytes) * 8) / iElapsed;
                    else
                        m_fTxRate = 0;
                }
                else m_fRxRate = m_fTxRate = 0;
    				
                // this snapshot
                m_nRxBytes = nRxBytes;
                m_nTxBytes = nTxBytes;
            }
        		
            // status
            bPPPOnline = true;
        }
    }; // if wManuDataLen

    // status
    if ( bPPPOnline )
    {
        if ( m_Status == atm_online )
        {
#ifdef CPROT
            CAPI_PROTOCOL_TEXT ("changed line status: %s -> %s",
                                (const char*)GetStatusString (m_Status).local8Bit (),
                                (const char*)GetStatusString (ppp_online).local8Bit ());
#endif // CPROT
            m_Status = ppp_online;
            m_iLineEvent = line_up;
            m_nRxBytes = 0;
            m_nTxBytes = 0;
        }
    }
    else
    {
        if ( m_Status == ppp_online )
        {
#ifdef CPROT
            CAPI_PROTOCOL_TEXT ("changed line status: %s -> %s",
                                (const char*)GetStatusString (m_Status).local8Bit (),
                                (const char*)GetStatusString (atm_online).local8Bit ());
#endif // CPROT
            m_Status = atm_online;
            m_iLineEvent = line_down;
            m_nRxBytes = 0;
            m_nTxBytes = 0;
        }
    }

   // buffer could be used again
   m_bManuAtmPending = false;
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
void CDslInfo::ProcessAdslLm (_cstruct pManuData)
{
    // struct length
    _cword wManuDataLen = (pManuData++)[0];
    if ( wManuDataLen == 0xFF )
    {
       	wManuDataLen = *((_cword*)pManuData);
       	pManuData += sizeof (_cword);
    }

    // status
    EDslStatus DslStatus = m_Status;
    if ( wManuDataLen > 0 )
    {
    	switch ( pManuData[0] )
    	{
            case 1:
            case 3:
                DslStatus = adsl_start;
                break;
            case 2:
                DslStatus = adsl_train;
                break;
            case 5:
                DslStatus = adsl_unplugged;
                break;
            case 4:
            case 16:
            case 17:
            case 18:
            case 19:
            case 20:
            case 21:
            case 22:
                DslStatus = adsl_retrain;
                break;
            case 8:
                DslStatus = adsl_online;
                break;
            default:
                DslStatus = offline;
    	} // switch
    } // if wManuDataLen

    bool bParam4recv = false;
    bool bParam4fp = false;
    bool bFp = false;
    if ( DslStatus == adsl_online )
    {
    	// more parameters
    	int iIdx, iNext = sizeof (_cdword), iParam = 0;
    	while ( iNext < wManuDataLen )
    	{
            if ( ++iParam == 1 )
            {
                // version and vendor information
                iNext += m_Ident.Set (pManuData + iNext);
                continue;
            }
    				
            _cword wParamLen = pManuData[iNext++];
            if ( wParamLen == 0xFF )
            {
                if ( (iNext + sizeof (_cword)) > wManuDataLen ) break;  // error: lack bytes
                wParamLen = *((_cword*)(pManuData + iNext));
                iNext += sizeof (_cword);
            }
            iIdx = iNext;
            iNext += wParamLen;
            if ( iNext > wManuDataLen ) break; // error: lack bytes

            if ( iParam == 2 )
            {
                // Fast Path?
                if ( wParamLen >= (2 * sizeof (_cword) + 4 * sizeof (_cdword)) )
                {
                    bParam4fp = true;
                    _cword w = pManuData[iIdx + 2 * sizeof (_cword)];
                    if ( (w < 4) &&
                         (pManuData[iIdx + 4 * sizeof (_cword) + w] != 0) )
                        bFp = true;
                    else
                        bFp = false;
                    w = pManuData[iIdx + 3 * sizeof (_cword)];
                    if ( (w < 4) &&
                         (pManuData[iIdx + 12 * sizeof (_cword) + 12 * sizeof (_cbyte) + w] != 0) )
                        bFp = true;
                    else
                        bFp = false;
                }
            }

            if ( iParam == 4 )
            {
                // ATM link online?
                if ( wParamLen >= (12 * sizeof (_cword) + 16 * sizeof (_cbyte)) )
                {
                    bParam4recv = true;
                    for (int ii = 0; ii < (4 * (int)sizeof (_cdword)); ii++)
                    {
                        if ( pManuData[iIdx + 2 * sizeof (_cword) + ii] != 0 )
                        {
                            DslStatus = atm_online;
                            break;
                        }
                    } // for
                }
            } // if
    	} // while
    } // if DslStatus == adsl_online
    else
    {
        bParam4fp = true;
        bFp = false;
    }

    // if no ATM link info received -> don't touch status if at least adsl_online
    if ( !bParam4recv && (DslStatus == adsl_online) && (m_Status > adsl_online) ) DslStatus = atm_online;

    // FP-Info received
    if ( bParam4fp && (m_bFastPath != bFp) ) m_bFastPath = bFp;

    // if still atm_online -> leave ppp state
    if ( ((DslStatus < atm_online) || (m_Status != ppp_online)) &&
         (DslStatus != m_Status) )
    {
#ifdef CPROT
        CAPI_PROTOCOL_TEXT ("changed line status: %s -> %s",
                            (const char*)GetStatusString (m_Status).local8Bit (),
                            (const char*)GetStatusString (DslStatus).local8Bit ());
#endif // CPROT
        m_Status = DslStatus;
        if ( m_Status != ppp_online )
        {
            m_nRxBytes = 0;
            m_nTxBytes = 0;
        }
    }
}

/*--------------------------------------------------------------------------*\
\*--------------------------------------------------------------------------*/
QString CDslInfo::GetFromVendorTable (_cdword dwRVI) const
{
    switch ( dwRVI )
    {
      	case 0x0001: return i18n ("not allocated");
      	case 0x0002: return i18n ("Westell, Inc.");
      	case 0x20fb:
      	case 0x0003: return i18n ("ECI Telecom");
      	case 0x0004: return i18n ("Texas Instruments");
      	case 0x0005: return i18n ("Intel");
      	case 0x0006: return i18n ("Amati Communcations Corp.");
      	case 0x0007: return i18n ("General Data Communications, Inc.");
      	case 0x0008: return i18n ("Level One Communications");
      	case 0x0009: return i18n ("Crystal Semiconductor");
      	case 0x000A: return i18n ("Lucent Technologies");
      	case 0x000B: return i18n ("Aware, Inc.");
      	case 0x000C: return i18n ("Brooktree");
      	case 0x000D: return i18n ("NEC");
      	case 0x000E: return i18n ("Samsung");
      	case 0x000F: return i18n ("Northern Telecom, Inc.");
      	case 0x0010: return i18n ("PairGain Technologies");
      	case 0x0011: return i18n ("Paradyne");
      	case 0x0012: return i18n ("Adtran");
      	case 0x0013: return i18n ("INC");
      	case 0x0014: return i18n ("ADC Telecommunications");
      	case 0x0015: return i18n ("Motorola");
      	case 0x0016: return i18n ("IBM Corp.");
      	case 0x0017: return i18n ("Newbridge Network Corp.");
      	case 0x0018: return i18n ("DSC");
      	case 0x0019: return i18n ("Teltrend");
      	case 0x001A: return i18n ("Exar Corp.");
      	case 0x001B: return i18n ("Siemens Telecom Networks");
      	case 0x001C: return i18n ("Analog Devices");
      	case 0x001D: return i18n ("Nokia");
      	case 0x001E: return i18n ("Ericsson Information Systems");
      	case 0x001F: return i18n ("Tellabs Operations, Inc.");
      	case 0x0020: return i18n ("Orckit Communications, Inc.");
      	case 0x0021: return i18n ("AWA");
      	case 0x0022: return i18n ("Alcatel Network Systems, Inc.");
      	case 0x0023: return i18n ("National Semiconductor Corp.");
      	case 0x0024: return i18n ("Italtel");
      	case 0x0025: return i18n ("SAT - Societe Anonyme de Telecommunications");
      	case 0x0026: return i18n ("Fujitsu Network Trans. Systems");
      	case 0x0027: return i18n ("MITEL");
      	case 0x0028: return i18n ("Conklin Corp.");
      	case 0x0029: return i18n ("Diamond Lane");
      	case 0x002A: return i18n ("Cabletron Systems, Inc.");
      	case 0x002B: return i18n ("Davicom Semiconductor, Inc.");
      	case 0x002C: return i18n ("Metalink");
      	case 0x002D: return i18n ("Pulsecom");
      	case 0x002E: return i18n ("US Robotics");
      	case 0x002F: return i18n ("AG Communications Systems");
      	case 0x0030: return i18n ("Rockwell");
      	case 0x0031: return i18n ("Harris");
      	case 0x0032: return i18n ("Hayes Microcomputer Products, Inc.");
      	case 0x0033: return i18n ("Co-optic");
      	case 0x0034: return i18n ("Netspeed, Inc.");
      	case 0x0035: return i18n ("3-Com");
      	case 0x0036: return i18n ("Copper Mountain, Inc");
      	case 0x0037: return i18n ("Silicon Automation Systems, Ltd");
      	case 0x0038: return i18n ("Ascom");
      	case 0x0039: return i18n ("Globespan Semiconductor, Inc.");
      	case 0x003A: return i18n ("STMicroelectronics");
      	case 0x003B: return i18n ("Coppercom");
      	case 0x003C: return i18n ("Compaq Computer Corp.");
      	case 0x003D: return i18n ("Integrated Technology Express");
      	case 0x003E: return i18n ("Bay Networks, Inc.");
      	case 0x003F: return i18n ("Next Level Communications");
      	case 0x0040: return i18n ("Multi-Tech Systems, Inc.");
      	case 0x0041: return i18n ("AMD");
      	case 0x0042: return i18n ("Sumitomo Electric");
      	case 0x0043: return i18n ("Philips M&N Systems");
      	case 0x0044: return i18n ("Efficient Networks, Inc.");
      	case 0x0045: return i18n ("Interspeed");
      	case 0x0046: return i18n ("Cisco Systems");
      	case 0x0047: return i18n ("Tollgrade Communications, Inc.");
      	case 0x0048: return i18n ("Cayman Systems");
      	case 0x0049: return i18n ("FlowPoint Corp.");
      	case 0x004A: return i18n ("I.C.COM");
      	case 0x004B: return i18n ("Matsushita");
      	case 0x004C: return i18n ("Siemens Semiconductor");
      	case 0x004D: return i18n ("Digital Link");
      	case 0x004E: return i18n ("Digitel");
      	case 0x004F: return i18n ("Alcatel Microelectronics");
      	case 0x0050: return i18n ("Centillium Corp.");
      	case 0x0051: return i18n ("Applied Digital Access, Inc.");
      	case 0x0052: return i18n ("Smart Link, Ltd.");
    }; // switch
		
    return QString ();
}
