/***************************************************************************
 *   Copyright (C) 2001-2003 AVM GmbH. All rights reserved.                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CCAPIINFO_H
#define CCAPIINFO_H

#undef __NO_CAPIUTILS__
#include <capi20.h>

#include <qstring.h>
#include <qsortedlist.h>
#include <qfile.h>


#define CAPIINFO_MAX_BCHANNELS							255
#define CAPI_MANUID_AVM 								0x214D5641
#define CAPI_MANUCLASS									0
#define CAPI_MANUFUNCTION_GETBCHANNELINFO 	            4

#define CAPI_B1PROTOCOL_ATM 						    (1 << 28)
#define CAPI_B2PROTOCOL_PPPOE							(1 << 30)
#define CAPI_B3PROTOCOL_PPPOE							(1 << 30)


class CCapiController
{
	friend class CCapiInfo;

public:
  	CCapiController (_cbyte byController);

  	_cbyte GetController () const { return m_byController; }
  	_cword GetChannelCount () const { return m_wChanCount; }
  	_cword GetChannelCount (bool bChannelUp) const;
  	bool GetChannelInfo (_cbyte byChannel) const;
    bool GotValidResponse () const { return m_bValidResponse; }
  	
  	// Info
  	QString GetManufacturer ();
	QString GetVersion () const;
	QString GetManufacturerVersion ();
	QString GetSerialNumber () const;
  	
  	// Profile
  	_cdword GetOptions () const { return m_dwOptions; }
  	_cword GetDProtocol () const { return m_wDProtocol; }
  	_cbyte GetLine () const { return m_byLine; }
  	_cdword GetB1 () const { return m_dwB1; }
  	_cdword GetB2 () const { return m_dwB2; }
  	_cdword GetB3 () const { return m_dwB3; }

    bool IsDsl () const;

  	int operator< (const CCapiController& C) const { return (m_byController < C.m_byController); }
  	int operator== (const CCapiController& C) const { return (m_byController == C.m_byController); }
  	
protected:
  	_cword 				m_wChanCount;
  	bool 				m_bChannelInfo[CAPIINFO_MAX_BCHANNELS];
  	_cdword				m_dwOptions;
  	_cdword				m_dwB1;
  	_cdword				m_dwB2;
  	_cdword				m_dwB3;
  	int					m_iAvmCapi;
    bool                m_bValidResponse;
  	
  	// manufacturer specific profile information
  	_cword				m_wDProtocol;
  	_cbyte				m_byLine;

	void SetProfile (_cbyte* pProfile);
  	  	
private:
	_cbyte 				m_byController;
};

inline bool CCapiController::GetChannelInfo (_cbyte byChannel) const
{
		return (byChannel < m_wChanCount) ? m_bChannelInfo[byChannel] : false;
}

inline bool CCapiController::IsDsl () const
{
    return (((GetB1 () & CAPI_B1PROTOCOL_ATM) != 0) &&
            ((GetB2 () & CAPI_B2PROTOCOL_PPPOE) != 0) &&
            ((GetB3 () & CAPI_B3PROTOCOL_PPPOE) != 0) );
}


class CCapiInfo
{
public:
		enum ELineEvent { line_nop = 0, line_up = 1, line_down = 2 };

		CCapiInfo();
		virtual ~CCapiInfo();

  	virtual bool Register(QString& strMsg);
  	virtual bool Release ();
  	bool IsInit() const { return m_bInit; }
  	bool IsAvmCapi () const { return ( IsInit () && m_bAvmCapi ); }
  	virtual unsigned int GetLastError () const { return m_nLastError; }
  	virtual unsigned int GetLastError (QString& strMsg) const;
  	unsigned int GetChannelCount (_cbyte byController);
  	unsigned int GetChannelCount (bool bChannelUp, _cbyte byController);
		CCapiController* GetController (_cbyte byController);

  	virtual bool ReadProfile ();
  	virtual bool Update();
  	virtual int TestLineEvent ();
  	
  	unsigned int ApplCount () const;
  	_cbyte ControllerCount () const { return m_Controller.count (); }
  	bool IsBChannelUp (_cbyte byController, _cbyte byBChannel);
    bool IsDslOnly ();
  	
#ifdef CPROT  	
  	static bool OpenProtocol (const char* szFileName);
  	static void CloseProtocol ();
#endif

protected:
  	bool													m_bInit;
  	_cword												m_wApplId;
  	_cword												m_wLastMsgNumber;
  	_cmsg													m_cmsg;
  	_cstruct											m_pManuData;
  	unsigned int									m_nLastError;
    int                           m_iLastErrno;
  	int														m_iLineEvent;

  	_cword												m_wApplCount;
  	bool													m_bAvmCapi;
  	QSortedList<CCapiController>	m_Controller;
	
  	virtual unsigned int GetManuDataSize () const { return 64; }
		void SetProfile (_cbyte* pProfile);
  	_cword GetNewMessageNumber ();
  	virtual _cword ManufacturerRequest (_cbyte byController, _cdword dwFunction);
  	virtual _cword PutRequestMessages ();
  	virtual _cword GetAndProcessMessage ();
};


inline _cword CCapiInfo::GetNewMessageNumber ()
{
		return (m_wLastMsgNumber = (_cword)(++m_wLastMsgNumber & 0x7FFF));
}

inline unsigned int CCapiInfo::ApplCount () const
{
	return ( m_wApplCount > 0 ) ? m_wApplCount - 1 : 0;
}

#endif // CCAPIINFO_H
