/* 
 * Copyright 2015 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License ("License") as 
 * published by the Free Software Foundation  (version 3 of the License). 
 * This software is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the 
 * License you received along with this software for more details.
 */

package de.avm.android.tr064.soap.wlanconfiguration;

import java.io.IOException;
import android.text.TextUtils;
import de.avm.android.tr064.exceptions.BaseException;
import de.avm.android.tr064.exceptions.DataMisformatException;
import de.avm.android.tr064.model.UnsignedNumber;
import de.avm.android.tr064.model.WLANInfo;
import de.avm.android.tr064.soap.ISoapCredentials;

public class GetAssociatedDeviceInfo extends WLANSoapHelper<WLANInfo>
{
	private int mIndex;
	private String mMac = null;
	
	/**
	 * Gets WLAN info by index
	 * 
	 * @param interfaceIndex
	 * 		Index of WLAN configuration interface
	 * @param deviceIndex
	 * 		Index of device to get info from
	 * @throws IllegalArgumentException
	 * 		illegal value of deviceIndex 
	 */
	public GetAssociatedDeviceInfo(ISoapCredentials soapCredentials, int interfaceIndex, int deviceIndex)
	{
		super(soapCredentials, interfaceIndex);
		UnsignedNumber.validateUi2(deviceIndex, "deviceIndex");
		mIndex = deviceIndex;
	}
	
	/**
	 * Gets WLAN info by MAC address
	 * 
	 * @param interfaceIndex
	 * 		Index of WLAN configuration interface
	 * @param mac
	 * 		MAC of device to get info from
	 */
	public GetAssociatedDeviceInfo(ISoapCredentials soapCredentials, int interfaceIndex, String mac)
	{
		super(soapCredentials, interfaceIndex);
		mMac = mac;
	}

    public int getIndex()
    {
        return mIndex;
    }

    /**
     * @param deviceIndex
     * 		Index of device to get info from
     * @throws IllegalArgumentException
     * 		illegal value of deviceIndex
     */
    public GetAssociatedDeviceInfo setIndex(int deviceIndex)
    {
        UnsignedNumber.validateUi2(deviceIndex, "deviceIndex");
        mIndex = deviceIndex;
        mMac = null;
        return this;
    }

    public String getMac()
    {
        return mMac;
    }

    public GetAssociatedDeviceInfo setMac(String mac)
    {
        mMac = mac;
        return this;
    }

	@Override
	public String getSoapMethodParameter()
	{
		return (TextUtils.isEmpty(mMac)) ?
				"<NewAssociatedDeviceIndex>" + Integer.toString(mIndex) + "</NewAssociatedDeviceIndex>" :
					"<NewAssociatedDeviceMACAddress>" + mMac + "</NewAssociatedDeviceMACAddress>";
	}

	@Override
	public WLANInfo getQualifiedResult()
			throws DataMisformatException, BaseException, IOException
	{
		String input = getSoapBody();
		
		WLANInfo result = new WLANInfo();
		if (TextUtils.isEmpty(mMac))
			result.setMacAdress(getValueByName("NewAssociatedDeviceMACAddress", input));
		else
			result.setMacAdress(mMac);
		result.setIpAdress(getValueByName("NewAssociatedDeviceIPAddress", input));
		String value = getValueByName("NewAssociatedDeviceAuthState", input);
		if ((value.length() > 0) && !value.equals("0")) result.setAuthState(true);
		result.setBandwidth(getIntByName("NewX_AVM-DE_Speed", input));
		result.setSignalStrength(getIntByName("NewX_AVM-DE_SignalStrength", input));
		
		return result;
	}

	@Override
	public String getSoapMethod()
	{
		return (TextUtils.isEmpty(mMac)) ?
				"GetGenericAssociatedDeviceInfo" : "GetSpecificAssociatedDeviceInfo";
	}
}
