/* 
 * Copyright 2015 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License ("License") as 
 * published by the Free Software Foundation  (version 3 of the License). 
 * This software is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the 
 * License you received along with this software for more details.
 */

package de.avm.android.tr064.soap.appsetup;

import de.avm.android.tr064.exceptions.BaseException;
import de.avm.android.tr064.exceptions.DataMisformatException;
import de.avm.android.tr064.model.RightEnum;
import de.avm.android.tr064.soap.ISoapCredentials;

import java.io.IOException;

public class RegisterApp extends AppSetupSoapHelper<Boolean>
{
    private String mAppId;

    private String mAppDisplayName;
    private String mAppDeviceMAC;
    private String mAppUsername;
    private String mAppPassword;
    private RightEnum mAppRight;
    private RightEnum mNasRight;
    private RightEnum mPhoneRight;
    private RightEnum mHomeautoRight;
    private boolean mAppInternetRights;
    /**
     * Registers a new app instance
     *
     * @param appId
     * 		Identifier of the app instance the VPN configuration belongs to
     * @param appDisplayName
     * 		User friendly display name of the app instance
     * @param appDeviceMAC
     * 		MAC address of the device (or device interface). Empty string means: unknown
     * @param appUsername
     * 		Username for the app instance
     * @param appPassword
     * 		Password for the app instance
     * @param appRight
     * 		FRITZ!OS app specific configuration right
     * @param nasRight
     * 		FRITZ!OS NAS specific right
     * @param phoneRight
     * 		FRITZ!OS phone specific right
     * @param homeautoRight
     * 		FRITZ!OS Smarthome specific right
     * @param appInternetRights
     * 		Does the app instance want access the box from the internet?
     */
    public RegisterApp(ISoapCredentials soapCredentials, String appId, String appDisplayName,
            String appDeviceMAC, String appUsername, String appPassword,
            RightEnum appRight, RightEnum nasRight, RightEnum phoneRight,
            RightEnum homeautoRight, boolean appInternetRights)
    {
        super(soapCredentials);

        mAppId = appId;
        mAppDisplayName = appDisplayName;
        mAppDeviceMAC = appDeviceMAC;
        mAppUsername = appUsername;
        mAppPassword = appPassword;
        mAppRight = appRight;
        mNasRight = nasRight;
        mPhoneRight = phoneRight;
        mHomeautoRight = homeautoRight;
        mAppInternetRights = appInternetRights;
    }

	@Override
	public Boolean getQualifiedResult()
			throws DataMisformatException, BaseException, IOException
	{
        // no out parameter, check for error response
        exceptOnFault(getSoapBody());
        return Boolean.TRUE;
    }

    @Override
    public String getSoapMethodParameter()
    {
        return "<NewAppId>" + encodeEntities(mAppId) + "</NewAppId>" +
                "<NewAppDisplayName>" + encodeEntities(mAppDisplayName) + "</NewAppDisplayName>" +
                "<NewAppDeviceMAC>" + mAppDeviceMAC + "</NewAppDeviceMAC>" +
                "<NewAppUsername>" + encodeEntities(mAppUsername) + "</NewAppUsername>" +
                "<NewAppPassword>" + encodeEntities(mAppPassword) + "</NewAppPassword>" +
                "<NewAppRight>" + mAppRight.toString() + "</NewAppRight>" +
                "<NewNasRight>" + mNasRight.toString() + "</NewNasRight>" +
                "<NewPhoneRight>" + mPhoneRight.toString() + "</NewPhoneRight>" +
                "<NewHomeautoRight>" + mHomeautoRight.toString() + "</NewHomeautoRight>" +
                "<NewAppInternetRights>" + Integer.toString(mAppInternetRights ? 1 : 0) + "</NewAppInternetRights>";
    }

    @Override
    protected String filterSoapBeforeTrace(String body)
    {
        return replaceSecrets(body, new String[]{"NewAppUsername", "NewAppPassword"});
    }

	@Override
	public String getSoapMethod()
	{
		return "RegisterApp";
	}

    public String getAppId() {
        return mAppId;
    }

    public void setAppId(String mAppId) {
        this.mAppId = mAppId;
    }

    public String getAppDisplayName() {
        return mAppDisplayName;
    }

    public void setAppDisplayName(String mAppDisplayName) {
        this.mAppDisplayName = mAppDisplayName;
    }
}