/* 
 * Copyright 2013 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License ("License") as 
 * published by the Free Software Foundation  (version 3 of the License). 
 * This software is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the 
 * License you received along with this software for more details.
 */

package de.avm.android.tr064.model;

import java.util.Locale;

import android.os.Parcel;
import android.os.Parcelable;

/*
 *  Represents information about WLAN hybrid mode, as received from or set to the Fritzbox 
 */
public class WLANHybridMode implements Parcelable
{
	/*
		NewEnable            out|in    Bool
		NewBeaconType        out|in    String "None", "WPAand11i"
		NewKeyPassphrase     out|in    String
		NewSSID              out|in    String
		NewBSSID             out|in    String MacAddress: Format: "00:11:22:33:44:55"
		NewTrafficMode       out|in    String - use of this WAN connection: "disabled", "hybrid", "exclusive"
		NewManualSpeed       out|in    Bool  - 1: use MaxSpeedDS and MaxSpeedUS values,  0 = use default values
		NewMaxSpeedDS        out|in    ui4   - applicable only if ManualSpeed=1 in kbit/s
		NewMaxSpeedUS        out|in    ui4   - applicable only if ManualSpeed=1 in kbit/s
	*/

	public enum BeaconType
	{
		None, WPAand11i; // no capitals here!!
		
		public static BeaconType getFromKey(String key)
		{
			for(BeaconType t : BeaconType.values())
			{
				if(key.toLowerCase(Locale.US).equals(t.toString()
						.toLowerCase(Locale.US)))
					return t;
			}
			// default
			return None;
		}
	}

	public enum TrafficMode
	{
		disabled, hybrid, exclusive; // no capitals here!!
		
		public static TrafficMode getFromKey(String key)
		{
			for(TrafficMode t : TrafficMode.values())
			{
				if(key.toLowerCase(Locale.US).equals(t.toString()
						.toLowerCase(Locale.US)))
					return t;
			}
			// default
			return disabled;
		}
	}
	
	private boolean mEnable = false;
	private BeaconType mBeaconType = BeaconType.None;
	private String mKeyPassphrase = "";
	private String mSsid = "";
	private String mBssid = "";
	private TrafficMode mTrafficMode = TrafficMode.disabled;
	private boolean mManualSpeed = false;
	private long mMaxSpeedDS = 0;
	private long mMaxSpeedUS = 0;

	public boolean isEnable()
	{
		return mEnable;
	}
	
	public void setEnable(boolean on)
	{
		mEnable = on;
	}
	
	public BeaconType getBeaconType()
	{
		return mBeaconType;
	}
	
	public void setBeaconType(BeaconType type)
	{
		mBeaconType = type;
	}
	
	public String getKeyPassphrase()
	{
		return mKeyPassphrase;
	}
	
	public void setKeyPassphrase(String phrase)
	{
		mKeyPassphrase = phrase;
	}
	
	public String getSsid()
	{
		return mSsid;
	}
	
	public void setSsid(String ssid)
	{
		mSsid = ssid;
	}
	
	public String getBssid()
	{
		return mBssid;
	}
	
	public void setBssid(String bssid)
	{
		mBssid = bssid;
	}
	
	public TrafficMode getTrafficMode()
	{
		return mTrafficMode;
	}
	
	public void setTrafficMode(TrafficMode mode)
	{
		mTrafficMode = mode;
	}
	
	public boolean isManualSpeed()
	{
		return mManualSpeed;
	}
	
	public long getMaxSpeedDS()
	{
		return mMaxSpeedDS;
	}
	
	public long getMaxSpeedUS()
	{
		return mMaxSpeedDS;
	}
	
	public void setDefaultSpeed()
	{
		mManualSpeed = false;
		mMaxSpeedDS = 0;
		mMaxSpeedUS = 0;
	}
	
	public void setManualSpeed(long maxSpeedDS, long maxSpeedUS)
	{
		mManualSpeed = false;
		mMaxSpeedDS = maxSpeedDS;
		mMaxSpeedUS = maxSpeedUS;
	}
	
	public WLANHybridMode()
	{
	}
	
	public WLANHybridMode(Parcel in)
	{
		mEnable = in.readInt() != 0;
		mBeaconType = BeaconType.values()[in.readInt()];
		mKeyPassphrase = in.readString();
		mSsid = in.readString();
		mBssid = in.readString();
		mTrafficMode = TrafficMode.values()[in.readInt()];
		mManualSpeed = in.readInt() != 0;
		mMaxSpeedDS = in.readLong();
		mMaxSpeedUS = in.readLong();
	}

	public int describeContents()
	{
		return 0;
	}

	public static final Parcelable.Creator<WLANHybridMode> CREATOR =
			new Parcelable.Creator<WLANHybridMode>()
	{
		public WLANHybridMode createFromParcel(Parcel in)
		{
			return new WLANHybridMode(in);
		}

		public WLANHybridMode[] newArray(int size)
		{
			return new WLANHybridMode[size];
		}
	};

	public void writeToParcel(Parcel out, int flags)
	{
		out.writeInt((mEnable) ? 1 : 0);
		out.writeInt(mBeaconType.ordinal());
		out.writeString(mKeyPassphrase);
		out.writeString(mSsid);
		out.writeString(mBssid);
		out.writeInt(mTrafficMode.ordinal());
		out.writeInt((mManualSpeed) ? 1 : 0);
		out.writeLong(mMaxSpeedDS);
		out.writeLong(mMaxSpeedUS);
	}
}