/* 
 * Copyright 2012 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License ("License") as 
 * published by the Free Software Foundation  (version 3 of the License). 
 * This software is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the 
 * License you received along with this software for more details.
 */

package de.avm.android.tr064.model;

import java.util.Locale;

import android.content.Context;
import android.os.Parcel;
import android.os.Parcelable;
import android.text.TextUtils;

/* Represents a number in a contact*/
public class ContactNumber implements Parcelable, IContactable
{
	public enum Type
	{
		HOME, MOBILE, WORK, INTERN, MEMO, OTHER,
		FIXEDLINE, FAX_WORK, FAX_HOME, GVOICE, PAGER;

		/**
		 * Gets the text resource.
		 * 
		 * @param context
		 *            the context
		 * 
		 * @return the text resource
		 */
		public String getTextResource(Context context)
		{
			int id = context.getResources()
					.getIdentifier("contact_number_" + toString()
							.toLowerCase(Locale.US), "string",
							context.getApplicationContext().getPackageName());
			if(id == 0) return "";
			return context.getString(id);
		}
		
		/**
		 * Gets the number type for key.
		 * 
		 * @param key
		 *            the key
		 * 
		 * @return the number type for key
		 */
		public static Type getFromKey(String key)
		{
			for(Type t : Type.values())
			{
				if(key.toLowerCase(Locale.US).equals(t.toString()
						.toLowerCase(Locale.US)))
				{
					return t;
				}
			}
			// default
			return OTHER;
		}
	};

	private static final String CUSTOM_LABEL = "label:";
	
	protected Type mType = Type.OTHER;
	protected String mCustomType = null;
	protected String mNumber = "";
	protected String mQuickdial = "";

	/**
	 * Instantiates a new contact number.
	 */
	public ContactNumber()
	{
		super();
	}

	/**
	 * Instantiates a new contact number.
	 */
	public ContactNumber(String number)
	{
		super();
		setNumber(number);
	}

	/**
	 * Instantiates a new contact number.
	 * 
	 * @param in
	 *            the in
	 */
	public ContactNumber(Parcel in)
	{
		mType = Type.values()[in.readInt()];
		mCustomType = in.readString();
		mNumber = in.readString();
		mQuickdial = in.readString();
	}

	/**
	 * Gets the type.
	 * 
	 * @return the type
	 */
	public Type getType()
	{
		return mType;
	}

	/**
	 * Sets the type.
	 * 
	 * @param type
	 *            the new type
	 */
	public void setType(Type type)
	{
		mType = type;
		mCustomType = null;
	}

	/**
	* Sets the type to custom type.
	* 
	* @param type
	*            the custom type
	*/
	public void setCustomType(String customType)
	{
		mType = Type.OTHER;
		mCustomType = (TextUtils.isEmpty(customType)) ? null : customType;
	}

	/**
	 * Sets the type from phonebook's type attribute
	 * 
	 * @param key
	 *			the attribute value
	 */
	public void setTypeFromKey(String key)
	{
		mType = Type.getFromKey(key);
		if ((mType == Type.OTHER) && key.toLowerCase(Locale.US)
				.startsWith(CUSTOM_LABEL))
			mCustomType = key.substring(CUSTOM_LABEL.length());
	}
	
	/**
	 * Gets phonebook's type attribute
	 * 
	 * @param customTypeSupport
	 * 			true to generate custom types too
	 * @return
	 * 			the phonebook's type attribute value
	 */
	public String getTypeAsKey(boolean customTypeSupport)
	{
		if (customTypeSupport && (mType == Type.OTHER))
			return CUSTOM_LABEL + ((mCustomType == null) ? "" : mCustomType);
		return mType.toString().toLowerCase(Locale.US);
	}

	/**
	 * Gets the plain number.
	 * 
	 * @return the number
	 */
	public String getNumberRaw()
	{
		return mNumber;
	}
	/**
	 * Gets the callable number.
	 * 
	 * @return the number
	 */
	public String getNumber()
	{
		if (mNumber.indexOf("@") >= 0)
		{
			// for SIP address as number return internal shortcut number
			String quickdial = getQuickdial();
			switch (quickdial.length())
			{
				case 1: return "**70" + quickdial;
				case 2: return "**7" + quickdial;
			}
			return ""; 
		}

		return mNumber;
	}

	/**
	 * Sets the number.
	 * 
	 * @param number
	 *            the new number
	 */
	public void setNumber(String number)
	{
		mNumber = (number == null) ? "" : number;
	}
	
	public String getQuickdial()
	{
		return mQuickdial;
	}
	
	public void setQuickdial(String quickdial)
	{
		mQuickdial = (quickdial == null) ? "" : quickdial;
	}
	
	public int describeContents()
	{
		return 0;
	}

	public void writeToParcel(Parcel out, int flags)
	{
		out.writeInt(mType.ordinal());
		out.writeString(mCustomType);
		out.writeString(mNumber);
		out.writeString(mQuickdial);
	}

	public static final Parcelable.Creator<ContactNumber> CREATOR =
			new Parcelable.Creator<ContactNumber>()
	{
		public ContactNumber createFromParcel(Parcel in)
		{
			return new ContactNumber(in);
		}

		public ContactNumber[] newArray(int size)
		{
			return new ContactNumber[size];
		}
	};

	public String getAddressDisplay()
	{
		return getNumberRaw();
	}

	public String getTypeDisplay(Context context)
	{
		if ((mType == Type.OTHER) && !TextUtils.isEmpty(mCustomType))
			return mCustomType;
		return mType.getTextResource(context);
	}

	public String getAddress()
	{
		return getNumber();
	}
}
