/* 
 * Copyright 2015 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License ("License") as 
 * published by the Free Software Foundation  (version 3 of the License). 
 * This software is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the 
 * License you received along with this software for more details.
 */

package de.avm.android.tr064.model;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

import android.os.Parcel;
import android.os.Parcelable;
import de.avm.android.tr064.exceptions.BaseException;
import de.avm.android.tr064.exceptions.DataMisformatException;
import de.avm.android.tr064.sax.SAXCallLogHandler;

/* Represents a list of calls as received from the FRITZ!Box */
public class CallLog implements Parcelable
{
	private ArrayList<Call> mCalls = new ArrayList<Call>();
    private long mTimestamp = 0;

    /**
	 * Instantiates a new call log.
	 */
	public CallLog()
    {
		super();
	}

	/**
	 * Instantiates a new call log.
	 * 
	 * @param in
	 *            the in
	 */
	public CallLog(Parcel in)
	{
		Parcelable[] content = in.readParcelableArray(null);
		for (Parcelable item : content)
			mCalls.add((Call) item);
        mTimestamp = in.readLong();
    }

	/**
	 * Instantiates a new call log.
	 * 
	 * @param xmlStream
	 *            the xml stream
	 */
	public CallLog(InputStream xmlStream)
    {
		SAXParserFactory spf = SAXParserFactory.newInstance();
		SAXParser sp;
		try
        {
			sp = spf.newSAXParser();
			XMLReader xr = sp.getXMLReader();
			SAXCallLogHandler saxHandler = new SAXCallLogHandler(this);
			xr.setContentHandler(saxHandler);
			xr.parse(new InputSource(xmlStream));
		}
        catch (ParserConfigurationException e)
        {
			throw new BaseException("Unhandled configuration Exception", e);
		}
        catch (SAXException e)
        {
			throw new DataMisformatException("Invalid Calllog Data",e);
		}
        catch (IOException e)
        {
			throw new DataMisformatException("Invalid Callog Data",e);
		}
	}

	/**
	 * Gets the calls.
	 * 
	 * @return the calls
	 */
	public ArrayList<Call> getCalls()
    {
		return mCalls;
	}

	/**
	 * Adds the call.
	 * 
	 * @param call
	 *            the call
	 */
	public void addCall(Call call)
    {
		mCalls.add(call);
	}

	/**
	 * Inserts the call.
	 * 
	 * @param position
	 *            position to insert to
	 * @param call
	 *            the call
	 */
	public void insertCall(int position, Call call)
	{
		mCalls.add(position, call);
	}

    /**
     * Gets last changed timestamp of the list
     * @return
     */
    public long getTimestamp()
    {
        return mTimestamp;
    }

    public void setTimestamp(long timestamp)
    {
        mTimestamp = timestamp;
    }

	public static final Parcelable.Creator<CallLog> CREATOR =
			new Parcelable.Creator<CallLog>()
	{
		public CallLog createFromParcel(Parcel in)
        {
			return new CallLog(in);
		}

		public CallLog[] newArray(int size)
        {
			return new CallLog[size];
		}
	};



	public int describeContents()
    {
		return 0;
	}

	public void writeToParcel(Parcel out, int flags)
	{
		Call[] callsArray = mCalls.toArray(new Call[0]);
		out.writeParcelableArray(callsArray, 0);
        out.writeLong(mTimestamp);
	}
}
