/* 
 * Copyright 2015 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License ("License") as 
 * published by the Free Software Foundation  (version 3 of the License). 
 * This software is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the 
 * License you received along with this software for more details.
 */

package de.avm.fundamentals.activities;


import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.support.v4.app.ShareCompat;
import android.support.v7.app.ActionBar;
import android.support.v7.app.ActionBarActivity;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.inputmethod.InputMethodManager;
import android.widget.EditText;
import android.widget.Switch;
import android.widget.Toast;
import de.avm.fundamentals.R;
import de.avm.fundamentals.logger.FileLog;
import de.avm.fundamentals.utils.Utils;

import java.io.IOException;
import java.util.List;

public class FeedbackActivity extends ActionBarActivity {

    private static final int REQUEST_CODE_FEEDBACK_SEND = 0;
    private EditText mBodyText;
    private Switch mSendLogSwitch;
    private ActionBar mActionBar;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.feedback_activity);
        initViews();
        initActionBar();
    }

    private void initActionBar() {
        mActionBar = getSupportActionBar();
        if(mActionBar != null){
            mActionBar.setTitle(R.string.actionbar_title_feedback);
            mActionBar.setDisplayHomeAsUpEnabled(true);
        }
    }

    private void initViews() {
        mBodyText = (EditText) findViewById(R.id.feedback_body_text);
        mSendLogSwitch = (Switch) findViewById(R.id.feedback_send_log_switch);
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        MenuInflater inflater = getMenuInflater();
        inflater.inflate(R.menu.feedback_menu, menu);
        return super.onCreateOptionsMenu(menu);
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        if(item.getItemId() == android.R.id.home){
            finish();
        } else if(item.getTitle().equals(getString(R.string.action_finished))) {
            onClickSendIntent();
        }
        return super.onOptionsItemSelected(item);
    }

    private void onClickSendIntent() {
        hideKeyboard();
        try {
            Intent shareIntent = new Intent(Intent.ACTION_SEND);
            shareIntent.setType("text/plain");
            shareIntent.putExtra(Intent.EXTRA_EMAIL, new String[]{getString(R.string.feedback_email)});
            shareIntent.putExtra(Intent.EXTRA_TEXT, mBodyText.getText().toString());
            shareIntent.putExtra(Intent.EXTRA_SUBJECT, getString(R.string.feedback_mail_subject, getApplicationName(), Utils.getAppVersion(getBaseContext()), Build.VERSION.RELEASE));
            shareIntent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);

            if (mSendLogSwitch.isChecked()) {
                //add log file as attachment if there are any logs
                Uri logFileUri = FileLog.getLogFileUri();
                if (logFileUri != null) {
                    shareIntent.putExtra(Intent.EXTRA_STREAM, logFileUri);
                    Utils.grantReadPermissionsOnIntent(this, shareIntent, logFileUri);
                }
            }

            tryStartActivity(shareIntent);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private void hideKeyboard() {
        InputMethodManager imm = (InputMethodManager)getSystemService(Context.INPUT_METHOD_SERVICE);
        imm.hideSoftInputFromWindow(mBodyText.getWindowToken(), 0);
    }

    private void tryStartActivity(Intent intent) {
        try {
            startActivityForResult(intent, REQUEST_CODE_FEEDBACK_SEND);
        } catch (ActivityNotFoundException e){
            Toast.makeText(this, R.string.no_activity_found, Toast.LENGTH_SHORT).show();
        }
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if(requestCode == REQUEST_CODE_FEEDBACK_SEND){
            finish();
        }
    }

    public String getApplicationName() {
        int stringId = getApplicationInfo().labelRes;
        return getString(stringId);
    }

    public void onClickShowLog(View v) {
        FileLog.showLogFileActivity(this);
    }
}
