/* 
 * driver.c
 * Copyright (C) 2002, AVM GmbH. All rights reserved.
 * 
 * This Software is  free software. You can redistribute and/or
 * modify such free software under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * The free software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this Software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA, or see
 * http://www.opensource.org/licenses/lgpl-license.html
 * 
 * Contact: AVM GmbH, Alt-Moabit 95, 10559 Berlin, Germany, email: info@avm.de
 */

#include <asm/atomic.h>
#include <asm/uaccess.h>
#include <linux/spinlock.h>
#include <linux/usb.h>
#include <linux/wait.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>
#include <linux/sched.h>
#include <linux/ctype.h>
#include <linux/string.h>
#include <linux/capi.h>
#include <linux/kernelcapi.h>
#include <stdarg.h>
#include "ca.h"
#include "libdefs.h"
#include "capilli.h"
#include "tables.h"
#include "queue.h"
#include "defs.h"
#include "lib.h"
#include "driver.h"

#if defined (LOG_MESSAGES)
# define mlog		log
#else
# define mlog(f, a...)	
#endif

#define	__entry

#ifndef HZ
# error HZ is not defined...
#endif
#define	TEN_MSECS		(HZ / 100)
#define	STACK_DELAY		(HZ / 2)
#define USB_EVENT_HARD_ERROR    0x800
#define	USB_CTRL_TIMEOUT	3 * HZ
#if defined (__fcusb2__)
#define	USB_CFG_BLK_SIZE	259
#define	USB_BULK_TIMEOUT	3 * HZ
#define	USB_MAX_FW_LENGTH	32768
#define	KMALLOC_LIMIT		128 * 1024
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,20)
#define	DEVREQUEST		devrequest
#define	DR_REQUEST_TYPE		requesttype
#define	DR_REQUEST		request
#define	DR_VALUE		value
#define	DR_INDEX		index
#define	DR_LENGTH		length
#else
#define	DEVREQUEST		struct usb_ctrlrequest
#define	DR_REQUEST_TYPE		bRequestType
#define	DR_REQUEST		bRequest
#define	DR_VALUE		wValue
#define	DR_INDEX		wIndex
#define	DR_LENGTH		wLength
#endif

static char *			invalid_msg	= "Invalid appl id #%d.\n";

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
dev_context_p			f_usb_capi_ctx		= NULL;
lib_callback_p			f_usb_capi_lib		= NULL;
struct capi_ctr *		f_usb_capi_ctrl		= NULL;
static int			started			= FALSE;
static int			nvers			= 0;
static atomic_t			scheduler_enabled	= ATOMIC_INIT (1);
static atomic_t			crit_count		= ATOMIC_INIT (0);
static volatile int		hard_error_issued;
static unsigned long		qt_flags;
static spinlock_t		qt_lock			= SPIN_LOCK_UNLOCKED;
static spinlock_t		stack_lock		= SPIN_LOCK_UNLOCKED;
static int			thread_pid		= -1;
static atomic_t			thread_run_flag;
static atomic_t			thread_capi_run_flag;
static atomic_t			tx_flag		= ATOMIC_INIT (0);
static atomic_t			rx_flag		= ATOMIC_INIT (0);

static DECLARE_WAIT_QUEUE_HEAD(wait);
static DECLARE_WAIT_QUEUE_HEAD(sched_wait);
static DECLARE_WAIT_QUEUE_HEAD(tx_wait);
static DECLARE_WAIT_QUEUE_HEAD(rx_wait);
static DECLARE_MUTEX_LOCKED(hotplug);

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#if defined (__fcusb2__)
int				card_config;

static DECLARE_MUTEX_LOCKED(config);
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static atomic_t			resetting_ctrl		= ATOMIC_INIT (0);

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static struct tq_struct		tq_close;
static atomic_t			closing_task_running	= ATOMIC_INIT (0);
static void		     (* close_func1) (void *)	= NULL;
static void *			close_data1		= NULL;
static void		     (* close_func2) (void *)	= NULL;
static void *			close_data2		= NULL;

static DECLARE_MUTEX_LOCKED(notify);

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void	scheduler_control (unsigned);
static void	enable_thread ();
static void 	disable_thread ();
static void	kick_scheduler (void);
static int	scheduler (void *);
void		usb_TxStart (void);
static void	usb_RxStart (void);
static void	tx_handler (dev_context_p pdc);
static void	rx_handler (dev_context_p pdc);
static void	tx_task (unsigned long);
static void	rx_task (unsigned long);

static DECLARE_TASKLET(tx_tasklet, tx_task, 0);
static DECLARE_TASKLET(rx_tasklet, rx_task, 0);

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void			scheduler_control (unsigned);
static void			wakeup_control (unsigned);
static void			version_callback (char *);
static unsigned			scheduler_suspend (unsigned long);
static unsigned			scheduler_resume (void);
static unsigned			controller_remove (void);
static unsigned			controller_add (void);

static functions_t		cafuncs = {

	7,
	scheduler_control,
	wakeup_control,
	version_callback,
	scheduler_suspend,
	scheduler_resume,
	controller_remove,
	controller_add
} ;

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
struct capi_driver_interface *	f_usb_di		= NULL;
struct capi_driver		f_usb_capi_interface	= {

	TARGET,
	"",
	f_usb_load_ware,
	f_usb_reset_ctrl,
	f_usb_remove_ctrl,
	f_usb_register_appl,
	f_usb_release_appl,
	f_usb_send_msg,
	f_usb_proc_info,
	f_usb_ctr_info,
	f_usb_drv_info,
	NULL,			/* add_card */
	NULL,
	NULL,
	0,
	NULL,
	""
} ;


/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static inline unsigned long atomic_xchg (
	volatile atomic_t *	v, 
	unsigned		value
) {
	return __xchg (value, &v->counter, sizeof (unsigned));
} /* atomic_xchg */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int nbchans (struct capi_ctr * ctrl) {
	dev_context_p	card;
	unsigned char *	prf;
	int		temp = 2;
    
	assert (ctrl);
	card = (dev_context_p) ctrl->driverdata;
	prf = (unsigned char *) card->string[6];
	if (prf != NULL) {
		temp = prf[2] + 256 * prf[3];
	}
	return temp;
} /* nbchans */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#if defined (__fxusb__)
static inline const char * name_of (int code) {

	switch (code) {

	default:
	case 1:		return "FRITZ!X USB";
	case 2:		return "FRITZ!X ISDN";
	case 3:		return "FRITZ!X USB";
	}
} /* name_of */
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static struct sk_buff * make_0xfe_request (unsigned appl) {
	unsigned char	 request[8];
	struct sk_buff * skb;

	if (NULL == (skb = alloc_skb (8, GFP_ATOMIC))) {
		lprintf (KERN_ERR, "Unable to allocate message buffer.\n");
	} else {    
		request[0] = 8;		    request[1] = 0;
		request[2] = appl & 0xFF;   request[3] = (appl >> 8) & 0xFF;
		request[4] = 0xFE;	    request[5] = 0x80;
		memcpy (skb_put (skb, 8), &request, 8);
	}
	return skb;
} /* make_0xfe_request */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void reset_completion (struct urb * purb) {
	DEVREQUEST * req;

	assert (NULL != purb);
	assert (0 == purb->status);
	log ("Completed device reset.\n");
	req = (DEVREQUEST *) purb->context;
	hfree (req);
	usb_free_urb (purb);
} /* reset_completion */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int reset (struct usb_device * dev, void (* cfunc) (struct urb *)) {
	DEVREQUEST *	req;
	struct urb *	purb;
	int		res;

	if (NULL == (req = (DEVREQUEST *) hmalloc (sizeof (DEVREQUEST)))) {
		error ("Not enough memory for device request.\n");
		return -ENOMEM;
	}
	if (NULL == (purb = usb_alloc_urb (0))) {
		error ("Not enough memory for device request URB.\n");
		hfree (req);
		return -ENOMEM;
	}
	req->DR_REQUEST_TYPE = 0;
	req->DR_REQUEST      = USB_REQ_SET_CONFIGURATION;
	req->DR_VALUE        = 0;	/* Little endian required here! */ 
	req->DR_INDEX        = 0;	/* Little endian required here! */
	req->DR_LENGTH       = 0;	/* Little endian required here! */
	usb_fill_control_urb (
		purb,
		dev,
		usb_sndctrlpipe (dev, 0),
		(unsigned char *) req,
		NULL,
		0,
		(cfunc == NULL) ? reset_completion : cfunc,
		req
	);
	if (0 != (res = usb_submit_urb (purb))) {
		hfree (req);
	}
	return res;
} /* reset */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#if defined (__fcusb2__)
static void set_config_completion (struct urb * purb) {
	DEVREQUEST * req;

	assert (NULL != purb);
	assert (0 == purb->status);
	log ("Completed SET_CONFIGURATION(0).\n");
	req = (DEVREQUEST *) purb->context;
	hfree (req);
	usb_free_urb (purb);
	up (&config);
} /* set_config_completion */
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#if defined (__fcusb2__)
static int select_config (struct usb_device * dev, int confix) {
	int	cfg, res;
	
	assert (0 == in_interrupt ());
	cfg = (confix == UNDEF_CONFIG) ? 0 : confix;
	if (cfg == card_config) {
		log ("select_config(%d) exited, already configured.\n", cfg);
		return 0;
	}
	log ("select_config(%d)\n", cfg);
	if (0 == cfg) {
		reset (dev, set_config_completion);
		down (&config);
		res = 0;	/* Well... nothing but an assumption! :-) */
	} else  {
		res = usb_set_configuration (dev, cfg);
	}
	if (0 == res) { 
		card_config = cfg;
	} else {
		log ("select_config(%d) returned %d.\n", cfg, res);
	}
	return res;
} /* select_config */
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void scan_version (dev_context_p card, const char * ver) {
	int	vlen, i;
	char *	vstr;

	vlen = (unsigned char) ver[0];
	card->version = vstr = (char *) hmalloc (vlen);
	if (NULL == card->version) {
		log ("Could not allocate version buffer.\n");
		return;
	}
	memcpy (card->version, ver + 1, vlen);
	memset (card->string, 0, sizeof (card->string));
	i = 0;
	while ((i < 8) && (vlen >= (vstr - card->version))) {
		card->string[i++] = vstr + 1;
		vstr += 1 + *vstr;
	} 
#ifdef NDEBUG
	lprintf (KERN_INFO, "Stack version %s\n", card->string[0]);
#endif
	log ("Library version:    %s\n", card->string[0]);
	log ("Card type:          %s\n", card->string[1]);
	log ("Capabilities:       %s\n", card->string[4]);
	log ("D-channel protocol: %s\n", card->string[5]);
} /* scan_version */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void copy_version (struct capi_ctr * ctrl) {
	char *		tmp;
	dev_context_p	card;

	assert (ctrl);
	card = (dev_context_p) ctrl->driverdata;
	if (NULL == (tmp = card->string[3])) {
		lprintf (KERN_ERR, "Do not have version information...\n");
		return;
	}
	strncpy (ctrl->serial, tmp, CAPI_SERIAL_LEN);
	memcpy (&ctrl->profile, card->string[6], sizeof (capi_profile));
	strncpy (ctrl->manu, "AVM GmbH", CAPI_MANUFACTURER_LEN);
	ctrl->version.majorversion = 2;
	ctrl->version.minorversion = 0;
	tmp = card->string[0];
	ctrl->version.majormanuversion = (((tmp[0] - '0') & 15) << 4)
					+ ((tmp[2] - '0') & 15);
	ctrl->version.minormanuversion = ((tmp[3] - '0') << 4)
					+ (tmp[5] - '0') * 10
					+ ((tmp[6] - '0') & 15);
} /* copy_version */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void kill_version (dev_context_p card) {
	int i;

	for (i = 0; i < 8; i++) {
		card->string[i] = NULL;
	}
	if (card->version != NULL) {
		hfree (card->version);
		card->version = NULL;
	}
	nvers = 0;
} /* kill_version */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void pprintf (char * page, int * len, const char * fmt, ...) {
	va_list args;

	va_start (args, fmt);
	*len += vsprintf (page + *len, fmt, args);
	va_end (args);
} /* pprintf */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void stop (dev_context_p card) {

	(*f_usb_capi_lib->cm_exit) ();
	if (-1 != thread_pid) {
		disable_thread ();
	}
	queue_exit (&card->queue);
	table_exit (&card->appls);
	kill_version (card);
	started = FALSE;
} /* stop */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int start (dev_context_p card) {

	card->count = 0;
	table_init (&card->appls);
	queue_init (&card->queue);
	(*f_usb_capi_lib->cm_register_ca_functions) (&cafuncs);
	(*f_usb_capi_lib->cm_start) ();
	(void) (*f_usb_capi_lib->cm_init) (0, 0);
	assert (nvers == 1);
	enter_critical ();
	if ((*f_usb_capi_lib->cm_activate) ()) {
		log ("Activate failed.\n");
		leave_critical ();
		stop (card);
		return FALSE;
	}
	leave_critical ();
	enable_thread ();
	return (started = TRUE); 
} /* start */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void appl_released_notify (int appl) {

	UNUSED_ARG (appl);
	if (atomic_read (&closing_task_running)) {
		log ("Note: appl %d released.\n", appl);
		up (&notify);
	}
} /* appl_released_notify */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void closing_task (void * ctx) {
	appl_t *	appp;
	dev_context_p	card = ctx;

	DECLARE_WAIT_QUEUE_HEAD(close_wait);

#define CALLBACK(ix)    if (close_func##ix != NULL) { \
				log ("Callback[%p]\n", close_func##ix); \
				(* (close_func##ix)) (close_data##ix); \
				close_func##ix = NULL; \
			}

	log ("Closing task started.\n");
	assert (ctx != NULL);
	atomic_set (&closing_task_running, 1);
	atomic_set (&resetting_ctrl, 1);
	if (0 != card->count) {
		assert (card->appls != NULL);
		if (card->appls != NULL) {
			appp = first_appl (card->appls);
			while (appp != NULL) {
				log ("Releasing appl #%d...\n", appp->id);
				f_usb_release_appl (
					f_usb_capi_ctrl,
					appp->id
				);
				kick_scheduler ();
				down (&notify);
				appp = next_appl (card->appls, appp);
			}
		}
	}
	if (atomic_read (&thread_capi_run_flag)) {
		init_waitqueue_head (&close_wait);
		kick_scheduler ();
		log ("Waiting...\n");
		interruptible_sleep_on_timeout (&close_wait, STACK_DELAY);
		log ("... completed!\n");
	}
	if (started) {
		stop (card);
	}
	atomic_set (&resetting_ctrl, 0);
	f_usb_reset_ctrl (f_usb_capi_ctrl);
	f_usb_remove_ctrl (f_usb_capi_ctrl);
	CALLBACK (1);
	CALLBACK (2);
	atomic_set (&closing_task_running, 0);
	log ("Closing task terminated.\n");

#undef CALLBACK

} /* closing_task */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void init_closing_task (void) {

	atomic_set (&closing_task_running, 0);
} /* init_closing_task */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void start_closing_task (void (* func) (void *), void * data) {
	unsigned long	flags;
	spinlock_t	close_lock	= SPIN_LOCK_UNLOCKED;

	spin_lock_irqsave (&close_lock, flags);
	assert ((close_func1 == NULL) || (close_func2 == NULL));
	if (close_func1 != NULL) {
		assert (close_func2 == NULL);
		close_func2 = func;
		close_data2 = data;
	} else {
		close_func1 = func;
		close_data1 = data;
	}
	spin_unlock_irqrestore (&close_lock, flags);
	if (atomic_read (&closing_task_running)) {
		log ("Closing task already running.\n");
		return;
	}
	tq_close.list.next = NULL;
	tq_close.sync = 0;
	tq_close.data = f_usb_capi_ctx;
	tq_close.routine = &closing_task;
	schedule_task (&tq_close);
	log ("Closing task scheduled.\n");
} /* start_closing_task */

/*---------------------------------------------------------------------------*\
\*-L-------------------------------------------------------------------------*/
void lock (void) {
	unsigned long	local_flags;
	
	info (!spin_is_locked (&qt_lock));
	spin_lock_irqsave (&qt_lock, local_flags);
	qt_flags = local_flags;
} /* lock */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void unlock (void) {
	unsigned long	local_flags = qt_flags;

	spin_unlock_irqrestore (&qt_lock, local_flags);
} /* unlock */

/*---------------------------------------------------------------------------*\
\*-C-------------------------------------------------------------------------*/ 
static inline int in_critical (void) {

	return (0 < atomic_read (&crit_count));
} /* in_critical */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static inline void check (void) {
	unsigned long	flags;
	spinlock_t	chk_lock = SPIN_LOCK_UNLOCKED;
	
	if (atomic_xchg (&rx_flag, 0)) {
		spin_lock_irqsave (&chk_lock, flags);
		rx_handler (f_usb_capi_ctx);
		spin_unlock_irqrestore (&chk_lock, flags);
	}
	if (atomic_xchg (&tx_flag, 0)) {
		spin_lock_irqsave (&chk_lock, flags);
		tx_handler (f_usb_capi_ctx);
		spin_unlock_irqrestore (&chk_lock, flags);
	}
} /* check */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
void enter_critical (void) {

	if (!atomic_read (&crit_count) && !in_interrupt ()) {
		check ();
	}
	atomic_inc (&crit_count);
} /* enter_critical */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
void leave_critical (void) {
	
	assert (in_critical ());
	atomic_dec (&crit_count);
	if (!atomic_read (&crit_count) && !in_interrupt ()) {
		check ();
	}
} /* leave_critical */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int f_usb_kcapi_init (void) {

	if (NULL != f_usb_capi_ctrl) {
		error ("Cannot handle two controllers!\n");
 		return -EBUSY;
	}
	f_usb_capi_ctrl = (*f_usb_di->attach_ctr) 
				(&f_usb_capi_interface, SHORT_LOGO, NULL);
	if (NULL == f_usb_capi_ctrl) {
		error ("Could not attach the controller.\n");
		return -EBUSY;
	}
	f_usb_capi_ctrl->driverdata = (void *) f_usb_capi_ctx;
	f_usb_capi_ctx->ctr = f_usb_capi_ctrl;
	return 0;
} /* f_usb_kcapi_init */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int f_usb_stack_init (void) {
	int	res = 0;

	if (!start (f_usb_capi_ctx)) {
		error ("Initialization failed.\n");
		return -EIO;
	}
#if !defined (__fcusb2__)
	if (0 != (res = f_usb_kcapi_init ())) {
		stop (f_usb_capi_ctx);
 		return res;
	}
#endif
	assert (nvers == 1);
	copy_version (f_usb_capi_ctrl);
	(*f_usb_capi_ctrl->ready) (f_usb_capi_ctrl);
	return res;
} /* f_usb_stack_init */

/*---------------------------------------------------------------------------*\
\*-F-------------------------------------------------------------------------*/
#if defined (__fcusb2__)
static int emit (dev_context_p ctx, char * ptr, int len, int * olen) {
	int	res;

	res = usb_bulk_msg (
		ctx->dev,
		usb_sndbulkpipe (ctx->dev, ctx->epwrite->bEndpointAddress),
		ptr,
		len,
		olen,
		USB_BULK_TIMEOUT
		);
	if (0 != res) {
		error ("USB I/O error, code %d!\n", res);
	}
	return res;
} /* emit */
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int __entry f_usb_load_ware (struct capi_ctr * ctrl, capiloaddata * ware) {
#if defined (__fcusb2__)
	dev_context_p	ctx;
	int		len, todo, olen;
	char *		tmp;
	char *		ptr;
	int		res	= 13;
#if !defined (NDEBUG)
	unsigned	count	= 0;
#endif

	assert (ware);
	assert (ctrl);
	if (f_usb_capi_lib == NULL) {
		error ("No controller!\n");
		return -EBUSY;
	}
	ctx = (dev_context_p) ctrl->driverdata;
	info (card_config == LOADING_CONFIG);
	if (LOADING_CONFIG != card_config) {
		res = select_config (ctx->dev, LOADING_CONFIG);
		if (0 != res) {
			error ("Could not set load config.\n");
			return -EIO;
		}
	}
	assert (USB_MAX_FW_LENGTH < KMALLOC_LIMIT);
	if ((todo = ware->firmware.len) > USB_MAX_FW_LENGTH) {
		error ("Invalid firmware file!\n");
		return -EIO;
	}
	if (NULL == (ptr = tmp = (char *) hmalloc (todo))) {
		error ("Cannot build firmware buffer!\n");
		return -ENOMEM;
	}
	if (ware->firmware.user) {
		if (copy_from_user (tmp, ware->firmware.data, todo)) {
			error ("Error copying firmware data!\n");
			return -EIO;
		}
	} else {
		memcpy (tmp, ware->firmware.data, todo);
	}
	while (todo > 0) {
		len = (todo > USB_CFG_BLK_SIZE) ? USB_CFG_BLK_SIZE : todo;
		if (0 != (res = emit (ctx, ptr, len, &olen))) {
			log (
				"Error in %d-byte block #%d.\n",
				USB_CFG_BLK_SIZE,
				count
			);
			break;
		}
		assert (len == olen);
		ptr  += olen;
		todo -= olen;
#if !defined (NDEBUG)
		++count;
#endif
	}
	if (0 == res) {
		res = emit (ctx, NULL, 0, &olen);
	}	
	hfree (tmp);
	if (0 == res) {
		res = select_config (ctx->dev, RUNNING_CONFIG);
	}
	if (0 == res) {
		return f_usb_stack_init ();
	}
	error ("Firmware does not respond!\n");
	(*ctx->ctr->reseted) (ctx->ctr);
	return -EBUSY;
#else
	UNUSED_ARG (ctrl);
	UNUSED_ARG (ware);

	lprintf (KERN_ERR, "Cannot load firmware onto passive controller.\n");
	return -EIO;
#endif
} /* f_usb_load_ware */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int __entry f_usb_add_card (struct capi_driver * drv, capicardparams * args) {
	
	UNUSED_ARG (drv);
	UNUSED_ARG (args);
	init_waitqueue_head (&wait);
	atomic_set (&resetting_ctrl, 0);
#if defined (__fcusb2__)
	return f_usb_kcapi_init ();
#else
	return f_usb_stack_init ();
#endif
} /* f_usb_add_card */ 

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int remove_appls (dev_context_p card) {
	appl_t *	appp;
	int		n = 0;

	if (0 != card->count) {
		log ("Removing registered applications.\n");
		assert (card->appls != NULL);
		if (card->appls != NULL) {
			appp = first_appl (card->appls);
			while (appp != NULL) {
				++n;
				f_usb_release_appl (
					f_usb_capi_ctrl,
					appp->id
				);
				appp = next_appl (card->appls, appp);
			}
		}
		kick_scheduler ();
	}
	return n;
} /* remove_appls */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry f_usb_reset_ctrl (struct capi_ctr * ctrl) {

	DECLARE_WAIT_QUEUE_HEAD(appl_wait);

	UNUSED_ARG (ctrl);
	atomic_set (&resetting_ctrl, 1);
	if (remove_appls (f_usb_capi_ctx)) {
		init_waitqueue_head (&appl_wait);
		interruptible_sleep_on_timeout (&appl_wait, STACK_DELAY);
	}
	if (started) {
		stop (f_usb_capi_ctx);
	}
	if (f_usb_capi_ctrl != NULL) {
		assert (f_usb_capi_ctrl->reseted != NULL);
		(* f_usb_capi_ctrl->reseted) (f_usb_capi_ctrl);
	}
	atomic_set (&resetting_ctrl, 0);
} /* f_usb_reset_ctrl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry f_usb_remove_ctrl (struct capi_ctr * ctrl) {

	UNUSED_ARG (ctrl);
	assert (f_usb_capi_ctrl != NULL);
	assert (f_usb_capi_ctx != NULL);
	enter_critical ();
	usb_unlink_urb (f_usb_capi_ctx->rx_urb);
	usb_unlink_urb (f_usb_capi_ctx->tx_urb);
	leave_critical ();
	(*f_usb_di->detach_ctr) (f_usb_capi_ctrl);
	f_usb_capi_ctrl = NULL;
	f_usb_capi_ctx  = NULL;
} /* f_usb_remove_ctrl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry f_usb_register_appl (
	struct capi_ctr *	ctrl, 
	u16			appl, 
	capi_register_params *	args
) {
	dev_context_p		card;
	appl_t *		appp;
	void *			ptr;
	unsigned		nc;

	mlog ("REGISTER(appl:%u)\n", appl);
	assert (ctrl);
	assert (args);
	card = (dev_context_p) ctrl->driverdata;
	if ((int) args->level3cnt < 0) {
		nc = nbchans (ctrl) * -((int) args->level3cnt);
	} else {
		nc = args->level3cnt;
	}
	if (0 == nc) {
		nc = nbchans (ctrl);
	}
	appp = create_appl (
			card->appls, 
			appl, 
			nc, 
			args->datablkcnt, 
			args->datablklen
			);
	if (NULL == appp) {
		log ("Unable to create application record.\n");
		return;
	}
	ptr = hcalloc (card->length);
	if (NULL == ptr) {
        	error ("Not enough memory for application data.\n");
		remove_appl (card->appls, appp);
	} else {
		card->count++;
		appp->data = ptr;
		appp->ctrl = ctrl;
		(*card->reg_func) (ptr, appl);
		(*ctrl->appl_registered) (ctrl, appl);
	}
} /* f_usb_register_appl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry f_usb_release_appl (struct capi_ctr * ctrl, u16 appl) {
	dev_context_p		card;
	struct sk_buff *	skb;
	appl_t *		appp;

	mlog ("RELEASE(appl:%u)\n", appl);
	assert (ctrl);
	card = (dev_context_p) ctrl->driverdata;
	appp = search_appl (card->appls, appl);
	if (NULL == appp) {
		log ("Attempt to release unknown application (id #%u)\n", appl);
		return;
	}
	skb = make_0xfe_request (appl);
	handle_message (card->appls, appp, skb);
	appp->dying = TRUE;
	kick_scheduler ();
} /* f_usb_release_appl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry f_usb_send_msg (struct capi_ctr * ctrl, struct sk_buff * skb) {
	dev_context_p	card;
	unsigned char *	byte;
	unsigned	appl;
	appl_t *	appp;

	assert (ctrl);
	assert (skb);
	card = (dev_context_p) ctrl->driverdata;
	byte = skb->data;
	appl = byte[2] + 256 * byte[3];
	mlog (
		"MESSAGE(appl:%u,cmd:%02x,subcmd:%02x)\n", 
		appl,
		byte[4], 
		byte[5]
	);
	appp = search_appl (card->appls, appl);
	if ((NULL == appp) || appp->dying) {
		log ("Message for unknown application (id %u).\n", appl);
		return;
	}
	handle_message (card->appls, appp, skb);
	kick_scheduler ();
} /* f_usb_send_msg */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
char * __entry f_usb_proc_info (struct capi_ctr * ctrl) {
	dev_context_p	card;
	static char	text[80];

	assert (ctrl);
	card = (dev_context_p) ctrl->driverdata;
	SNPRINTF (
		text, 
		sizeof (text),
		"%s %s %d",
		card->version ? card->string[1] : "A1",
		card->version ? card->string[0] : "-",
		card->dev->devnum
	);
	return text;
} /* f_usb_proc_info */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int __entry f_usb_ctr_info (
	char *			page, 
	char **			start, 
	off_t			ofs,
	int			count, 
	int *			eof, 
	struct capi_ctr *	ctrl
) {
	dev_context_p		card;
	char *			temp;
	unsigned char		flag;
	int			len = 0;

	assert (ctrl);
	card = (dev_context_p) ctrl->driverdata;
#if defined (__fxusb__)
	assert (card->dc <= MAX_DEVICE_CODE);
	pprintf (page, &len, "%-16s %s (%s)\n", "name", SHORT_LOGO, name_of (card->dc));
#else
	pprintf (page, &len, "%-16s %s\n", "name", SHORT_LOGO);
#endif
	pprintf (page, &len, "%-16s %d\n", "dev", card->dev->devnum);
	temp = card->version ? card->string[1] : "A1";
	pprintf (page, &len, "%-16s %s\n", "type", temp);
	temp = card->version ? card->string[0] : "-";
	pprintf (page, &len, "%-16s %s\n", "ver_driver", temp);
	pprintf (page, &len, "%-16s %s\n", "ver_cardtype", SHORT_LOGO);

	flag = ((unsigned char *) (ctrl->profile.manu))[3];
	if (flag) {
		pprintf (page, &len, "%-16s%s%s%s%s%s%s%s\n", "protocol",
			(flag & 0x01) ? " DSS1" : "",
			(flag & 0x02) ? " CT1" : "",
			(flag & 0x04) ? " VN3" : "",
			(flag & 0x08) ? " NI1" : "",
			(flag & 0x10) ? " AUSTEL" : "",
			(flag & 0x20) ? " ESS" : "",
			(flag & 0x40) ? " 1TR6" : ""
		);
	}
	flag = ((unsigned char *) (ctrl->profile.manu))[5];
	if (flag) {
		pprintf (page, &len, "%-16s%s%s%s%s\n", "linetype",
			(flag & 0x01) ? " point to point" : "",
			(flag & 0x02) ? " point to multipoint" : "",
			(flag & 0x08) ? " leased line without D-channel" : "",
			(flag & 0x04) ? " leased line with D-channel" : ""
		);
	}
	if (len < ofs) {
		return 0;
	}
	*eof = 1;
	*start = page - ofs;
	return ((count < len - ofs) ? count : len - ofs);
} /* f_usb_ctr_info */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int __entry f_usb_drv_info (
	char *			page, 
	char **			start, 
	off_t			ofs, 
	int			count, 
	int *			eof, 
	struct capi_driver *	drv
) {
	int			len = 0;

	UNUSED_ARG (drv);
	pprintf (page, &len, "%-16s %s\n", "name", TARGET);
	pprintf (page, &len, "%-16s %s\n", "revision", 
					f_usb_capi_interface.revision);
	if (len < ofs) {
		return 0;
	}
	*eof = 1;
	*start = page - ofs;
	return ((count < len - ofs) ? count : len - ofs);
} /* f_usb_drv_info */
 
/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void * data_by_id (unsigned appl_id) {
	appl_t * appp;

	appp = search_appl (f_usb_capi_ctx->appls, appl_id);
	return (appp != NULL) ? appp->data : NULL;
} /* data_by_id */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
struct capi_ctr * card_by_id (unsigned appl_id) {
	appl_t * appp;

	appp = search_appl (f_usb_capi_ctx->appls, appl_id);
	return (appp != NULL) ? appp->ctrl : NULL;
} /* card_by_id */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void * first_data (int * res) {
	appl_t * appp;

	assert (res);
	appp = first_appl (f_usb_capi_ctx->appls);
	*res = (appp != NULL) ? 0 : -1;
	return (appp != NULL) ? appp->data  : NULL;
} /* first_data */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void * next_data (int * res) {
	appl_t * appp;

	assert (res);
	if (NULL != (appp = get_appl (f_usb_capi_ctx->appls, *res))) {
		appp = next_appl (f_usb_capi_ctx->appls, appp);
	}
	*res = (appp != NULL) ? 1 + *res : -1;
	return (appp != NULL) ? appp->data  : NULL;
} /* next_data */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int appl_profile (unsigned appl_id, unsigned * bs, unsigned * bc) {
	appl_t * appp;

	appp = search_appl (f_usb_capi_ctx->appls, appl_id);
	if (NULL == appp) {
		return 0;
	}
	if (bs) *bs = appp->blk_size;
	if (bc) *bc = appp->blk_count;
	return 1;
} /* appl_profile */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int msg2stack (unsigned char * msg) {
	unsigned		mlen;
	unsigned		appl;
	__u32			ncci;
	unsigned		hand;
	unsigned char *		mptr;
	unsigned char *		dptr;
	struct sk_buff *	skb;
	unsigned		temp;
	int			res = 0;

	assert (msg != NULL);
	if (!queue_is_empty (f_usb_capi_ctx->queue)) {
		skb = queue_peek (f_usb_capi_ctx->queue);
		res = 1;
		assert (skb);
		mptr = (unsigned char *) skb->data;
		mlen = mptr[0] + 256 * mptr[1]; 
		appl = mptr[2] + 256 * mptr[3];
		mlog (
			"PUT_MESSAGE(appl:%u,cmd:%02X,subcmd:%02X)\n", 
			appl, 
			mptr[4], 
			mptr[5]
		);
		if ((0x86 == mptr[4]) && (0x80 == mptr[5])) { /* DATA_B3_REQ */
			ncci = mptr[8] + 256 * (mptr[9] + 256 * 
						(mptr[10] + 256 * mptr[11]));
			hand = mptr[18] + 256 * mptr[19];
			temp = (unsigned) (mptr + mlen);
			dptr = mptr + 12;
			*dptr++ = temp & 0xFF;	temp >>= 8;
			*dptr++ = temp & 0xFF;	temp >>= 8;
			*dptr++ = temp & 0xFF;	temp >>= 8;
			*dptr++ = temp & 0xFF;
			memcpy (msg, mptr, mlen);
			queue_park (f_usb_capi_ctx->queue, appl, ncci, hand);
		} else {
			memcpy (msg, mptr, mlen);
			queue_drop (f_usb_capi_ctx->queue);
		}
	} 
	return res;
} /* msg2stack */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void msg2capi (unsigned char * msg) {
	unsigned		mlen; 
	unsigned		appl;
	__u32			ncci;
	unsigned		hand;
	unsigned		dlen;
	unsigned char *		dptr;
	unsigned		temp;
	struct sk_buff *	skb;
	struct capi_ctr *	card;

	assert (msg != NULL);
	mlen = msg[0] + 256 * msg[1];
	appl = msg[2] + 256 * msg[3];
	if ((0x86 == msg[4]) && (0x81 == msg[5])) { /* DATA_B3_CONF */
		hand = msg[12] + 256 * msg[13];
		ncci = msg[8] + 256 * (msg[9] + 256 * 
						(msg[10] + 256 * msg[11]));
		queue_conf (f_usb_capi_ctx->queue, appl, ncci, hand);
	}
	if ((0x86 == msg[4]) && (0x82 == msg[5])) { /* DATA_B3_IND */
		dlen = msg[16] + 256 * msg[17];
		skb = alloc_skb (
			mlen + dlen + ((mlen < 30) ? (30 - mlen) : 0), 
			GFP_ATOMIC
			);
		if (NULL == skb) {
			error ("Unable to build CAPI message skb." 
							" Message lost.\n");
			return;
		}
		/* Messages are expected to come with 32 bit data pointers. 
		 * The kernel CAPI works with extended (64 bit ready) message 
		 * formats so that the incoming message needs to be fixed, 
		 * i.e. the length gets adjusted and the required 64 bit data 
		 * pointer is added.
		 */
		temp = msg[12] + 256 * (msg[13] + 256 * 
						(msg[14] + 256 * msg[15]));
		dptr = (unsigned char *) temp;
		if (mlen < 30) {
			msg[0] = 30;
			ncci   = 0;			
			memcpy (skb_put (skb, mlen), msg, mlen);
			memcpy (skb_put (skb, 4), &ncci, 4);
			memcpy (skb_put (skb, 4), &ncci, 4);
		} else {
			memcpy (skb_put (skb, mlen), msg, mlen);
		}
		memcpy (skb_put (skb, dlen), dptr, dlen); 
	} else {
		if (NULL == (skb = alloc_skb (mlen, GFP_ATOMIC))) {
			error ("Unable to build CAPI message skb." 
							" Message lost.\n");
			return;
		}
		memcpy (skb_put (skb, mlen), msg, mlen);
	}
	mlog (
		"GET_MESSAGE(appl:%u,cmd:%02X,subcmd:%02X)\n", 
		appl, 
		msg[4], 
		msg[5]
	);
	card = card_by_id (appl);
	info (card);
	if (card != NULL) {
		(*card->handle_capimsg) (card, appl, skb);
	}
} /* msg2capi */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void new_ncci (
	unsigned	appl_id, 
	__u32		ncci,
	unsigned	winsize, 
	unsigned	blksize
) {
	appl_t *	appp;
	ncci_t *	nccip;

	mlog ("NEW NCCI(appl:%u,ncci:0x%08X)\n", appl_id, ncci);
	appp = search_appl (f_usb_capi_ctx->appls, appl_id);
	if (NULL == appp) {
		lprintf (KERN_ERR, invalid_msg, appl_id);
		return;
	}
	nccip = create_ncci (
			f_usb_capi_ctx->appls, 
			appp, 
			(NCCI_t) ncci, 
			winsize, 
			blksize
			);
	if (NULL == nccip) {
		log ("Cannot handle new NCCI...\n");
		return;
	}
	assert (appp->ctrl);
	assert (appp->ctrl->new_ncci);
	(*appp->ctrl->new_ncci) (appp->ctrl, appl_id, ncci, winsize);
} /* new_ncci */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void free_ncci (unsigned appl_id, __u32 ncci) {
	appl_t * appp;
	ncci_t * nccip;

	appp = search_appl (f_usb_capi_ctx->appls, appl_id);
	if (NULL == appp) {
		lprintf (KERN_ERR, invalid_msg, appl_id);
		return;
	}
	assert (appp->ctrl);
	if (0xFFFFFFFF == ncci) {			/* 2nd phase RELEASE */
		mlog ("FREE APPL(appl:%u)\n", appl_id);
		assert (appp->dying);
		f_usb_capi_ctx->count--;
		(*f_usb_capi_ctx->rel_func) (appp->data);
		remove_appl (f_usb_capi_ctx->appls, appp);
		appl_released_notify (appl_id);
		if (atomic_read (&resetting_ctrl)) {
			return;
		}
		assert (appp->ctrl->appl_released);
		(*appp->ctrl->appl_released) (appp->ctrl, appl_id);
	} else if (NULL != (nccip = locate_ncci (appp, ncci))) {
		mlog ("FREE NCCI(appl:%u,ncci:0x%08X)\n", appl_id, ncci);
		assert (appp->ctrl->free_ncci);
		(*appp->ctrl->free_ncci) (appp->ctrl, appl_id, ncci);
		remove_ncci (f_usb_capi_ctx->appls, appp, nccip);
	} else {
		lprintf (KERN_ERR, "Attempt to free unknown NCCI.\n");
	}
} /* free_ncci */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
unsigned char * data_block (unsigned appl_id, __u32 ncci, unsigned handle) {
	appl_t *	appp;
	char *		buf;
 
	appp = search_appl (f_usb_capi_ctx->appls, appl_id);
	if (NULL == appp) {
		lprintf (KERN_ERR, invalid_msg, appl_id);
		return NULL;
	}
	buf = ncci_data_buffer (f_usb_capi_ctx->appls, appp, ncci, handle);
	return buf;
} /* data_block */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
void init (
	unsigned len, 
	void  (* reg) (void *, unsigned),
	void  (* rel) (void *),
	void  (* dwn) (void)
) {
	assert (reg);
	assert (rel);
	assert (dwn);

	f_usb_capi_ctx->length   = len;
	f_usb_capi_ctx->reg_func = reg;
	f_usb_capi_ctx->rel_func = rel;
	f_usb_capi_ctx->dwn_func = dwn;
} /* init */

/*---------------------------------------------------------------------------*\
\*-U-------------------------------------------------------------------------*/
static void issue_hard_error (dev_context_p pdc, const char * msg, int arg) {
	
	lprintf (KERN_INFO, msg, arg);
	if (!hard_error_issued) {
		hard_error_issued = TRUE;
		(*pdc->ev_handler) (USB_EVENT_HARD_ERROR);
	} else {
		lprintf (KERN_INFO, "Extra events ignored.\n");
	}
} /* issue_hard_error */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void usb_write_completion (struct urb * purb) {

	UNUSED_ARG (purb);
	tasklet_schedule (&tx_tasklet);
} /* usb_write_completion */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void usb_read_completion (struct urb * purb) {

	UNUSED_ARG (purb);
	tasklet_schedule (&rx_tasklet);
} /* usb_read_completion */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void usb_write (dev_context_p pdc, unsigned long len) {
	int	res;

	assert (pdc);
	if (!atomic_read (&pdc->is_open)) {
		log ("Attempt to write on closed device.");
		return;
	}
	assert (atomic_read (&pdc->tx_pending));
	FILL_BULK_URB (
		pdc->tx_urb,
		pdc->dev,
		usb_sndbulkpipe (pdc->dev, pdc->epwrite->bEndpointAddress),
		pdc->tx_buffer,
		len,
		usb_write_completion,
		pdc
		);
	if (0 != (res = usb_submit_urb (pdc->tx_urb))) {
		atomic_set (&pdc->tx_pending, 0);
		issue_hard_error (
			pdc, 
			"Tx URB submission failed, code %d.\n", 
			res
		);
	}
} /* usb_write */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void usb_read (dev_context_p pdc) {
	int	res;

	assert (pdc);
	if (!atomic_read (&pdc->is_open)) {
		log ("Attempt to read from closed device.");
		return;
	}
	FILL_BULK_URB (
		pdc->rx_urb,
		pdc->dev,
		usb_rcvbulkpipe (pdc->dev, pdc->epread->bEndpointAddress),
		pdc->rx_buffer,
		MAX_TRANSFER_SIZE,
		usb_read_completion,
		pdc
		);
	if (0 != (res = usb_submit_urb (pdc->rx_urb))) {
		atomic_set (&pdc->rx_pending, 0);
		issue_hard_error (
			pdc, 
			"Rx URB submission failed, code %d.\n", 
			res
		);
	}
} /* usb_read */

/*---------------------------------------------------------------------------*\
\*-T-------------------------------------------------------------------------*/
static void tx_handler (dev_context_p pdc) {
	struct urb *	purb;

	assert (pdc != NULL);
	purb = pdc->tx_urb;
	assert (atomic_read (&pdc->tx_pending));
	atomic_set (&tx_flag, 0);
	if (!atomic_read (&pdc->is_open)) {
		atomic_set (&pdc->tx_pending, 0);
		log ("TX: Device closed!\n");
		return;
	}
	if (0 != purb->status) {
		atomic_set (&pdc->tx_pending, 0);
		issue_hard_error (
			pdc, 
			"Tx URB status: %d\n", 
			purb->status
		);
		return;
	}
	if ((purb->actual_length > 0) 
	&& ((purb->actual_length % 64) == 0)) {
		log (
			"Inserting zero-length block! (Wrote %d bytes)\n",
			purb->actual_length
		);
		usb_write (pdc, 0);
	} else {
		atomic_set (&pdc->tx_pending, 0);
		usb_TxStart ();
		kick_scheduler ();
	}
} /* tx_handler */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void tx_task (unsigned long arg) {
	dev_context_p	pdc = f_usb_capi_ctx;
	
	UNUSED_ARG (arg);
	if (in_critical ()) {
		atomic_set (&tx_flag, 1);
	} else if (spin_trylock (&stack_lock)) {
		tx_handler (pdc);
		spin_unlock (&stack_lock);
	} else {
		atomic_set (&tx_flag, 1);
	}
} /* tx_task */
		
/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void rx_handler (dev_context_p pdc) {
	struct urb *	purb;
	
	assert (pdc != NULL);
	purb = pdc->rx_urb;
	atomic_set (&pdc->rx_pending, 0);
	atomic_set (&rx_flag, 0);
	if (!atomic_read (&pdc->is_open)) {
		log ("RX: Device closed!\n");
		return;
	}
	if (0 != purb->status) {
		issue_hard_error (
			pdc, 
			"Rx URB status: %d\n", 
			purb->status
		);
		return;
	}
	if (0 != purb->actual_length) {
		(*pdc->rx_avail) (pdc->rx_buffer, purb->actual_length);
		kick_scheduler ();
	}
	usb_RxStart ();
} /* rx_handler */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void rx_task (unsigned long arg) {
	dev_context_p	pdc = f_usb_capi_ctx;
	
	UNUSED_ARG (arg);
	if (in_critical ()) {
		atomic_set (&rx_flag, 1);
	} else if (spin_trylock (&stack_lock)) {
		rx_handler (pdc);
		spin_unlock (&stack_lock);
	} else {
		atomic_set (&rx_flag, 1);
	}
} /* rx_task */

/*---------------------------------------------------------------------------*\
\*-S-------------------------------------------------------------------------*/
static void scheduler_control (unsigned f) {
	int	flag = (int) f;
	int	changed;

	changed = (atomic_read (&scheduler_enabled) != flag);
	if (changed) {
		if (flag) {
			kick_scheduler ();
		} else {
			atomic_set (&scheduler_enabled, 0);
			wake_up_interruptible (&wait);
		}
	}
} /* scheduler_control */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int scheduler (void * arg) {
	
	UNUSED_ARG (arg);
	SNPRINTF (current->comm, 16, "%s_sched", TARGET);
	log ("Starting scheduler thread '%s'...\n", current->comm);
	while (atomic_read (&thread_run_flag)) {
		if (!atomic_read (&thread_capi_run_flag)) {
			interruptible_sleep_on (&sched_wait);
		} else {
			if (atomic_read (&scheduler_enabled)) {
				interruptible_sleep_on_timeout (&wait, TEN_MSECS);
			} else {
				interruptible_sleep_on (&wait);
			}
		}
		if (!atomic_read (&thread_run_flag)) {
			info (atomic_read (&thread_run_flag));
			break;
		}
		if (!atomic_read (&thread_capi_run_flag)) {
			info (atomic_read (&thread_capi_run_flag));
			continue;
		}
		if (!atomic_read (&scheduler_enabled)) {
			info (atomic_read (&scheduler_enabled));
			continue;
		}
		if (f_usb_capi_lib == NULL) {
			log ("Library lost! Hot unplug?\n");
			break;
		}
		if (spin_trylock (&stack_lock)) {
			os_timer_poll ();
			if ((*f_usb_capi_lib->cm_schedule) ()) {
				scheduler_control (TRUE);
			}
			spin_unlock (&stack_lock);
		} else {
			kick_scheduler ();
		}
	}
	log ("Scheduler thread stopped.\n");
	up (&hotplug);
	return 0;
} /* scheduler */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void make_thread (void) {

	assert (thread_pid == -1);
	atomic_set (&thread_run_flag, 1);
	atomic_set (&thread_capi_run_flag, 0);
	init_waitqueue_head (&sched_wait);
	thread_pid = kernel_thread (scheduler, NULL, 0);
	log ("Scheduler thread[%d] started.\n", thread_pid);
} /* make_thread */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void kill_thread (void) {

	if (thread_pid != -1) {
		atomic_set (&thread_run_flag, 0);
		if (NULL == find_task_by_pid (thread_pid)) {
			log ("Thread[%d] has died before!\n", thread_pid);
		} else {
			if (!atomic_read (&thread_capi_run_flag)) {
				wake_up_interruptible (&sched_wait);
			} else {
				wake_up_interruptible (&wait);
			}
			log ("Scheduler thread signalled, waiting...\n");
			down (&hotplug);
			log ("Scheduler thread[%d] terminated.\n", thread_pid);
		}
		thread_pid = -1;
	} else {
		log ("No scheduler thread.\n");
	}
} /* kill_thread */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void enable_thread (void) {

	assert (atomic_read (&thread_run_flag));
	info (!atomic_read (&thread_capi_run_flag));
	if (!atomic_read (&thread_capi_run_flag)) {
		atomic_set (&thread_capi_run_flag, 1);
		wake_up_interruptible (&sched_wait);
		log ("Thread enabled.\n");
	}
} /* enable_thread */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void disable_thread (void) {

	assert (atomic_read (&thread_run_flag));
	info (atomic_read (&thread_capi_run_flag));
	if (atomic_read (&thread_capi_run_flag)) {
		atomic_set (&thread_capi_run_flag, 0);
		wake_up_interruptible (&wait);
		log ("Thread disabled.\n");
	}
} /* disable_thread */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void kick_scheduler (void) {

	atomic_set (&scheduler_enabled, 1);
	wake_up_interruptible (&wait);
} /* kick_scheduler */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void wakeup_control (unsigned enable_flag) { 
	
	enable_flag = 0; 
} /* wakeup_control */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void version_callback (char * vp) { 
	
	assert (nvers == 0);
	log ("Version string #%d:\n", nvers);
	scan_version (f_usb_capi_ctx, vp);
	++nvers;
} /* version_callback */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static unsigned	scheduler_suspend (unsigned long timeout) { 
	
	timeout = 0; 
	return 0; 
} /* scheduler_suspend */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static unsigned	scheduler_resume (void) { 
	
	return 0; 
} /* scheduler_resume */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static unsigned	controller_remove (void) { 

	assert (0);
	return 0; 
} /* controller_remove */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static unsigned	controller_add (void) { 
	
	assert (0);
	return 0; 
} /* controller_add */
	
/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void stop_base (void) {
	dev_context_p	pdc = f_usb_capi_ctx;

	if (NULL == pdc) {
		return;
	}
	atomic_set (&pdc->is_open, 0);
	atomic_set (&pdc->rx_pending, 0);
	atomic_set (&pdc->tx_pending, 0);
	atomic_set (&rx_flag, 0);
	atomic_set (&tx_flag, 0);
	log ("USB layer stopped.\n");
} /* stop_base */
		
/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void usb_Open (
	unsigned			max_buffer_size,
	get_next_tx_buffer_func_p	get_next_tx_buffer,
	event_handler_func_p		event_handler,
	new_rx_buffer_avail_func_p	new_rx_buffer_avail,
	complete_func_p			open_complete,
	unsigned			ref_data
) {
	dev_context_p			pdc = f_usb_capi_ctx;

	atomic_set (&pdc->is_open, 1);
	atomic_set (&pdc->tx_pending, 0);
	atomic_set (&pdc->rx_pending, 0);
	pdc->max_buf_size = max_buffer_size;
	pdc->tx_get_next  = get_next_tx_buffer;
	pdc->rx_avail     = new_rx_buffer_avail;
	pdc->ev_handler   = event_handler;
	hard_error_issued = FALSE;
	assert (open_complete);
	(*open_complete) (0, ref_data);
	usb_RxStart ();
} /* usb_Open */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void usb_Close (complete_func_p close_complete, unsigned ref_data) {

	stop_base ();
	assert (close_complete);
	(*close_complete) (0, ref_data);
	if (hard_error_issued) {
		start_closing_task (NULL, NULL);
	}
} /* usb_Close */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void usb_RxStart (void) {
	dev_context_p	pdc;

	pdc = f_usb_capi_ctx;
	assert (pdc != NULL);
	assert (0 == atomic_read (&pdc->rx_pending));
	atomic_set (&pdc->rx_pending, 1);
	usb_read (pdc);
} /* usb_RxStart */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void usb_TxStart (void) {
	dev_context_p	pdc = f_usb_capi_ctx;
	unsigned long	len;

	assert (pdc != NULL);
	if (atomic_xchg (&pdc->tx_pending, 1)) {
		return;
	}
	assert (pdc->tx_get_next);
	len = (*pdc->tx_get_next) (pdc->tx_buffer);
	assert (pdc->max_buf_size);
	assert (len <= pdc->max_buf_size);
	if (0 == len) {
		atomic_set (&pdc->tx_pending, 0);
		return;
	}
	info (atomic_read (&pdc->is_open));
	if (atomic_read (&pdc->is_open)) {
		assert (atomic_read (&pdc->tx_pending));
		usb_write (pdc, len);
	}
} /* usb_TxStart */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void usb_GetDeviceInfo (device_info_p device_info) {

	assert (device_info);
	device_info->product_id = f_usb_capi_ctx->dev->descriptor.idProduct;
	device_info->release_number = f_usb_capi_ctx->dev->descriptor.bcdDevice;
	device_info->self_powered = 0;
} /* usb_GetDeviceInfo */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/

