/* 
 * main.c
 * Copyright (C) 2002, AVM GmbH. All rights reserved.
 * 
 * This Software is  free software. You can redistribute and/or
 * modify such free software under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * The free software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this Software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA, or see
 * http://www.opensource.org/licenses/lgpl-license.html
 * 
 * Contact: AVM GmbH, Alt-Moabit 95, 10559 Berlin, Germany, email: info@avm.de
 */

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/ctype.h>
#include <linux/skbuff.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/capi.h>
#include <linux/usb.h>
#include "capilli.h"
#include "lib.h"
#include "driver.h"

#define	INIT_FAILED		-1
#define	INIT_SUCCESS		0

#define	VENDOR_ID_AVM		0x057C
#if defined (__fcusb__)
#define	PRODUCT_ID		0x0C00
#elif defined (__fcusb2__)
#define	PRODUCT_ID		0x1000
#define	PRODUCT_ID2		0x1900
#elif defined (__fxusb__)
#define	PRODUCT_ID		0x2000
#else
#error You have to define a card identifier!
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
EXPORT_NO_SYMBOLS;

#if defined (MODULE_LICENSE)
MODULE_LICENSE ("Proprietary");
#endif
#if defined (MODULE_DESCRIPTION)
MODULE_DESCRIPTION ("CAPI4Linux: Driver for " PRODUCT_LOGO);
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
static struct usb_device_id f_usb_id_table[] __initdata = {
	{ USB_DEVICE(VENDOR_ID_AVM, PRODUCT_ID) },
#if defined (__fcusb2__)
	{ USB_DEVICE(VENDOR_ID_AVM, PRODUCT_ID2) },
#endif
	{ /* Terminating entry */ }
} ;

MODULE_DEVICE_TABLE (usb, f_usb_id_table);
#else
#define __exit
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void f_usb_disconnect (struct usb_device *, void *);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
static void * f_usb_probe (struct usb_device *, unsigned int);
#else
static void * f_usb_probe (struct usb_device *, unsigned int, const struct usb_device_id *);
#endif

static struct usb_driver f_usb_driver = {
	name:		TARGET,
	probe:		f_usb_probe,
	disconnect:	f_usb_disconnect,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
	id_table:	f_usb_id_table,
#endif
} ;

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static char * 		REVCONST = "$Revision: $";
char          		REVISION[32];

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#if !defined (NDEBUG)
static void base_address (void) {

	log ("BASE ADDRESS %p\n", base_address);
} /* base_address*/
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
static void * f_usb_probe (struct usb_device * dev, unsigned int ifnum) 
#else
static void * f_usb_probe (
		struct usb_device * 		dev, 
		unsigned int 			ifnum,
		const struct usb_device_id *	devid
)
#endif
{
	dev_context_p				pdc;
	struct usb_interface_descriptor *	iface;
#if !defined (__fcusb2__)
	struct usb_config_descriptor *		cfgd;
#else
	struct usb_config_descriptor *		cfgd1;
	struct usb_config_descriptor *		cfgd2;
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
	UNUSED_ARG (devid);
#endif

#if defined (__fcusb2__)
	if ( (VENDOR_ID_AVM != dev->descriptor.idVendor)  || 
	     (  
	       (PRODUCT_ID  != dev->descriptor.idProduct)  &&
	       (PRODUCT_ID2 != dev->descriptor.idProduct)
	   ) ) 
#else
	if ((VENDOR_ID_AVM != dev->descriptor.idVendor)
	||  (PRODUCT_ID != dev->descriptor.idProduct)) 
#endif
	{
	exit1:
		return NULL;
	}
	MOD_INC_USE_COUNT;
	log ("Probe for device #%d.\n", dev->devnum);
	if (NULL == (pdc = (dev_context_p) hmalloc (sizeof (dev_context_t)))) {
		error ("Could not allocate device context.\n");
	exit2:
		MOD_DEC_USE_COUNT;
		goto exit1;
	}
	f_usb_capi_ctx = pdc;
	memset (pdc, 0, sizeof (dev_context_t));
	pdc->dev = dev;
	atomic_set (&pdc->is_open, 0);
#if defined (__fcusb2__)
	cfgd1 = dev->config + 0;
	cfgd2 = dev->config + 1;
	if (usb_interface_claimed (cfgd1->interface + ifnum)
	||  usb_interface_claimed (cfgd2->interface + ifnum)) {
		error ("Interface(s) already claimed.\n");
	exit3:
		hfree (pdc);
		goto exit2;
	}
	pdc->if1 = cfgd1->interface + ifnum;
	pdc->if2 = cfgd2->interface + ifnum;
	usb_driver_claim_interface (&f_usb_driver, pdc->if1, pdc);
	usb_driver_claim_interface (&f_usb_driver, pdc->if2, pdc);
	iface = cfgd1->interface[ifnum].altsetting + 0;
#else
	cfgd = dev->config + 0;
	if (usb_interface_claimed (cfgd->interface + ifnum)) {
		error ("Interface already claimed.\n");
	exit3:
		hfree (pdc);
		goto exit2;
	} 
	iface = cfgd->interface[ifnum].altsetting + 0;
#endif
	pdc->epwrite = iface->endpoint + 0;
        pdc->epread  = iface->endpoint + 1;
#if !defined (__fcusb2__)
	if (usb_set_configuration (dev, cfgd->bConfigurationValue)) {
		error ("Could not set configuration.\n");
	exit4:
		goto exit3;
	}
	if (usb_set_interface (dev, 0, 0)) {
		error ("Could not set interface.\n");
	exit5:
		goto exit4;
	}
#else
	card_config = cfgd1->bConfigurationValue;
	log ("Device is in config #%d.\n", card_config);
#endif
	if (NULL == (pdc->tx_buffer = (char *) hmalloc (MAX_TRANSFER_SIZE))) {
		error ("Could not allocate tx buffer.\n");
	exit6:
#if !defined (__fcusb2__)
		goto exit5;
#else
		usb_driver_release_interface (&f_usb_driver, pdc->if1);
		usb_driver_release_interface (&f_usb_driver, pdc->if2);
		goto exit3;
#endif
	}
	if (NULL == (pdc->rx_buffer = (char *) hmalloc (MAX_TRANSFER_SIZE))) {
		error ("Could not allocate rx buffer.\n");
	exit7:
		hfree (pdc->tx_buffer);
		goto exit6;
	}
	if (NULL == (pdc->tx_urb = usb_alloc_urb (0))) {
		error ("Could not allocate tx urb.\n");
	exit8:
		hfree (pdc->rx_buffer);
		goto exit7;
	}
	if (NULL == (pdc->rx_urb = usb_alloc_urb (0))) {
		error ("Could not allcate rx urb.\n");
	exit9:
		usb_free_urb (pdc->tx_urb);
		goto exit8;
	}
	if (NULL == (f_usb_capi_lib = link_library ())) {
		error ("Linking to library failed.\n");
	exitA:
		usb_free_urb (pdc->rx_urb);
		goto exit9;
	}
	make_thread ();
	if (0 != f_usb_add_card (&f_usb_capi_interface, NULL)) {
		free_library ();
		goto exitA;
	}
	return pdc;
} /* f_usb_probe */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void f_usb_disconnect (struct usb_device * dev, void * ptr) {
	dev_context_p	pdc = (dev_context_p) ptr;

	UNUSED_ARG (dev);
	assert (ptr);
	log ("Disconnect for device #%d.\n", dev->devnum);
	if (NULL != f_usb_capi_ctrl) {	
		/* In case of sudden removal... */
		unplug_notify ();
		f_usb_reset_ctrl (f_usb_capi_ctrl);
		f_usb_remove_ctrl (f_usb_capi_ctrl);
	}
	free_library ();
	if (NULL != pdc->rx_buffer) hfree (pdc->rx_buffer);
	if (NULL != pdc->tx_buffer) hfree (pdc->tx_buffer);
	if (NULL != pdc->rx_urb) usb_free_urb (pdc->rx_urb);
	if (NULL != pdc->tx_urb) usb_free_urb (pdc->tx_urb);
#if defined (__fcusb2__)
	usb_driver_release_interface (&f_usb_driver, pdc->if1);
	usb_driver_release_interface (&f_usb_driver, pdc->if2);
#endif
	f_usb_capi_ctx = NULL;
	f_usb_capi_lib = NULL;
	hfree (pdc);
	MOD_DEC_USE_COUNT;
} /* f_usb_disconnect */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void dec_use_count (void) { 
#if !defined (NDEBUG)
	int	uc = GET_USE_COUNT(THIS_MODULE);
#endif

	MOD_DEC_USE_COUNT; 
	log ("Use count: %d -> %d\n", uc, uc - 1);
} /* dec_use_count */

void inc_use_count (void) {
#if !defined (NDEBUG)
	int	uc = GET_USE_COUNT(THIS_MODULE);
#endif

	MOD_INC_USE_COUNT;
	log ("Use count: %d -> %d\n", uc, uc + 1);
} /* inc_use_count */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int __init f_usb_init (void) {
	int	urr, res = INIT_FAILED;
	char *	tmp;

#if !defined (NDEBUG)
	base_address ();
#endif
	MOD_INC_USE_COUNT;
	if ((NULL != (tmp = strchr (REVCONST, ':'))) && isdigit (*(tmp + 2))) {
		strncpy (REVISION, tmp + 1, sizeof (REVISION));
		tmp = strchr (REVISION, '$');
		*tmp = 0;
	} else {
		strcpy (REVISION, REV_DEFAULT);
	}
        lprintf (KERN_INFO, "%s, revision %s\n", DRIVER_LOGO, REVISION);
	lprintf (KERN_INFO, "(%s built on %s at %s)\n", TARGET, __DATE__, __TIME__);

	/*-------------------------------------------------------------------*\
	 * 64 bit CAPI is not supported yet.
	\*-------------------------------------------------------------------*/
	if (sizeof (char *) > 4) {
		lprintf (KERN_ERR, "Cannot deal with 64 bit CAPI messages!\n");
		return -ENOSYS;
	}
	
	lprintf (KERN_INFO, "Loading...\n");
	strncpy (f_usb_capi_interface.revision, REVISION,
					sizeof (f_usb_capi_interface.revision));
	if (NULL == (f_usb_di = attach_capi_driver (&f_usb_capi_interface))) {
		error ("Could not attach the driver.\n");
		goto exit;
	}
	if (0 > (urr = usb_register (&f_usb_driver))) {
		detach_capi_driver (&f_usb_capi_interface);
		error ("Function usb_register returned %d.\n", urr);
	} else {
		res = INIT_SUCCESS;
	}
exit:
	MOD_DEC_USE_COUNT;
	lprintf (KERN_INFO, "%soaded.\n", res == INIT_SUCCESS ? "L" : "Not l");
	return res;
} /* f_usb_init */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void __exit f_usb_exit (void) {
#ifndef NDEBUG
	unsigned u;
#endif
	lprintf (KERN_INFO, "Removing...\n");
	kill_thread ();
	detach_capi_driver (&f_usb_capi_interface);
	usb_deregister (&f_usb_driver);
#ifndef NDEBUG
	if (0 != (u = hallocated ())) {
		log ("%u bytes are still in use...\n", u);
	}
#endif
	lprintf (KERN_INFO, "Removed.\n");
} /* f_usb_exit */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
module_init (f_usb_init);
module_exit (f_usb_exit);

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/

