/* 
 * driver.c
 * Copyright (C) 2002, AVM GmbH. All rights reserved.
 * 
 * This Software is  free software. You can redistribute and/or
 * modify such free software under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * The free software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this Software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA, or see
 * http://www.opensource.org/licenses/lgpl-license.html
 * 
 * Contact: AVM GmbH, Alt-Moabit 95, 10559 Berlin, Germany, email: info@avm.de
 */

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/atomic.h>
#include <asm/uaccess.h>
#include <asm/system.h>
#include <linux/version.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/tqueue.h>
#include <linux/skbuff.h>
#include <linux/kernel.h>
#include <linux/spinlock.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <linux/ctype.h>
#include <linux/string.h>
#include <linux/capi.h>
#include <linux/kernelcapi.h>
#include <stdarg.h>
#include "libstub.h"
#include "ca.h"
#include "capilli.h"
#include "tables.h"
#include "queue.h"
#include "lib.h"
#include "main.h"
#include "tools.h"
#include "defs.h"
#include "fw.h"
#include "devif.h"
#include "driver.h"

#if defined (LOG_MESSAGES)
# define mlog		log
#else
# define mlog(f,a...)
#endif

#define	__entry

#define	KILOBYTE		1024
#define	MEGABYTE		(1024*KILOBYTE)
#define	_2_MEGS_		(2*MEGABYTE)
#define	_4_MEGS_		(4*MEGABYTE)
#define	_8_MEGS_		(8*MEGABYTE)
#define	TEN_MSECS		(HZ/100)

#define	MAPPED_DATA_LEN		_4_MEGS_
#define	MAPPED_CODE_LEN		_8_MEGS_
#define	IO_REGION_LEN		16
#define	DEBUG_MEM_SIZE		MEGABYTE
#define	DEBUG_TIMEOUT		100
#define	RESET_DELAY		10

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#define	DBG_FORMAT_LEN		1024
#define	DBG_MEM_SIZE		(1024 * 1024)
#define	DBG_TIMEOUT		100
#define	DBG_MIN_SIZE		1024

typedef struct __db {

	dma_struct_p	pdma;
	atomic_t	stop;
	unsigned	len;
	unsigned	tout;
	char *		pbuf;
	int		pid;
} dbg_buf_t, * dbg_buf_p;

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
card_p				capi_card		= NULL;
lib_callback_t *		capi_lib		= NULL;
atomic_t			crit_count		= ATOMIC_INIT(0);
spinlock_t			stack_lock		= SPIN_LOCK_UNLOCKED;

struct capi_driver_interface *	capi_driver		= NULL;
static int			load_flag		= FALSE;
static int			init_flag		= FALSE;
static int			nvers			= 0;
static atomic_t			scheduler_enabled	= ATOMIC_INIT (1);
struct capi_ctr *		capi_controller[2]	= { NULL, NULL } ;
static bundle_t			ctrl_context[2];
static per_ctrl_t		ctrl_params[2];
static unsigned long		qt_flags;
static spinlock_t		qt_lock			= SPIN_LOCK_UNLOCKED;
static atomic_t			thread_flag		= ATOMIC_INIT (0);
static atomic_t			thread_capi_flag	= ATOMIC_INIT (0);
static int			thread_pid		= -1;
static char *			firmware_ptr		= NULL;
static unsigned			firmware_len		= 0;
static unsigned			pc_ack_offset		= 0;
static unsigned			timer_offset		= 0;
static dbg_buf_p		dbgbuf			= NULL;

static DECLARE_WAIT_QUEUE_HEAD(wait);
static DECLARE_WAIT_QUEUE_HEAD(capi_wait);
static DECLARE_WAIT_QUEUE_HEAD(dbg_wait);

static DECLARE_MUTEX_LOCKED(thread_sync);

static int			irq_callback (void *);
static void			enable_thread (void);
static void			disable_thread (void);
static int			make_thread (void);
static void			kill_thread (void);
static void			scheduler_control (unsigned);
static void			wakeup_control (unsigned);
static void			version_callback (char *);
static void			kill_version (struct capi_ctr *);
static unsigned			scheduler_suspend (unsigned long);
static unsigned			scheduler_resume (void);
static unsigned			controller_remove (void);
static unsigned			controller_add (void);

struct capi_driver		capi_interface		= {

	TARGET,
	REV_DEFAULT,			
	load_ware,
	reset_ctrl,
	remove_ctrl,
	register_appl,
	release_appl,
	send_msg,
	proc_info,
	ctr_info,
	drv_info,
	NULL,		/* add_card */
	NULL,
	NULL,
	0,
	NULL,
	""
} ;

static functions_t		ca_funcs = {

	7,
	scheduler_control,
	wakeup_control,
	version_callback,
	scheduler_suspend,
	scheduler_resume,
	controller_remove,
	controller_add
} ;

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static char *	invalid_msg   = "Invalid application id #%d.\n";

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static unsigned char  minb (unsigned addr) { return MINB(addr); }
static unsigned short minw (unsigned addr) { return MINW(addr); }
static unsigned long  minl (unsigned addr) { return MINL(addr); }
static unsigned char  pinb (unsigned addr) { return PINB(addr); }
static unsigned short pinw (unsigned addr) { return PINW(addr); }
static unsigned long  pinl (unsigned addr) { return PINL(addr); }

static void moutb (unsigned char  val, unsigned addr) { MOUTB(addr, val); }
static void moutw (unsigned short val, unsigned addr) { MOUTW(addr, val); }
static void moutl (unsigned long  val, unsigned addr) { MOUTL(addr, val); }
static void poutb (unsigned char  val, unsigned addr) { POUTB(addr, val); }
static void poutw (unsigned short val, unsigned addr) { POUTW(addr, val); }
static void poutl (unsigned long  val, unsigned addr) { POUTL(addr, val); }

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#define	PCI_NO_RESOURCE		4
#define PCI_NO_PCI_KERN		3
#define PCI_NO_CARD		2
#define PCI_NO_PCI		1
#define PCI_OK			0

static int alloc_resources (struct pci_dev * dev, card_p cp) {

	UNUSED_ARG (dev);
	if (!request_region (cp->addr_mmio, IO_REGION_LEN, TARGET)) {
		error ("Could not claim i/o region.\n");
	io_problem:
		cp->mmio_base = 0;
		return PCI_NO_RESOURCE;
	}
	cp->mmio_base = cp->addr_mmio;
	cp->mmio_access_ok = TRUE;
	log ("I/O region at 0x%08x\n", cp->mmio_base);
	if (!(cp->data_base = ioremap_nocache (cp->addr_data, cp->len_data))) {
		log ("Could not map data memory into virtual memory.\n");
	data_map_exit:
		release_region (cp->addr_mmio, IO_REGION_LEN);
		goto io_problem;
	}
	cp->data_access_ok = TRUE;
	log ("Controller data memory mapped to %08x\n", cp->data_base);
	if (!(cp->code_base = ioremap_nocache (cp->addr_code, cp->len_code))) {
		log ("Could not map code memory into virtual memory.\n");
	code_map_exit:
		iounmap (cp->data_base);
		cp->data_base = NULL;
		goto data_map_exit;
	}
	cp->code_access_ok = TRUE;
	log ("Controller code memory mapped to %08x\n", cp->code_base);
	pci_set_master (dev);
	return PCI_OK;
} /* alloc_resources */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void free_interrupt (card_p cp) {

	log ("Free IRQ...\n");
	assert (cp);
	if (cp->irq != 0) {
		free_irq (cp->irq, cp);
		cp->irq = 0;
		log ("IRQ freed.\n");
	}
} /* free_interrupt */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void free_resources (card_p cp) {

	assert (cp->mmio_base != 0);
	assert (cp->data_base != NULL);
	assert (cp->code_base != NULL);

	free_interrupt (cp);
	clear_interrupt_callback ();
	log ("Unmapping code, data, i/o regions...\n");
	iounmap (cp->code_base);
	iounmap (cp->data_base);
	release_region (cp->addr_mmio, IO_REGION_LEN);
	info (cp->piowin != NULL);
	if (cp->piowin != NULL) {
		hfree (cp->piowin);
	}
	log ("Resources freed.\n");
} /* free_resources */

/*---------------------------------------------------------------------------*\
\*-B-------------------------------------------------------------------------*/
#define	DBG_FORMAT_START	0xABCD1234
#define	DBG_FORMAT_STOP		0x4321DCBA

void dump (char * p1, char * p2, char * end, unsigned l1, unsigned l2) {

	p1 += sizeof (unsigned);
	p2 += sizeof (unsigned);
	memdump (p1, l1, p1 - p2, "");
	if (l2 != 0) {
		memdump (p2, l2, 0, "");
	}
} /* dump */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void dump_debug_buffer (dbg_buf_p dbp) {
	static char *	bp = NULL;
	unsigned	part1_len, part2_len;
	unsigned	total_len, check_len;
	int		delta;
	
	assert (dbp != NULL);
	if ((dbp->pbuf == NULL) || atomic_read (&dbp->stop)) {
		log ("No debug buffer???\n");
		return;
	}
	
	assert (dbp->pbuf != NULL);
	assert (dbp->pdma != NULL);
	assert (dbp->pbuf >= (char *) dbp->pdma->virt_addr);
	assert (dbp->pbuf < (char *) (dbp->pdma->virt_addr + dbp->len));
	assert ((((unsigned) dbp->pbuf) & 3) == 0);
	if (bp == NULL) {
		bp = dbp->pbuf;
	}
	part1_len = part2_len = 0;
		
	assert (bp >= dbp->pbuf);
	assert (bp < (dbp->pbuf + dbp->len));
	total_len = * ((unsigned *) bp);
	info (total_len < dbp->len);
	if ((total_len > dbp->len) || ((total_len & 3) != 0)) {
		log ("Debug thread needs debugging!\n");
		return;
	}

	delta = ((char *) bp + total_len) - (dbp->pbuf + dbp->len);
	if (delta > 0) {
		part2_len = (unsigned) delta;
		part1_len = total_len - part2_len;
	} else {
		part2_len = 0;
		part1_len = total_len;
	}
	assert ((part2_len & 3) == 0);
	assert ((part1_len & 3) == 0);
	assert (part2_len < dbp->len);

	if (part1_len != 0) {
		check_len = (part2_len == 0) 
			? * ((unsigned *) (bp + part1_len) + 1)
			: * ((unsigned *) (dbp->pbuf + part2_len) + 1);
		assert (check_len == total_len);
		
		dump (
			bp, 
			dbp->pbuf, 
			dbp->pbuf + dbp->len, 
			part1_len, 
			part2_len
		);

		if (part2_len == 0) {
			bp += 2 * sizeof (unsigned) + part1_len;
		} else {
			bp = dbp->pbuf + part2_len + sizeof (unsigned);
		}
	}
} /* dump_debug_buffer */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int debug_thread (void * arg) {
	dbg_buf_p	dbp = (dbg_buf_p) arg;
	unsigned	to;

	assert (dbp != NULL);
	to = (((dbp->tout < 10) ? 10 : dbp->tout) / 10) * TEN_MSECS;
	while (!atomic_read (&dbp->stop)) {
		interruptible_sleep_on_timeout (&dbg_wait, to);
		dump_debug_buffer (dbp);
	}
	log ("Debug thread stopped.\n");
	if (dbp->pdma != NULL) {
		dma_free_buffer (dbp->pdma);
		dma_free (&dbp->pdma);
	}
	hfree (dbp);
	return 0;
} /* debug_thread */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int init_debug_buffer (dbg_buf_p dbp) {
	int		res	= 0;
	unsigned	size	= DBG_MEM_SIZE;
	dma_struct_p	pdma;
	
	assert (dbp != NULL);
	assert (size >= DBG_MIN_SIZE);
	assert (DBG_TIMEOUT > 0);
	assert (dbp->pbuf == NULL);
	pdma = dma_alloc ();
	res = FALSE;
	for ( ; size >= DBG_MIN_SIZE; size /= 2) {
		if ((res = dma_setup (pdma, size))) {
			break;
		}
		log ("Failed to allocate %u byte DMA buffer...\n", size);
	}
	if (!res) {
		dma_exit (pdma);
		dma_free (&pdma);
		return FALSE;
	}
	log (
		"Debug DMA buffer, %u bytes, vaddr %p, paddr %p\n", 
		pdma->length, 
		(void *) pdma->virt_addr, 
		(void *) pdma->phys_addr
	);
	assert (size == pdma->length);
	dbp->pdma = pdma;
	dbp->pbuf = (void *) pdma->virt_addr;
	dbp->len  = size;
	dbp->tout = DBG_TIMEOUT;
	atomic_set (&dbp->stop, 0);
	dbp->pid = kernel_thread (debug_thread, dbp, 0);
	return TRUE;
} /* init_debug_buffer */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void exit_debug_buffer (dbg_buf_p dbp) {
	
	assert (dbp != NULL);
	info (find_task_by_pid (dbp->pid) != NULL);
	log ("Stopping debug thread...\n");
	atomic_set (&dbp->stop, 1);
	if (find_task_by_pid (dbp->pid)) {
		wake_up_interruptible (&dbg_wait);
	} else {
		hfree (dbp);
	}
} /* exit_debug_buffer */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void reset (card_p cp) {
	int	done = 0;

	assert (cp != NULL);
	if (fw_ready ()) {
		assert (cp->c6205_ctx != NULL);
		log ("Loadfile based reset.\n");
		done = fw_send_reset (cp->c6205_ctx);
	}
	if (!done) {
		assert (cp->mmio_base != 0);
		log ("I/O based reset.\n");
		POUTL (
			cp->mmio_base + C6205_PCI_HDCR_OFFSET, 
			C6205_PCI_HDCR_WARMRESET
		);
	}
	mdelay (RESET_DELAY);
} /* reset */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static inline void kill_versions (void) {
	
	if (ctrl_context[0].ctrl != NULL) {
		kill_version (capi_controller[0]);
	}
	kill_version (capi_controller[1]);
} /* kill_versions */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int start_stack (card_p cp) {
	int		res = 0;
	unsigned	num = 0, n;
	unsigned	ofs, len, bof;
	ioaddr_p	pwin;
	dma_struct_p *	ppdma;
	
	/* preparing the stack */
	cp->count = 0;
	table_init (&cp->appls);
	queue_init (&cp->queue);

	assert (capi_lib != NULL);
	assert (capi_lib->cm_register_ca_functions != NULL);
	assert (capi_lib->cm_start != NULL);
	assert (capi_lib->cm_init != NULL);
	(*capi_lib->cm_register_ca_functions) (&ca_funcs);
	if ((*capi_lib->cm_start) ()) {
		error ("Starting the stack failed...\n");
	early_exit:
		queue_exit (&cp->queue);
		table_exit (&cp->appls);
		return FALSE;
	}
	(void) (*capi_lib->cm_init) (0, cp->irq);
	assert (nvers == 2);
	
	/* install an interrupt handler */
	set_interrupt_callback (&irq_callback, cp);
	if (0 != (res = request_irq (cp->irq, &device_interrupt, SA_INTERRUPT | SA_SHIRQ, TARGET, cp))) {
		error ("Could not install irq handler.\n");
	stack_exit:
		assert (capi_lib->cm_exit != NULL);
		(*capi_lib->cm_exit) ();
		clear_interrupt_callback ();
		kill_versions ();
		goto early_exit;
	} else {
		log ("IRQ #%d assigned to " TARGET " driver.\n", cp->irq);
	}

	/* debug buffer */
	assert (capi_lib->cc_need_debug_buffer != NULL);
	if ((*capi_lib->cc_need_debug_buffer) ()) {
		log ("Setting up debug buffer.\n");
		if (dbgbuf == NULL) {
			if (NULL != (dbgbuf = (dbg_buf_p) hcalloc (sizeof (dbg_buf_t)))) {
				if (!init_debug_buffer (dbgbuf)) {
					hfree (dbgbuf);
					goto stack_exit;
				}
				assert (capi_lib->cc_init_debug != NULL);
				(*capi_lib->cc_init_debug) (
					(void *) dbgbuf->pdma->phys_addr, 
					dbgbuf->pdma->length
				);
			}
		}
	} else {
		log ("No debug buffer.\n");
	}

	/* establish DMA buffers */
	ppdma = dma_get_struct_list (&num);
	assert (capi_lib->cc_num_link_buffer != NULL);
	assert (capi_lib->cc_init_dma != NULL);
	n = (*capi_lib->cc_num_link_buffer) (TRUE),
	(*capi_lib->cc_init_dma) (
		&ppdma[0], n,
		&ppdma[n], (*capi_lib->cc_num_link_buffer) (FALSE)
	);
	
	assert (capi_lib->cc_link_version != NULL);
	assert (capi_lib->cc_buffer_params != NULL);
	if ((*capi_lib->cc_link_version) () > 1) {
		num = (*capi_lib->cc_buffer_params) (&ofs, &pwin, &len, &bof);
		dif_set_params (num, ofs, pwin, len, bof);
	}
	reset (cp);
	assert (capi_lib->cc_run != NULL);
	res = (*capi_lib->cc_run) ();
	if (res == FALSE) {	/* Failure */
		error ("Firmware does not respond!\n");
	reset_exit:
		reset (cp);
		if (dbgbuf != NULL) {
			exit_debug_buffer (dbgbuf);
		}
		goto stack_exit;
	}
	assert (capi_lib->cc_compress_code != NULL);
	(*capi_lib->cc_compress_code) ();

	/* start the stack */
	enter_critical ();
	if ((*capi_lib->cm_activate) ()) {
		error ("Activation of the card failed.\n");
		leave_critical ();
		goto reset_exit;
	}
	leave_critical ();
	make_thread ();
	enable_thread ();
	init_flag = 1;
	return TRUE;
} /* start_stack */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void stop_stack (card_p card) {

	if (!init_flag) {
		log ("Stack not initialized.\n");
		return;
	}
	if (dbgbuf != NULL) {
		exit_debug_buffer (dbgbuf);
		dbgbuf = NULL;
	}
	(*capi_lib->cm_exit) ();
	if (thread_pid != -1) {
	     // disable_thread ();
		kill_thread ();
	}
	queue_exit (&card->queue);
	table_exit (&card->appls);
	kill_versions ();
	nvers = 0;
} /* stop_stack */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int reset_card (card_p cp) {

	reset (cp);
	fw_exit (&cp->c6205_ctx);
	return TRUE;
} /* reset_card */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int test_card (card_p cp) {

	UNUSED_ARG (cp);
	return TRUE;
} /* test_card */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int params_ok (card_p cp) {

	UNUSED_ARG (cp);
	return TRUE;
} /* params_ok */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int find_card (card_p cp) {
#if !defined (CONFIG_PCI)
	return PCI_NO_PCI_KERN;
#else
	struct pci_dev * dev = NULL;

	if (!pci_present ()) {
		return PCI_NO_PCI;
	}
	dev = pci_find_device (PCI_VENDOR_ID_AVM, PCI_DEVICE_ID_FCDSL2, dev);
	if (NULL == dev) {
		return PCI_NO_CARD;
	}
	if (0 != pci_enable_device (dev)) {
		log ("Failed to enable the controller.\n");
		return PCI_NO_CARD;
	}

	assert (pci_resource_len (dev, 0) == MAPPED_DATA_LEN);
	assert (pci_resource_len (dev, 1) == MAPPED_CODE_LEN);
	assert (pci_resource_len (dev, 2) == IO_REGION_LEN);

	cp->addr_data = pci_resource_start (dev, 0);
	cp->len_data  = pci_resource_len (dev, 0);
	cp->addr_code = pci_resource_start (dev, 1);
	cp->len_code  = pci_resource_len (dev, 1);
	cp->addr_mmio = pci_resource_start (dev, 2);
	cp->len_mmio  = pci_resource_len (dev, 2);
	cp->irq       = dev->irq;

	log (
		"PCI: " PRODUCT_LOGO ", dev %04x, irq %d\n",
		dev->device,
		cp->irq
	);
	log (
		"     data %08x (%dM), code %08x (%dM), mmio %04x\n",
		cp->addr_data, cp->len_data / MEGABYTE,
		cp->addr_code, cp->len_code / MEGABYTE,
		cp->addr_mmio
	);

	return alloc_resources (dev, cp);
#endif
} /* find_card */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static struct capi_ctr * get_capi_ctr (int ctrl) {
	int	ix;
	
	assert (capi_card);
	assert (capi_card->ctrl2 > 0);
	ix = (capi_card->ctrl2 == ctrl) ? 1 : 0;
	info (NULL != capi_controller[ix]);
	return capi_controller[ix];
} /* get_capi_ctr */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int nbchans (struct capi_ctr * ctrl) {
	per_ctrl_p	C;
	unsigned char *	prf;
	int		temp = 2;
    
	assert (ctrl);
	assert (ctrl->driverdata);
	C = GET_CTRL (ctrl);
	prf = (unsigned char *) C->string[6];
	if (prf != NULL) {
		temp = prf[2] + 256 * prf[3];
	}
	return temp;
} /* nbchans */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static struct sk_buff * make_0xfe_request (unsigned appl) {
	unsigned char    request[8];
	struct sk_buff * skb;

	if (NULL == (skb = alloc_skb (8, GFP_ATOMIC))) {
		lprintf (KERN_ERR, "Unable to allocate message buffer.\n");
	} else {    
		request[0] = 8;
		request[1] = 0;
		request[2] = appl & 0xFF;
		request[3] = (appl >> 8) & 0xFF;
		request[4] = 0xFE;
		request[5] = 0x80;
		memcpy (skb_put (skb, 8), &request, 8);
	}
	return skb;
} /* make_0xfe_request */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void scan_version (struct capi_ctr * ctrl, const char * ver) {
	int		vlen, i;
	char *		vstr;
	per_ctrl_p	C;

	assert (ctrl);
	assert (ctrl->driverdata);
	C = GET_CTRL (ctrl);

	vlen = (unsigned char) ver[0];
	C->version = vstr = (char *) hmalloc (vlen);
	if (NULL == vstr) {
		log ("Could not allocate version buffer.\n");
		return;
	}
	memcpy (C->version, ver + 1, vlen);
	i = 0;
	for (i = 0; i < 8; i++) {
		C->string[i] = vstr + 1;
		vstr += 1 + *vstr;
	} 
#if defined (NDEBUG)
	lprintf (KERN_INFO, "Stack version %s\n", C->string[0]);
#endif
	log ("Library version:    %s\n", C->string[0]);
	log ("Card type:          %s\n", C->string[1]);
	log ("Capabilities:       %s\n", C->string[4]);
	log ("D-channel protocol: %s\n", C->string[5]);
} /* scan_version */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void copy_version (struct capi_ctr * ctrl) {
	char *		tmp;
	per_ctrl_p	C;
	
	assert (ctrl);
	assert (ctrl->driverdata);
	C = GET_CTRL (ctrl);
	
	if (NULL == (tmp = C->string[3])) {
		lprintf (KERN_ERR, "Do not have version information...\n");
		return;
	}
	strncpy (ctrl->serial, tmp, CAPI_SERIAL_LEN);
#if defined (__fcdslsl__)
	if (ctrl == capi_controller[0]) {
		memset (&ctrl->profile, 0, sizeof (capi_profile));
	} else {
		memcpy (&ctrl->profile, C->string[6], sizeof (capi_profile));
	}
#else
	memcpy (&ctrl->profile, C->string[6], sizeof (capi_profile));
#endif
	strncpy (ctrl->manu, "AVM-GmbH", CAPI_MANUFACTURER_LEN);
	ctrl->version.majorversion = 2;
	ctrl->version.minorversion = 0;
	tmp = C->string[0];
	ctrl->version.majormanuversion = (((tmp[0] - '0') & 15) << 4)
					+ ((tmp[2] - '0') & 15);
	ctrl->version.minormanuversion = ((tmp[3] - '0') << 4)
					+ (tmp[5] - '0') * 10
					+ ((tmp[6] - '0') & 15);
} /* copy_version */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void kill_version (struct capi_ctr * ctrl) {
	int		i;
	per_ctrl_p	C;
	
	assert (ctrl);
	assert (ctrl->driverdata);
	C = GET_CTRL (ctrl);

	for (i = 0; i < 8; i++) {
		C->string[i] = NULL;
	}
	if (C->version != NULL) {
		hfree (C->version);
		C->version = NULL;
	}
} /* kill_version */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void pprintf (char * page, int * len, const char * fmt, ...) {
	va_list args;

	va_start (args, fmt);
	*len += vsprintf (page + *len, fmt, args);
	va_end (args);
} /* pprintf */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void stack_wait (unsigned msec) {

	log ("Stack wait request for %u msecs\n", msec);
	assert (!in_interrupt ());
	mdelay (msec);
} /* stack_wait */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
unsigned get_timer (void) {
	unsigned	t = 0;

	if (!capi_card->code_access_ok) {
		t = MINL (capi_card->code_base + timer_offset);
	}
	return t;
} /* get_timer */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void pc_acknowledge (unsigned char num) {
	
	assert (capi_lib != NULL);
	assert (capi_lib->cc_link_version != NULL);
	assert ((*capi_lib->cc_link_version) () > 3);
#if !defined (NDEBUG) && defined (LOG_LINK)
	log ("ACK for buffer #%u\n", num);
#endif
	MOUTL (capi_card->data_base + pc_ack_offset, num);
	assert (MINL (capi_card->data_base + pc_ack_offset) == num);
} /* pc_acknowledge */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int __entry load_ware (struct capi_ctr * ctrl, capiloaddata * ware) {
	card_p		cp;
	c6205_context	ctx;
	char *		buffer;
	int		res;
	ioaddr_p	io;

	assert (ctrl != NULL);
	assert (ware != NULL);
	cp = GET_CARD(ctrl);
	if (load_flag) {
		log ("Firmware has already been loaded!\n");
		return 0;
	}
	assert (firmware_ptr == NULL);
	if (NULL == (buffer = (char *) hmalloc (ware->firmware.len))) {
		error ("Could not allocate firmware buffer.\n");
		return -EIO;
	}
        if (ware->firmware.user) {
		if (copy_from_user (buffer, ware->firmware.data, ware->firmware.len)) {
			error ("Error copying firmware data!\n");
			return -EIO;
		}
	} else {
		memcpy (buffer, ware->firmware.data, ware->firmware.len);
	}
	log ("Loaded %d bytes of firmware.\n", ware->firmware.len);
	firmware_ptr = buffer;
	firmware_len = ware->firmware.len;

	if (NULL == (io = (ioaddr_p) hmalloc (3 * sizeof (ioaddr_t)))) {
		log ("Could not claim memory for i/o structures.\n");
		return -ENOMEM;
	}
	cp->piowin = io;
	
#define	INIT_IO(x,n,f)	x.virt_addr	= (unsigned) cp->n##_base;	\
			x.phys_addr	= cp->addr_##n;			\
			x.length	= cp->len_##n;			\
			x.inb		= &(f##inb);			\
			x.inw		= &(f##inw);			\
			x.inl		= &(f##inl);			\
			x.outb		= &(f##outb);			\
			x.outw		= &(f##outw);			\
			x.outl		= &(f##outl);
			
	INIT_IO(io[0], data, m);
	INIT_IO(io[1], code, m);
	INIT_IO(io[2], mmio, p);
	
#undef INIT_IO
	
	ctx = fw_init (
		cp, 
		firmware_ptr, 
		firmware_len, 
		3, io, 
		&res
	);
	if ((NULL == ctx) || (res != 0)) {
		error ("Error while processing firmware (%d)!\n", res);
	error1:
		fw_exit (&ctx);
		hfree (firmware_ptr);
		hfree (cp->piowin);
		firmware_ptr = NULL;
		cp->piowin = NULL;
		return -EIO;
	}
	cp->c6205_ctx = ctx;
	assert (capi_lib != NULL);
	assert (capi_lib->cc_link_version != NULL);
	assert ((*capi_lib->cc_link_version) () > 1);
	if (!dif_init (cp, (*capi_lib->cc_link_version) () > 1, &get_timer, &pc_acknowledge)) {
		error ("Error while setting up device structures!\n");
	error2:
		goto error1;
	}
	assert (capi_lib->cc_timer_offset != NULL);
	timer_offset = (*capi_lib->cc_timer_offset) ();
	if ((*capi_lib->cc_link_version) () > 3) {
		pc_ack_offset = (*capi_lib->cc_pc_ack_offset) ();
	}

	fw_setup (ctx);
	if (!start_stack (cp)) {
		error ("Error while starting protocoll stack!\n");
	error3:
		dif_exit ();
		goto error2;
	}
	assert (load_flag == FALSE);
	load_flag = TRUE;
	if (ctrl_context[0].ctrl != NULL) {
		assert (capi_controller[0]->ready);
		(*capi_controller[0]->ready) (capi_controller[0]);
	}
	assert (capi_controller[1]->ready);
	(*capi_controller[1]->ready) (capi_controller[1]);
	return 0;
} /* load_ware */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void init_ctrl (void) {

	ctrl_context[0].card = ctrl_context[1].card = NULL;
	ctrl_context[0].ctrl = ctrl_context[1].ctrl = NULL;
	capi_controller[0]   = capi_controller[1]   = NULL;
} /* init_ctrl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int setup_ctrl (card_p cp, int ix) {
	int			cix;
	struct capi_ctr *	cptr;
	
	assert ((0 < ix) && (ix < 3));
	cptr = (*capi_driver->attach_ctr) (&capi_interface, SHORT_LOGO, NULL);
	if (NULL == cptr) {
		free_resources (cp);
		lprintf (KERN_INFO, "Error: Could not attach controller %d.\n", ix);
		return -EBUSY;
	}
	if ((1 == ix) && (cptr->cnr != 1)) {
		cix = ix;
	} else {
		cix = ix - 1;
	}
	log ("Controller %d (%d) --> %d\n", ix, cptr->cnr, cix);
	capi_controller[cix] = cptr;
	ctrl_context[cix].card = cp;
	ctrl_context[cix].ctrl = &ctrl_params[cix];
	capi_controller[cix]->driverdata = (void *) &ctrl_context[cix];
	return 0;
} /* setup_ctrl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int __entry add_card (struct capi_driver * drv, capicardparams * args) {
	card_p  cp;
	int	res = 0;
	char *	msg;

	UNUSED_ARG (drv);
	UNUSED_ARG (args);
	if (NULL != capi_card) {
		lprintf (KERN_ERR, "Cannot handle two controllers!\n");
 		return -EBUSY;
	}
	if (NULL == (cp = (card_p) hcalloc (sizeof (card_t)))) {
		lprintf (KERN_ERR, "Card object allocation failed.\n");
		return -EIO;
	}
	capi_card = cp;
	init_waitqueue_head (&wait);

	switch (find_card (cp)) {
        
	case PCI_OK:
		break;
	case PCI_NO_RESOURCE:
		msg = "Could not claim required resources.";
		res = -EBUSY;
		break;
        case PCI_NO_PCI_KERN:
		msg = "No PCI kernel support available.";
		res = -ESRCH;
		break;
        case PCI_NO_CARD:
		msg = "Could not locate any " PRODUCT_LOGO ".";
		res = -ESRCH;
		break;
        case PCI_NO_PCI:
		msg = "No PCI busses available.";
		res = -ESRCH;
		break;
        default:
		msg = "Unknown PCI related problem...";
		res = -EINVAL;
		break;
	}
	if (res != 0) {
	msg_exit:
		error ("Error: %s\n", msg);
		hfree (cp);
		return res;
	}
	if (!params_ok (cp)) {
		free_resources (cp);
		msg = "Error: Invalid parameters!";
		res = -EINVAL;
		goto msg_exit;
	}
	inc_use_count ();
	init_ctrl ();
#if defined (__fcdslsl__)
	if (0 != (res = setup_ctrl (cp, 2))) {
		dec_use_count ();
		return res;
	}
#else
	if (0 != (res = setup_ctrl (cp, 1))) {
		dec_use_count ();
		return res;
	}
	if (ctrl_context[0].ctrl != NULL) {
		if (0 != (res = setup_ctrl (cp, 2))) {
			(*capi_driver->detach_ctr) (capi_controller[0]);
			capi_controller[0] = NULL;
			dec_use_count ();
			return res;
		}
	}
#endif
	cp->ctrl2 = capi_controller[1]->cnr;
	cp->ctxp  = NULL;
	return res;
} /* add_card */ 

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry reset_ctrl (struct capi_ctr * ctrl) {
	card_p		cp;
	appl_t *	appp;

	assert (ctrl != NULL);
	assert (ctrl->driverdata != NULL);
	cp = GET_CARD (ctrl);
	assert (cp != NULL);
        if (0 != cp->count) {
		lprintf (KERN_INFO, "Removing registered applications!\n");
		info (cp->appls);
		if (cp->appls != NULL) {
			appp = first_appl (cp->appls);
			while (appp != NULL) {
				free_ncci (appp->id, (unsigned) -1);
				appp = next_appl (cp->appls, appp);
			}
		}
	}
	stop_stack (cp);
	/* to do: remove marked buffers */
	reset_card (cp);
	if (firmware_ptr != NULL) {
		hfree (firmware_ptr);
		firmware_ptr = NULL;
		firmware_len = 0;
	}
	dif_exit ();
	load_flag = FALSE;
	if (ctrl_context[0].ctrl != NULL) {
		assert (capi_controller[0]->reseted);
		(*capi_controller[0]->reseted) (capi_controller[0]);
	}
	assert (capi_controller[1]->reseted);
	(*capi_controller[1]->reseted) (capi_controller[1]);
} /* reset_ctrl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry remove_ctrl (struct capi_ctr * ctrl) {
	card_p	cp;

	assert (ctrl);
	assert (ctrl->driverdata);
	cp = GET_CARD (ctrl);
	info (load_flag == FALSE);
	if (load_flag) {
		lprintf (KERN_ERR, "Remove controller w/o reset!\n");
		reset_card (cp);
	}
	free_resources (cp);
	assert (capi_driver->detach_ctr);
	if (ctrl_context[0].ctrl != NULL) {
		(*capi_driver->detach_ctr) (capi_controller[0]);
	}
	(*capi_driver->detach_ctr) (capi_controller[1]);
	dec_use_count ();
	init_ctrl ();
	hfree (capi_card);
	capi_card = NULL;
} /* remove_ctrl */

/*---------------------------------------------------------------------------*\
\*-L-------------------------------------------------------------------------*/
void lock (void) {
	unsigned long	local_flags;
	
	info (!spin_is_locked (&qt_lock));
	spin_lock_irqsave (&qt_lock, local_flags);
	qt_flags = local_flags;
} /* lock */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void unlock (void) {
	unsigned long	local_flags = qt_flags;

	spin_unlock_irqrestore (&qt_lock, local_flags);
} /* unlock */

/*---------------------------------------------------------------------------*\
\*-C-------------------------------------------------------------------------*/
static inline void check (void) {
	unsigned long	flags;
	spinlock_t	chk_lock = SPIN_LOCK_UNLOCKED;

	if (atomic_xchg (&xfer_flag, 0)) {
		spin_lock_irqsave (&chk_lock, flags);
		xfer_handler (capi_card);
		spin_unlock_irqrestore (&chk_lock, flags);
	}
} /* check */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void enter_critical (void) {

	if (!atomic_read (&crit_count) && !in_interrupt ()) {
		check ();
	}
	atomic_inc (&crit_count);
} /* enter_critical */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void leave_critical (void) {
	
	assert (in_critical ());
	atomic_dec (&crit_count);
	if (!atomic_read (&crit_count) && !in_interrupt ()) {
		check ();
	}
} /* leave_critical */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry register_appl (
	struct capi_ctr *	ctrl, 
	u16			appl, 
	capi_register_params *	args
) {
	card_t *		card;
	appl_t *		appp;
	void *			ptr;
	unsigned		nc;

	mlog ("REGISTER(ctrl:%d,appl:%u)\n", ctrl->cnr, appl);
	assert (ctrl);
	assert (ctrl->driverdata);
	assert (args);
	card = GET_CARD (ctrl);
	if ((int) args->level3cnt < 0) {
		nc = nbchans (ctrl) * -((int) args->level3cnt);
	} else {
		nc = args->level3cnt;
	}
	if (0 == nc) {
		nc = nbchans (ctrl);
	}
	if (ctrl->cnr == card->ctrl2) {	
		appp = create_appl (
				card->appls, 
				appl, 
				nc, 
				args->datablkcnt, 
				args->datablklen
				);
		if (NULL == appp) {
			log ("Unable to create application record.\n");
			return;
		}
		ptr = hcalloc (card->length);		
		if (NULL == ptr) {
			lprintf (KERN_ERR, "Not enough memory for application data.\n");
			remove_appl (card->appls, appp);
			return;
		} else {
			inc_use_count ();
			card->count++;
			appp->data = ptr;
			assert (card->reg_func);
			(*card->reg_func) (ptr, appl);
		}
	}
	assert (ctrl->appl_registered);
	(*ctrl->appl_registered) (ctrl, appl);
} /* register_appl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry release_appl (struct capi_ctr * ctrl, u16 appl) {
	card_t *		card;
	struct sk_buff *	skb;
	appl_t *		appp;

	mlog ("RELEASE(ctrl:%d,appl:%u)\n", ctrl->cnr, appl);
	assert (ctrl);
	assert (ctrl->driverdata);
	card =  GET_CARD (ctrl);
	if (ctrl->cnr == card->ctrl2) {	
		if (NULL == (appp = search_appl (card->appls, appl))) {
			log ("Attempt to release unknown application (id #%u)\n", appl);
			return;
		}
		skb = make_0xfe_request (appl);	
		handle_message (card->queue, appp, skb);
		kick_scheduler ();
		appp->dying = TRUE;
	}
} /* release_appl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry send_msg (struct capi_ctr * ctrl, struct sk_buff * skb) {
	card_t *	card;
	unsigned char *	byte;
	unsigned	appl;
	appl_t *	appp;

	assert (ctrl);
	assert (ctrl->driverdata);
	assert (skb);
	card = GET_CARD (ctrl);
	byte = skb->data;
	appl = byte[2] + 256 * byte[3];
	appp = search_appl (card->appls, appl);
	if ((NULL == appp) || appp->dying) {
		log ("Message for unknown application (id %u).\n", appl);
		return;
	}
	handle_message (card->queue, appp, skb);
	kick_scheduler ();
} /* send_msg */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
char * __entry proc_info (struct capi_ctr * ctrl) {
	card_p		cp;
	per_ctrl_p	C;
	static char	text[80];

	assert (ctrl);
	assert (ctrl->driverdata);
	cp = GET_CARD (ctrl);
	C  = GET_CTRL (ctrl);
	if (load_flag) {
		SNPRINTF (
			text, 
			sizeof (text),
			"%s %s data %08lx code %08lx io %lx %u",
			C->version ? C->string[1] : "A1",
			C->version ? C->string[0] : "-",
			cp->addr_code, cp->addr_data, cp->addr_mmio,
			cp->irq
		);
	} else {
		SNPRINTF (
			text,
			sizeof (text),
			TARGET " device data %08lx code %08lx io %lx irq %u",
			cp->addr_code, cp->addr_data, cp->addr_mmio,
			cp->irq
		);
	}
	return text;
} /* proc_info */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int __entry ctr_info (
	char *			page, 
	char **			start, 
	off_t			ofs,
	int			count, 
	int *			eof,
	struct capi_ctr *	ctrl
) {
	card_p			cp;
	per_ctrl_p		C;
	char *			temp;
	unsigned char		flag;
	int			len = 0;

	assert (ctrl);
	assert (ctrl->driverdata);
	cp = GET_CARD (ctrl);
	C  = GET_CTRL (ctrl);
	pprintf (page, &len, "%-16s %s\n", "name", SHORT_LOGO);
	pprintf (page, &len, "%-16s %d\n", "irq", cp->irq);
	pprintf (page, &len, "%-16s %08lx\n", "data", cp->data_base);
	pprintf (page, &len, "%-16s %08lx\n", "code", cp->code_base);
	pprintf (page, &len, "%-16s %lx\n", "io", cp->mmio_base);

	if (load_flag) {
		temp = C->version ? C->string[1] : "A1";
		pprintf (page, &len, "%-16s %s\n", "type", temp);

		temp = C->version ? C->string[0] : "-";
		pprintf (page, &len, "%-16s %s\n", "ver_driver", temp);
		pprintf (page, &len, "%-16s %s\n", "ver_cardtype", SHORT_LOGO);

		flag = ((unsigned char *) (ctrl->profile.manu))[3];
		if (flag) {
			pprintf(page, &len, "%-16s%s%s%s%s%s%s%s\n", "protocol",
				(flag & 0x01) ? " DSS1" : "",
				(flag & 0x02) ? " CT1" : "",
				(flag & 0x04) ? " VN3" : "",
				(flag & 0x08) ? " NI1" : "",
				(flag & 0x10) ? " AUSTEL" : "",
				(flag & 0x20) ? " ESS" : "",
				(flag & 0x40) ? " 1TR6" : ""
			);
		}
		flag = ((unsigned char *) (ctrl->profile.manu))[5];
		if (flag) {
			pprintf(page, &len, "%-16s%s%s%s%s\n", "linetype",
				(flag & 0x01) ? " point to point" : "",
				(flag & 0x02) ? " point to multipoint" : "",
				(flag & 0x08) ? " leased line without D-channel" : "",
				(flag & 0x04) ? " leased line with D-channel" : ""
			);
		}
	}

	if (len < ofs) {
		return 0;
	}
	*eof = 1;
	*start = page - ofs;
	return ((count < len - ofs) ? count : len - ofs);
} /* ctr_info */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int __entry drv_info (
	char *			page, 
	char **			start, 
	off_t			ofs,
	int			count, 
	int *			eof, 
	struct capi_driver *	drv
) {
	int			len = 0;

	UNUSED_ARG (drv);
	pprintf (page, &len, "%-16s %s\n", "name", TARGET);
	pprintf (page, &len, "%-16s %s\n", "revision", capi_interface.revision);
	if (len < ofs) {
		return 0;
	}
	*eof = 1;
	*start = page - ofs;
	return ((count < len - ofs) ? count : len - ofs);
} /* drv_info */
 
/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void * data_by_id (unsigned appl_id) {
	appl_t * appp;

	appp = search_appl (capi_card->appls, appl_id);	
	return (appp != NULL) ? appp->data : NULL;
} /* data_by_id */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
unsigned attr_by_id (unsigned appl_id) {
	appl_t * appp;

	appp = search_appl (capi_card->appls, appl_id);	
	return (appp != NULL) ? appp->attr : 0;
} /* attr_by_id */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void * first_data (int * res) {
	appl_t * appp;

	assert (res);
	appp = first_appl (capi_card->appls);
	*res = (appp != NULL) ? 0 : -1;
	return (appp != NULL) ? appp->data  : NULL;
} /* first_data */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void * next_data (int * res) {
	appl_t * appp;

	assert (res);
	if (NULL != (appp = get_appl (capi_card->appls, *res))) {
		appp = next_appl (capi_card->appls, appp);
	}
	*res = (appp != NULL) ? 1 + *res : -1;
	return (appp != NULL) ? appp->data  : NULL;
} /* next_data */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int appl_profile (unsigned appl_id, unsigned * bs, unsigned * bc) {
	appl_t * appp;

	appp = search_appl (capi_card->appls, appl_id);	
	if (NULL == appp) {
		return 0;
	}
	*bs = appp->blk_size; 
	*bc = appp->blk_count; 
	return 1;
} /* appl_profile */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int msg2stack (unsigned char * msg) {
	unsigned		mlen;
	unsigned		appl;
	unsigned		nctr;
	__u32			ncci;
	unsigned		hand;
	unsigned char *		mptr;
	unsigned char *		dptr;
	struct sk_buff *	skb;
	unsigned		temp;
	int			res = 0;

	assert (msg);
	if (!queue_is_empty (capi_card->queue)) {
		res = 1;
		skb = queue_peek (capi_card->queue);	
		assert (skb);
		mptr = (unsigned char *) skb->data;
		mlen = mptr[0] + 256 * mptr[1]; 
		appl = mptr[2] + 256 * mptr[3];
		nctr = mptr[8] & 127;
		
		mlog (
			"PUT_MESSAGE(ctrl:%d,appl:%u,cmd:0x%02X,subcmd:0x%02X)\n", 
			nctr,
			appl, 
			mptr[4], 
			mptr[5]
		);

		if ((0x86 == mptr[4]) && (0x80 == mptr[5])) { /* DATA_B3_REQ */
			ncci = mptr[8] 
				+ 256 * (mptr[9] 
					+ 256 * (mptr[10] 
						+ 256 * mptr[11]
					)	);
			hand = mptr[18] + 256 * mptr[19];
			temp = (unsigned) (mptr + mlen);
			dptr = mptr + 12;
			*dptr++ = temp & 0xFF;	temp >>= 8;
			*dptr++ = temp & 0xFF;	temp >>= 8;
			*dptr++ = temp & 0xFF;	temp >>= 8;
			*dptr++ = temp & 0xFF;
			memcpy (msg, mptr, mlen);
			queue_park (capi_card->queue, appl, ncci, hand);
		} else {
			memcpy (msg, mptr, mlen);
			queue_drop (capi_card->queue);
		}
	}
	return res;
} /* msg2stack */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void msg2capi (unsigned char * msg) {
	unsigned		mlen; 
	unsigned		appl;
	unsigned		nctr;
	__u32			ncci;
	unsigned		hand;
	unsigned		dlen;
	unsigned char *		dptr;
	unsigned		temp;
	struct sk_buff *	skb;
	struct capi_ctr *	ctrl;

	assert (msg);
	mlen = msg[0] + 256 * msg[1];
	appl = msg[2] + 256 * msg[3];
	nctr = msg[8] & 127;

	mlog (
		"GET_MESSAGE(ctrl:%u,appl:%d,cmd:0x%02X,subcmd:0x%02X)\n", 
		nctr,
		appl, 
		msg[4], 
		msg[5]
	);
	if ((0x86 == msg[4]) && (0x81 == msg[5])) {	/* DATA_B3_CONF */
		hand = msg[12] + 256 * msg[13];
		ncci = msg[8] 
			+ 256 * (msg[9] 
				+ 256 * (msg[10] 
					+ 256 * msg[11]
				)	);
		queue_conf (capi_card->queue, appl, ncci, hand);
	}
	if ((0x86 == msg[4]) && (0x82 == msg[5])) {	/* DATA_B3_IND */
		dlen = msg[16] + 256 * msg[17];
		skb = alloc_skb (
				mlen + dlen + ((mlen < 30) ? (30 - mlen) : 0), 
				GFP_ATOMIC
				); 
		if (NULL == skb) {
			lprintf (
				KERN_ERR, 
				"Unable to build CAPI message skb. Message lost.\n"
			);
			return;
		}
		/* Messages are expected to come with 32 bit data pointers. 
		 * The kernel CAPI works with extended (64 bit ready) message 
		 * formats so that the incoming message needs to be fixed, 
		 * i.e. the length gets adjusted and the required 64 bit data 
		 * pointer is added.
		 */
		temp = msg[12] + 256 * (msg[13] 
					+ 256 * (msg[14] + 256 * msg[15]));
		dptr = (unsigned char *) temp;
		if (mlen < 30) {
			msg[0] = 30;
			ncci   = 0;	
			memcpy (skb_put (skb, mlen), msg, mlen);
			memcpy (skb_put (skb, 4), &ncci, 4);	
			memcpy (skb_put (skb, 4), &ncci, 4);
		} else {
			memcpy (skb_put (skb, mlen), msg, mlen);
        	}
		memcpy (skb_put (skb, dlen), dptr, dlen); 
	} else {
		if (NULL == (skb = alloc_skb (mlen, GFP_ATOMIC))) {
			lprintf (
				KERN_ERR, 
				"Unable to build CAPI message skb."
							" Message lost.\n"
			);
			return;
		}
		memcpy (skb_put (skb, mlen), msg, mlen);
	}
	ctrl = get_capi_ctr (nctr);
	info (ctrl != NULL);
	if (ctrl != NULL) {
		assert (ctrl->cnr == (int) nctr);
		assert (ctrl->handle_capimsg != NULL);
		(*ctrl->handle_capimsg) (ctrl, appl, skb);
	}
} /* msg2capi */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void new_ncci (
	unsigned		appl_id, 
	__u32			ncci,
	unsigned		winsize,
	unsigned		blksize
) {
	appl_t *		appp;
	ncci_t *		nccip;
	struct capi_ctr *	ctrl;
	unsigned		nctr;
	
	nctr = ncci & 127;
	mlog ("NEW NCCI(ctrl:%u,appl:%u,ncci:0x%08X)\n", nctr, appl_id, ncci);
	if (NULL == (appp = search_appl (capi_card->appls, appl_id))) {
		lprintf (KERN_ERR, invalid_msg, appl_id);
		return;
	}
	nccip = create_ncci (
			capi_card->appls, 
			appp, 
			(NCCI_t) ncci, 
			winsize, 
			blksize
			);
	if (NULL == nccip) {
		log ("Cannot handle new NCCI...\n");
		return;
	}
	ctrl = get_capi_ctr (nctr);
	assert (ctrl != NULL);	
	assert (ctrl->cnr == (int) nctr);
	assert (ctrl->new_ncci != NULL);
	(*ctrl->new_ncci) (ctrl, appl_id, ncci, winsize);
} /* new_ncci */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void free_ncci (unsigned appl_id, __u32 ncci) {
	appl_t *		appp;
	ncci_t *		nccip;
	struct capi_ctr *	ctrl;
	unsigned		nctr;
	
	appp = search_appl (capi_card->appls, appl_id);
	if (NULL == appp) {
		lprintf (KERN_ERR, invalid_msg, appl_id);
		return;
	}
	if (0xFFFFFFFF == ncci) {		/* 2nd phase RELEASE */
		assert (appp->dying);
		dec_use_count ();
		capi_card->count--;
		mlog ("FREE APPL(appl:%u)\n", appl_id);
		(*capi_card->rel_func) (appp->data);			
		remove_appl (capi_card->appls, appp);		
		if (ctrl_context[0].ctrl != NULL) {
			assert (capi_controller[0]->appl_released);
			(*capi_controller[0]->appl_released) (capi_controller[0], appl_id);
		}
		assert (capi_controller[1]->appl_released);
		(*capi_controller[1]->appl_released) (capi_controller[1], appl_id);
	} else if (NULL != (nccip = locate_ncci (appp, ncci))) {
		nctr = ncci & 127;
		ctrl = get_capi_ctr (nctr);
		assert (ctrl);		
		assert (ctrl->cnr == (int) nctr);
		mlog ("FREE NCCI(ctrl:%u, appl:%u,ncci:0x%08X)\n", nctr,  appl_id, ncci);
		(*ctrl->free_ncci) (ctrl, appl_id, ncci);
		remove_ncci (capi_card->appls, appp, nccip);
	} else {
		lprintf (KERN_ERR, "Attempt to free unknown NCCI.\n");
	}
} /* free_ncci */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
unsigned char * data_block (unsigned appl_id, __u32 ncci, unsigned handle) {
	appl_t * appp;
 
	appp = search_appl (capi_card->appls, appl_id);
	if (NULL == appp) {
		lprintf (KERN_ERR, invalid_msg, appl_id);
		return NULL;
	}
	return ncci_data_buffer (capi_card->appls, appp, ncci, handle);
} /* data_block */

/*-S-------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int sched_thread (void * arg) {

	UNUSED_ARG (arg);
	daemonize ();
	SNPRINTF (current->comm, 16, "%s_thread", TARGET);	/* See: sched.h */
	log ("Starting kicker thread '%s'...\n", current->comm);
	while (atomic_read (&thread_flag)) {
		/* Start/stop logic */
		if (!atomic_read (&thread_capi_flag)) {
			interruptible_sleep_on (&capi_wait);
		} else {
			if (atomic_read (&scheduler_enabled)) {
				interruptible_sleep_on_timeout (&wait, TEN_MSECS);
			} else {
				interruptible_sleep_on (&wait);
			}
		}
		assert (!in_critical ());
		/* Enable/disable logic */
		if (!atomic_read (&thread_flag)) {
			info (atomic_read (&thread_flag));
			break;
		}
		if (!atomic_read (&thread_capi_flag)) {
			info (atomic_read (&thread_capi_flag));
			continue;
		}
                if (!atomic_read (&scheduler_enabled)) {
			info (atomic_read (&scheduler_enabled));
			continue;
		}
		/* Body of thread, invoke scheduler */
		if (spin_trylock (&stack_lock)) {
			os_timer_poll ();
			assert (capi_lib->cm_schedule);
			if ((*capi_lib->cm_schedule) ()) {
				scheduler_control (TRUE); 
			}
			spin_unlock (&stack_lock);
		}
	}
	log ("Scheduler thread stopped.\n");
	up (&thread_sync);
	return 0;
} /* sched_thread */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int make_thread (void) {

	if (-1 != thread_pid) {
		log ("Thread[%d] already running.\n", thread_pid);
		return thread_pid;
	}
	atomic_set (&thread_flag, 1);
	atomic_set (&thread_capi_flag, 0);
	init_waitqueue_head (&capi_wait);
	thread_pid = kernel_thread (sched_thread, NULL, 0);
	log ("Thread[%d] started.\n", thread_pid);
	return thread_pid;
} /* make_thread */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void kill_thread (void) {

	atomic_set (&thread_flag, 0);
	if (NULL == find_task_by_pid (thread_pid)) {
		log ("Thread[%d] has died before!\n", thread_pid);
	} else {
		if (!atomic_read (&thread_capi_flag)) {
			wake_up_interruptible (&capi_wait);
		} else {
			wake_up_interruptible (&wait);
		}
		log ("Thread signalled, waiting for termination...\n");
		down (&thread_sync);
		log ("Thread[%d] terminated.\n", thread_pid);
	}
	thread_pid = -1;
} /* kill_thread */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void enable_thread (void) {

	info (!atomic_read (&thread_capi_flag));
	if (!atomic_read (&thread_capi_flag)) {
		atomic_set (&thread_capi_flag, 1);
		wake_up_interruptible (&capi_wait);
		log ("Thread enabled.\n");
	}
} /* enable_thread */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void disable_thread (void) {

	info (atomic_read (&thread_capi_flag));
	if (atomic_read (&thread_capi_flag)) {
		atomic_set (&thread_capi_flag, 0);
		wake_up_interruptible (&wait);
		log ("Thread disabled.\n");
	}
} /* disable_thread */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
int irq_callback (void * arg) {
	card_p	cp = (card_p) arg;

	assert (cp == capi_card);
	kick_scheduler ();
	return TRUE;
} /* irq_callback */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void wakeup_control (unsigned enable_flag) { 
	
	enable_flag = 0; 
} /* wakeup_control */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void version_callback (char * vp) { 
	
	assert (nvers < 2);
	if ((0 == nvers) && (NULL == ctrl_context[0].ctrl)) {
		++nvers;
		log ("Version string #0 rejected.\n");
		return;
	}
	log ("Version string #%d:\n", nvers);
	scan_version (capi_controller[nvers], vp);
	copy_version (capi_controller[nvers]);
	++nvers;
} /* version_callback */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static unsigned	scheduler_suspend (unsigned long timeout) { 
	
	timeout = 0; 
	return 0; 
} /* scheduler_suspend */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static unsigned	scheduler_resume (void) { 
	
	return 0; 
} /* scheduler_resume */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static unsigned	controller_remove (void) { 

	assert (0);
	return 0; 
} /* controller_remove */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static unsigned	controller_add (void) { 
	
	assert (0);
	return 0; 
} /* controller_add */
	
/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
void kick_scheduler (void) {

	if ((thread_pid != -1) && (NULL != find_task_by_pid (thread_pid))) {
		atomic_set (&scheduler_enabled, 1);
		wake_up_interruptible (&wait);
	} 
} /* kick_scheduler */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void scheduler_control (unsigned ena) {
	int	enabled = (int) ena;
	int	changed;

	enter_critical ();
	changed = (atomic_read (&scheduler_enabled) != enabled);
	leave_critical ();
	if (changed) {
		if (enabled) {
			kick_scheduler ();
		} else {
			atomic_set (&scheduler_enabled, 0);
			wake_up_interruptible (&wait);
		}
	}
} /* scheduler_control */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
void init (
	unsigned len, 
	void  (* reg) (void *, unsigned),
	void  (* rel) (void *),
	void  (* dwn) (void)
) {
	assert (reg);
	assert (rel);
	assert (dwn);

	capi_card->length   = len;
	capi_card->reg_func = reg;
	capi_card->rel_func = rel;
	capi_card->dwn_func = dwn;
} /* init */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int driver_init (void) {

	strncpy (
		capi_interface.revision, 
		REVISION, 
		sizeof (capi_interface.revision)
	);
	return (NULL != (capi_lib = link_library (&capi_card)));
} /* driver_init */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void driver_exit (void) {

	assert (capi_lib);
	free_library ();
	capi_lib = NULL;
} /* driver_exit */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/

