/* 
 * main.c
 * Copyright (C) 2002, AVM GmbH. All rights reserved.
 * 
 * This Software is  free software. You can redistribute and/or
 * modify such free software under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * The free software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this Software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA, or see
 * http://www.opensource.org/licenses/lgpl-license.html
 * 
 * Contact: AVM GmbH, Alt-Moabit 95, 10559 Berlin, Germany, email: info@avm.de
 */

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/ctype.h>
#include <linux/skbuff.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/capi.h>
#include <linux/usb.h>
#include "capilli.h"
#include "lib.h"
#include "driver.h"
#include "main.h"

#define	INIT_FAILED		-1
#define	INIT_SUCCESS		0

#define	VENDOR_ID_AVM		0x057C
#if defined (__fcdslusb__)
#define	PRODUCT_ID		0x2300
#else
#error You have to define a card identifier!
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
EXPORT_NO_SYMBOLS;

short int	VCC = 0x01;
short int	VPI = 0x01;
short int	VCI = 0x20;

MODULE_PARM (VCC, "h");
MODULE_PARM (VPI, "h");
MODULE_PARM (VCI, "h");

MODULE_PARM_DESC (VCC, "VCC - Virtual Channel Connection");
MODULE_PARM_DESC (VPI, "VPI - Virtual Path Identifier");
MODULE_PARM_DESC (VCI, "VCI - Virtual Channel Identifier");

#if defined (MODULE_LICENSE)
MODULE_LICENSE ("Proprietary");
#endif
#if defined (MODULE_DESCRIPTION)
MODULE_DESCRIPTION ("CAPI4Linux: Driver for " PRODUCT_LOGO);
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
static struct usb_device_id fdslusb_id_table[] __initdata = {
	{ USB_DEVICE(VENDOR_ID_AVM, PRODUCT_ID) },
	{ /* Terminating entry */ }
} ;

MODULE_DEVICE_TABLE (usb, fdslusb_id_table);
#else
#define __exit
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void fdslusb_disconnect (struct usb_device *, void *);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
static void * fdslusb_probe (struct usb_device *, unsigned int);
#else
static void * fdslusb_probe (struct usb_device *, unsigned int, const struct usb_device_id *);
#endif

static struct usb_driver fdslusb_driver = {
	name:		TARGET,
	probe:		fdslusb_probe,
	disconnect:	fdslusb_disconnect,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
	id_table:	fdslusb_id_table,
#endif
} ;

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static char * 		REVCONST = "$Revision: $";
char          		REVISION[32];

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#if !defined (NDEBUG)
static void base_address (void) {

	log ("Base address: %p\n", base_address);
} /* base_address*/
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
static void * fdslusb_probe (struct usb_device * dev, unsigned int ifnum) 
#else
static void * fdslusb_probe (
		struct usb_device * 		dev, 
		unsigned int 			ifnum,
		const struct usb_device_id *	devid
)
#endif
{
	dev_context_p				pdc;
	struct usb_interface_descriptor *	iface;
	struct usb_config_descriptor *		cfgd;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
	UNUSED_ARG (devid);
#endif

	if ((VENDOR_ID_AVM != dev->descriptor.idVendor)
	||  (PRODUCT_ID != dev->descriptor.idProduct)) 
	{
	exit1:
		log ("Probe function failed!\n");
		return NULL;
	}
	MOD_INC_USE_COUNT;
	log ("Probe for device #%d, interface %d.\n", dev->devnum, ifnum);
	if (NULL == (pdc = (dev_context_p) hmalloc (sizeof (dev_context_t)))) {
		error ("Could not allocate device context.\n");
	exit2:
		MOD_DEC_USE_COUNT;
		goto exit1;
	}
	fdslusb_capi_ctx = pdc;
	memset (pdc, 0, sizeof (dev_context_t));
	pdc->dev = dev;
	atomic_set (&pdc->is_open, 0);

	cfgd = dev->actconfig;
	assert (cfgd != NULL);
	if (usb_interface_claimed (cfgd->interface + ifnum)) {
		error ("Interface already claimed.\n");
	exit3:
		hfree (pdc);
		goto exit2;
	} 
	iface = cfgd->interface[ifnum].altsetting + 0;
	assert (iface != NULL);

	pdc->epwrite = iface->endpoint + 0;
        pdc->epread  = iface->endpoint + 1;
	if (usb_set_configuration (dev, cfgd->bConfigurationValue)) {
		error ("Could not set configuration.\n");
	exit4:
		goto exit3;
	}
	if (usb_set_interface (dev, 0, 0)) {
		error ("Could not set interface.\n");
	exit5:
		goto exit4;
	}
	card_config = cfgd->bConfigurationValue;
	log ("Device is in config #%d.\n", card_config);

	if (NULL == (pdc->tx_buffer = (char *) hmalloc (MAX_TRANSFER_SIZE))) {
		error ("Could not allocate tx buffer.\n");
	exit6:
		goto exit5;
	}
	if (NULL == (pdc->tx_urb = usb_alloc_urb (0))) {
		error ("Could not allocate tx urb.\n");
	exit7:
		hfree (pdc->tx_buffer);
		goto exit6;
	}
	if (!init_urbs ()) {
		error ("Error while allocating Rx URBs.\n");
	exit8:
		usb_free_urb (pdc->tx_urb);
		goto exit7;
	}
	if (NULL == (fdslusb_capi_lib = link_library (&fdslusb_capi_ctx))) {
		error ("Linking to library failed.\n");
	exit9:
		free_urbs ();
		goto exit8;
	}
	init_closing_task ();
	make_thread ();
	if (0 != fdslusb_add_card (&fdslusb_capi_interface, NULL)) {
		kill_thread ();
		free_library ();
		goto exit9;
	}
	return pdc;
} /* fdslusb_probe */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void close_complete (void * ptr) {
	dev_context_p	pdc = (dev_context_p) ptr;

	log ("Disconnect completed.\n");
	assert (ptr != NULL);
	free_library ();
	if (NULL != pdc->tx_buffer) hfree (pdc->tx_buffer);
	if (NULL != pdc->tx_urb) usb_free_urb (pdc->tx_urb);
	free_urbs ();
	fdslusb_capi_ctx = NULL;
	fdslusb_capi_lib = NULL;
	hfree (pdc);
	MOD_DEC_USE_COUNT;
} /* close_complete */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void fdslusb_disconnect (struct usb_device * dev, void * ptr) {
	dev_context_p	pdc = (dev_context_p) ptr;

	UNUSED_ARG (dev);
	assert (ptr != NULL);
	log ("Disconnect for device #%d.\n", dev->devnum);
	if (NULL != fdslusb_capi_ctrl[1]) {	
		/* In case of sudden removal... */
		start_closing_task (&close_complete, pdc);
	} else {
		close_complete (pdc);
	}
} /* fdslusb_disconnect */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int __init fdslusb_init (void) {
	int	urr, res = INIT_FAILED;
	char *	tmp;

#if !defined (NDEBUG)
	base_address ();
#endif
	MOD_INC_USE_COUNT;
	if ((NULL != (tmp = strchr (REVCONST, ':'))) && isdigit (*(tmp + 2))) {
		strncpy (REVISION, tmp + 1, sizeof (REVISION));
		tmp = strchr (REVISION, '$');
		*tmp = 0;
	} else {
		strcpy (REVISION, REV_DEFAULT);
	}
        lprintf (KERN_INFO, "%s, revision %s\n", DRIVER_LOGO, REVISION);
	lprintf (KERN_INFO, "(%s built on %s at %s)\n", TARGET, __DATE__, __TIME__);

	/*-------------------------------------------------------------------*\
	 * 64 bit CAPI is not supported yet.
	\*-------------------------------------------------------------------*/
	if (sizeof (char *) > 4) {
		lprintf (KERN_ERR, "Cannot deal with 64 bit CAPI messages!\n");
		goto exit;
	}
	
	lprintf (KERN_INFO, "Loading...\n");
	strncpy (fdslusb_capi_interface.revision, REVISION,
					sizeof (fdslusb_capi_interface.revision));
	if (NULL == (fdslusb_drv_interface = attach_capi_driver (&fdslusb_capi_interface))) {
		error ("Could not attach the driver.\n");
		goto exit;
	}
	if (0 > (urr = usb_register (&fdslusb_driver))) {
		detach_capi_driver (&fdslusb_capi_interface);
		error ("Function usb_register returned %d.\n", urr);
	} else {
		res = INIT_SUCCESS;
	}
exit:
	MOD_DEC_USE_COUNT;
	lprintf (KERN_INFO, "%soaded.\n", res == INIT_SUCCESS ? "L" : "Not l");
	return res;
} /* fdslusb_init */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void __exit fdslusb_exit (void) {
#ifndef NDEBUG
	unsigned b, n;
#endif
	lprintf (KERN_INFO, "Removing...\n");
	kill_thread ();
	detach_capi_driver (&fdslusb_capi_interface);
	usb_deregister (&fdslusb_driver);
#ifndef NDEBUG
	hallocated (&b, &n);
	if (n != 0) {
		log ("%u bytes (%u blocks) are still in use...\n", b, n);
	}
#endif
	lprintf (KERN_INFO, "Removed.\n");
} /* fdslusb_exit */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
module_init (fdslusb_init);
module_exit (fdslusb_exit);

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/

