/* 
 * driver.c
 * Copyright (C) 2002, AVM GmbH. All rights reserved.
 * 
 * This Software is  free software. You can redistribute and/or
 * modify such free software under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * The free software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this Software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA, or see
 * http://www.opensource.org/licenses/lgpl-license.html
 * 
 * Contact: AVM GmbH, Alt-Moabit 95, 10559 Berlin, Germany, email: info@avm.de
 */

#include <asm/atomic.h>
#include <asm/uaccess.h>
#include <linux/spinlock.h>
#include <linux/usb.h>
#include <linux/wait.h>
#include <linux/version.h>
#include <linux/interrupt.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>
#include <linux/sched.h>
#include <linux/ctype.h>
#include <linux/string.h>
#include <linux/capi.h>
#include <linux/kernelcapi.h>
#include <stdarg.h>
#include "ca.h"
#include "libdefs.h"
#include "capilli.h"
#include "tables.h"
#include "queue.h"
#include "defs.h"
#include "lib.h"
#include "main.h"
#include "driver.h"

#if defined (LOG_MESSAGES)
# define mlog		log
#else
# define mlog(f, a...)	
#endif

#define	__entry

#ifndef HZ
# error HZ is not defined...
#endif
#define	TEN_MSECS		(HZ / 100)
#define	STACK_DELAY		(HZ / 2)
#define USB_EVENT_HARD_ERROR    0x800
#define	USB_CTRL_TIMEOUT	3 * HZ
#define	USB_CFG_BLK_SIZE	777
#define	USB_BULK_TIMEOUT	3 * HZ
#define	USB_MAX_FW_LENGTH	262144

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,20)
#define	DEVREQUEST		devrequest
#define	DR_REQUEST_TYPE		requesttype
#define	DR_REQUEST		request
#define	DR_VALUE		value
#define	DR_INDEX		index
#define	DR_LENGTH		length
#else
#define	DEVREQUEST		struct usb_ctrlrequest
#define	DR_REQUEST_TYPE		bRequestType
#define	DR_REQUEST		bRequest
#define	DR_VALUE		wValue
#define	DR_INDEX		wIndex
#define	DR_LENGTH		wLength
#endif

// CLOSING TASK

static char *			invalid_msg	= "Invalid appl id #%d.\n";

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
dev_context_p			fdslusb_capi_ctx	= NULL;
lib_callback_p			fdslusb_capi_lib	= NULL;
struct capi_ctr *		fdslusb_capi_ctrl[2]	= { NULL, NULL };
static bundle_t			fdslusb_context[2];
static per_ctrl_t		fdslusb_params[2];
static int			started			= FALSE;
static int			nvers			= 0;
static struct tq_struct		tq_dpc;			/* for tq_immediate */
static struct tq_struct		tq_tx_dpc;		/* for tq_private */
static struct tq_struct		tq_rx_dpc;		/* for tq_private */
static atomic_t			scheduler_enabled	= ATOMIC_INIT (1);
static atomic_t			crit_count		= ATOMIC_INIT (0);
static volatile int		hard_error_issued;
static unsigned long		qtflags;
static spinlock_t		qtlock			= SPIN_LOCK_UNLOCKED;
static atomic_t			qtlockcount		= ATOMIC_INIT (0);
static int			thread_pid		= -1;
static atomic_t			thread_run_flag;
static atomic_t			thread_capi_run_flag;
int				card_config;

static DECLARE_TASK_QUEUE(tq_private);
static DECLARE_WAIT_QUEUE_HEAD(wait);
static DECLARE_WAIT_QUEUE_HEAD(capi_wait);
static DECLARE_MUTEX_LOCKED(hotplug);
static DECLARE_MUTEX_LOCKED(config);

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static atomic_t			resetting_ctrl		= ATOMIC_INIT (0);

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static struct tq_struct		tq_close;
static atomic_t			closing_task_running	= ATOMIC_INIT (0);
static void 		     (*	close_func1) (void *)	= NULL;
static void *			close_data1		= NULL;
static void 		     (*	close_func2) (void *)	= NULL;
static void *			close_data2		= NULL;

static DECLARE_MUTEX_LOCKED(notify);

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
typedef struct {

	struct urb *		urb;
	char *			buf;
	int			idx;
} ubi_t, * ubi_p;

static spinlock_t		ullock			= SPIN_LOCK_UNLOCKED;

static ubi_t			ubi[MAX_URB_COUNT + 1];
static int			ubi_free		= MAX_URB_COUNT;
static int			ubi_in			= MAX_URB_COUNT;
static int			ubi_out			= MAX_URB_COUNT;
static int			ubi_n			= -1;
#if !defined (NDEBUG)
static int			ubi_nmin		= MAX_URB_COUNT;
#endif

#define	URB_SUBMITTED		-1
#define	URB_UNLINKED		-2

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void			dpc (void *);
static void			tx_dpc (void *);
static void			rx_dpc (void *);
static void			enable_thread ();
static void 			disable_thread ();
static void			kick_scheduler (void);
static int			scheduler (void *);
void				usb_TxStart (void);
static void			usb_RxStart (void);

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void			scheduler_control (unsigned);
static void			wakeup_control (unsigned);
static void			version_callback (char *);
static unsigned			scheduler_suspend (unsigned long);
static unsigned			scheduler_resume (void);
static unsigned			controller_remove (void);
static unsigned			controller_add (void);

static functions_t		ca_funcs = {

	7,
	scheduler_control,
	wakeup_control,
	version_callback,
	scheduler_suspend,
	scheduler_resume,
	controller_remove,
	controller_add
} ;

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
struct capi_driver_interface *	fdslusb_drv_interface	= NULL;
struct capi_driver		fdslusb_capi_interface	= {

	TARGET,
	"",
	fdslusb_load_ware,
	fdslusb_reset_ctrl,
	fdslusb_remove_ctrl,
	fdslusb_register_appl,
	fdslusb_release_appl,
	fdslusb_send_msg,
	fdslusb_proc_info,
	fdslusb_ctr_info,
	fdslusb_drv_info,
	NULL,			/* add_card */
	NULL,
	NULL,
	0,
	NULL,
	""
} ;

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static struct capi_ctr * get_capi_ctr (int ctrl) {
	int	ix;
	
	assert (fdslusb_capi_ctx != NULL);
	assert (fdslusb_capi_ctx->ctrl2 > 0);
	ix = (fdslusb_capi_ctx->ctrl2 == ctrl) ? 1 : 0;
	info (NULL != fdslusb_capi_ctrl[ix]);
	return fdslusb_capi_ctrl[ix];
} /* get_capi_ctr */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void init_ctrl (void) {
	
	fdslusb_context[0].card = fdslusb_context[1].card = NULL;
	fdslusb_context[0].ctrl = fdslusb_context[1].ctrl = NULL;
	fdslusb_capi_ctrl[0] = fdslusb_capi_ctrl[1] = NULL;
} /* init_ctrl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int setup_ctrl (dev_context_p ctx, int ix) {
	int			cix;
	struct capi_ctr *	cptr;
	
	assert ((0 < ix) && (ix < 3));
	cptr = (*fdslusb_drv_interface->attach_ctr) (&fdslusb_capi_interface, SHORT_LOGO, NULL);
	if (NULL == cptr) {
		lprintf (KERN_INFO, "Error: Could not attach controller %d.\n", ix);
		return -EBUSY;
	}
	if ((1 == ix) && (cptr->cnr != 1)) {
		cix = ix;
	} else {
		cix = ix - 1;
	}
	log ("Controller %d (%d) --> %d\n", ix, cptr->cnr, cix);
	fdslusb_capi_ctrl[cix] = cptr;
	fdslusb_context[cix].card = ctx;
	fdslusb_context[cix].ctrl = &fdslusb_params[cix];
	fdslusb_capi_ctrl[cix]->driverdata = (void *) &fdslusb_context[cix];
	return 0;
} /* setup_ctrl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int nbchans (struct capi_ctr * ctrl) {
	per_ctrl_p	pcp;
	unsigned char *	prf;
	int		temp = 2;
    
	assert (ctrl);
	assert (ctrl->driverdata);
	pcp = GET_CTRL (ctrl);
	prf = (unsigned char *) pcp->string[6];
	if (prf != NULL) {
		temp = prf[2] + 256 * prf[3];
	}
	return temp;
} /* nbchans */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static struct sk_buff * make_0xfe_request (unsigned appl) {
	unsigned char	 request[8];
	struct sk_buff * skb;

	if (NULL == (skb = alloc_skb (8, GFP_ATOMIC))) {
		lprintf (KERN_ERR, "Unable to allocate message buffer.\n");
	} else {    
		request[0] = 8;		    request[1] = 0;
		request[2] = appl & 0xFF;   request[3] = (appl >> 8) & 0xFF;
		request[4] = 0xFE;	    request[5] = 0x80;
		memcpy (skb_put (skb, 8), &request, 8);
	}
	return skb;
} /* make_0xfe_request */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void reset_completion (struct urb * purb) {
	DEVREQUEST * req;

	assert (NULL != purb);
	assert (0 == purb->status);
	log ("Completed device reset.\n");
	req = (DEVREQUEST *) purb->context;
	hfree (req);
	usb_free_urb (purb);
} /* reset_completion */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int reset (struct usb_device * dev, void (* cfunc) (struct urb *)) {
	DEVREQUEST *	req;
	struct urb *	purb;
	int		res;

	if (NULL == (req = (DEVREQUEST *) hmalloc (sizeof (DEVREQUEST)))) {
		error ("Not enough memory for device request.\n");
		return -ENOMEM;
	}
	if (NULL == (purb = usb_alloc_urb (0))) {
		error ("Not enough memory for device request URB.\n");
		hfree (req);
		return -ENOMEM;
	}
	req->DR_REQUEST_TYPE = 0;
	req->DR_REQUEST      = USB_REQ_SET_CONFIGURATION;
	req->DR_VALUE        = 0;	/* Little endian required here! */ 
	req->DR_INDEX        = 0;	/* Little endian required here! */
	req->DR_LENGTH       = 0;	/* Little endian required here! */
	usb_fill_control_urb (
		purb,
		dev,
		usb_sndctrlpipe (dev, 0),
		(unsigned char *) req,
		NULL,
		0,
		(cfunc == NULL) ? reset_completion : cfunc,
		req
	);
	if (0 != (res = usb_submit_urb (purb))) {
		hfree (req);
	}
	return res;
} /* reset */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void set_config_completion (struct urb * purb) {
	DEVREQUEST * req;

	assert (NULL != purb);
	assert (0 == purb->status);
	log ("Completed SET_CONFIGURATION(0).\n");
	req = (DEVREQUEST *) purb->context;
	hfree (req);
	usb_free_urb (purb);
	up (&config);
} /* set_config_completion */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int select_config (struct usb_device * dev, int confix) {
	int	cfg, res;
	
	assert (0 == in_interrupt ());
	cfg = (confix == UNDEF_CONFIG) ? 0 : confix;
	if (cfg == card_config) {
		log ("select_config(%d) exited, already configured.\n", cfg);
		return 0;
	}
	log ("select_config(%d)\n", cfg);
	if (0 == cfg) {
		reset (dev, set_config_completion);
		down (&config);
		res = 0;	/* Well... nothing but an assumption! :-) */
	} else  {
		res = usb_set_configuration (dev, cfg);
	}
	if (0 == res) { 
		card_config = cfg;
	} else {
		log ("select_config(%d) returned %d.\n", cfg, res);
	}
	return res;
} /* select_config */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void scan_version (struct capi_ctr * ctrl, const char * ver) {
	int		vlen, i;
	char *		vstr;
	per_ctrl_p	pcp;

	assert (ctrl != NULL);
	assert (ctrl->driverdata != NULL);
	assert (ver != NULL);
	pcp = GET_CTRL (ctrl);

	vlen = (unsigned char) ver[0];
	pcp->version = vstr = (char *) hmalloc (vlen);
	if (NULL == pcp->version) {
		log ("Could not allocate version buffer.\n");
		return;
	}
	memcpy (pcp->version, ver + 1, vlen);
	for (i = 0; i < 8; i++) {
		pcp->string[i] = vstr + 1;
		vstr += 1 + *vstr;
	} 
#ifdef NDEBUG
	lprintf (KERN_INFO, "Stack version %s\n", pcp->string[0]);
#endif
	log ("Library version:    %s\n", pcp->string[0]);
	log ("Card type:          %s\n", pcp->string[1]);
	log ("Capabilities:       %s\n", pcp->string[4]);
	log ("D-channel protocol: %s\n", pcp->string[5]);
} /* scan_version */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void copy_version (struct capi_ctr * ctrl) {
	char *		tmp;
	per_ctrl_p	pcp;

	assert (ctrl != NULL);
	assert (ctrl->driverdata != NULL);
	pcp = GET_CTRL (ctrl);
	if (NULL == (tmp = pcp->string[3])) {
		lprintf (KERN_ERR, "Do not have version information...\n");
		return;
	}
	strncpy (ctrl->serial, tmp, CAPI_SERIAL_LEN);
	memcpy (&ctrl->profile, pcp->string[6], sizeof (capi_profile));
	strncpy (ctrl->manu, "AVM GmbH", CAPI_MANUFACTURER_LEN);
	ctrl->version.majorversion = 2;
	ctrl->version.minorversion = 0;
	tmp = pcp->string[0];
	ctrl->version.majormanuversion = (((tmp[0] - '0') & 15) << 4)
					+ ((tmp[2] - '0') & 15);
	ctrl->version.minormanuversion = ((tmp[3] - '0') << 4)
					+ (tmp[5] - '0') * 10
					+ ((tmp[6] - '0') & 15);
} /* copy_version */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void kill_version (struct capi_ctr * ctrl) {
	int		i;
	per_ctrl_p	pcp;

	assert (ctrl != NULL);
	assert (ctrl->driverdata != NULL);
	pcp = GET_CTRL (ctrl);
	for (i = 0; i < 8; i++) {
		pcp->string[i] = NULL;
	}
	if (pcp->version != NULL) {
		hfree (pcp->version);
		pcp->version = NULL;
	}
} /* kill_version */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void pprintf (char * page, int * len, const char * fmt, ...) {
	va_list args;

	va_start (args, fmt);
	*len += vsprintf (page + *len, fmt, args);
	va_end (args);
} /* pprintf */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void stop (dev_context_p card) {

	assert (started == TRUE);
	(*fdslusb_capi_lib->cm_exit) ();
	if (-1 != thread_pid) {
		disable_thread ();
	}
	queue_exit (&card->queue);
	table_exit (&card->appls);
	if (fdslusb_context[0].ctrl != NULL) {
		kill_version (fdslusb_capi_ctrl[0]);
	}
	kill_version (fdslusb_capi_ctrl[1]);
	nvers = 0;
	started = FALSE;
} /* stop */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int start (dev_context_p card) {

	card->count = 0;
	table_init (&card->appls);
	queue_init (&card->queue);
	enable_thread ();
	(*fdslusb_capi_lib->cm_register_ca_functions) (&ca_funcs);
	if ((*fdslusb_capi_lib->cm_start) ()) {
		log ("Starting the stack failed...\n");
		return FALSE;
	}
	(void) (*fdslusb_capi_lib->cm_init) (0, 0);
	assert (nvers == 2);
	enter_critical ();
	if ((*fdslusb_capi_lib->cm_activate) ()) {
		log ("Activate failed.\n");
		leave_critical ();
		stop (card);
		return FALSE;
	}
	leave_critical ();
	return (started = TRUE); 
} /* start */

/*-C-------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void appl_released_notify (int appl) {

	UNUSED_ARG (appl);
	if (atomic_read (&closing_task_running)) {
		log ("Note: appl %d released.\n", appl);
		up (&notify);
	}
} /* appl_released_notify */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void closing_task (void * ctx) {
	appl_t *	appp;
	dev_context_p	card = ctx;

	DECLARE_WAIT_QUEUE_HEAD(close_wait);

#define CALLBACK(ix)	if (close_func##ix != NULL) { \
				log ("Callback[%p]\n", close_func##ix); \
				(* (close_func##ix)) (close_data##ix); \
				close_func##ix = NULL; \
			}

	log ("Closing task started.\n");
	assert (ctx != NULL);
	atomic_set (&closing_task_running, 1);
	atomic_set (&resetting_ctrl, 1);
	if (0 != card->count) {
		assert (card->appls != NULL);
		if (card->appls != NULL) {
			appp = first_appl (card->appls);
			while (appp != NULL) {
				log ("Releasing appl #%d...\n", appp->id);
				fdslusb_release_appl (
					fdslusb_capi_ctrl[1],
					appp->id
				);
				kick_scheduler ();
				down (&notify);
				appp = next_appl (card->appls, appp);
			}
		}
	}
	if (atomic_read (&thread_capi_run_flag)) {
		init_waitqueue_head (&close_wait);
		kick_scheduler ();
		log ("Waiting...\n");
		interruptible_sleep_on_timeout (&close_wait, STACK_DELAY);
		log ("... completed!\n");
	}
	if (started) {
		stop (card);
	}
	atomic_set (&resetting_ctrl, 0);
	fdslusb_reset_ctrl (fdslusb_capi_ctrl[1]);
	fdslusb_remove_ctrl (fdslusb_capi_ctrl[1]);
	CALLBACK (1);
	CALLBACK (2);
	atomic_set (&closing_task_running, 0);
	log ("Closing task terminated.\n");

#undef CALLBACK

} /* closing_task */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void init_closing_task (void) {

	atomic_set (&closing_task_running, 0);
} /* init_closing_task */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void start_closing_task (void (* func) (void *), void * data) {
	unsigned long	flags;
	spinlock_t	close_lock	= SPIN_LOCK_UNLOCKED;

	spin_lock_irqsave (&close_lock, flags);
	assert ((close_func1 == NULL) || (close_func2 == NULL));
	if (close_func1 != NULL) {
		assert (close_func2 != NULL);
		close_func2 = func;
		close_data2 = data;
	} else {
		close_func1 = func;
		close_data1 = data;
	}
	spin_unlock_irqrestore (&close_lock, flags);
	if (atomic_read (&closing_task_running)) {
		log ("Closing task already running.\n");
		return;
	}
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	tq_close.next = NULL;
#else
	tq_close.list.next = NULL;
#endif
	tq_close.sync = 0;
	tq_close.data = fdslusb_capi_ctx;
	tq_close.routine = &closing_task;
	schedule_task (&tq_close);
	log ("Closing task scheduled.\n");
} /* start_closing_task */

/*---------------------------------------------------------------------------*\
\*-L-------------------------------------------------------------------------*/
void lock (void) {
	unsigned long	local_flags;
	
	if (!atomic_inc_and_test (&qtlockcount)) {
		spin_lock_irqsave (&qtlock, local_flags);
		qtflags = local_flags;
	}
} /* lock */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void unlock (void) {
	unsigned long	local_flags = qtflags;

	if (!atomic_dec_and_test (&qtlockcount)) {
		spin_unlock_irqrestore (&qtlock, local_flags);
	}
} /* unlock */

/*---------------------------------------------------------------------------*\
\*-C-------------------------------------------------------------------------*/ 
static void kick_queue (struct tq_struct * task) {

	if (NULL != task) {
		queue_task (task, &tq_private);
	}
	queue_task (&tq_dpc, &tq_immediate);
	mark_bh (IMMEDIATE_BH);
} /* kick_queue */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
void enter_critical (void) {

	atomic_inc (&crit_count);
} /* enter_critical */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static inline int in_critical (void) {

	return (0 < atomic_read (&crit_count));
} /* in_critical */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
void leave_critical (void) {
        static spinlock_t	lock = SPIN_LOCK_UNLOCKED;
	unsigned long		flags;

	assert (in_critical ());
	spin_lock_irqsave (&lock, flags);
	if (1 == atomic_read (&crit_count)) {
		/* About to leave critical area... */
		kick_queue (NULL);
	}
	atomic_dec (&crit_count);
	spin_unlock_irqrestore (&lock, flags);
} /* leave_critical */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int fdslusb_kcapi_init (void) {
	int	res;
	
	if (NULL != fdslusb_capi_ctrl[1]) {
		error ("Cannot handle two controllers.\n");
 		return -EBUSY;
	}
	init_ctrl ();
	if (0 != (res = setup_ctrl (fdslusb_capi_ctx, 1))) {
		return res;
	}
	if (fdslusb_context[0].ctrl != NULL) {
		if (0 != (res = setup_ctrl (fdslusb_capi_ctx, 2))) {
			(* fdslusb_drv_interface->detach_ctr) (fdslusb_capi_ctrl[0]);
			init_ctrl ();
			return res;
		}
	}
	fdslusb_capi_ctx->ctrl2 = fdslusb_capi_ctrl[1]->cnr;
	fdslusb_capi_ctx->ctxp = NULL;
	return 0;
} /* fdslusb_kcapi_init */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int fdslusb_stack_init (void) {

	if (!start (fdslusb_capi_ctx)) {
		error ("Initialization failed.\n");
		return -EIO;
	}
	if (fdslusb_capi_ctrl[0] != NULL) {
		assert (fdslusb_capi_ctrl[0]->ready != NULL);
		(* fdslusb_capi_ctrl[0]->ready) (fdslusb_capi_ctrl[0]);
	}
	assert (fdslusb_capi_ctrl[1]->ready != NULL);
	(*fdslusb_capi_ctrl[1]->ready) (fdslusb_capi_ctrl[1]);
	return 0;
} /* fdslusb_stack_init */

/*---------------------------------------------------------------------------*\
\*-F-------------------------------------------------------------------------*/
static int emit (dev_context_p ctx, char * ptr, int len, int * olen) {
	int	res;

	assert (ctx->dev != NULL);
	assert (ctx->epwrite != NULL);
	assert ((ptr != NULL) || (len == 0));
	assert (olen != NULL);
	res = usb_bulk_msg (
		ctx->dev,
		usb_sndbulkpipe (ctx->dev, ctx->epwrite->bEndpointAddress),
		ptr,
		len,
		olen,
		USB_BULK_TIMEOUT
		);
	if (0 != res) {
		error ("USB I/O error, code %d.\n", res);
	}
	return res;
} /* emit */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int __entry fdslusb_load_ware (struct capi_ctr * ctrl, capiloaddata * ware) {
	dev_context_p	ctx;
	int		len, todo, olen;
	char *		buf;
	char *		src;
	char *		ptr;
	int		res	= 13;
#if !defined (NDEBUG)
	unsigned	count	= 0;
#endif

	assert (ware != NULL);
	assert (ctrl != NULL);
	assert (ctrl->driverdata != NULL);
	ctx = GET_CARD (ctrl);
	info (card_config == LOADING_CONFIG);
	if (LOADING_CONFIG != card_config) {
		res = select_config (ctx->dev, LOADING_CONFIG);
		if (0 != res) {
			error ("Could not set load config.\n");
			return -EIO;
		}
	}
	if ((todo = ware->firmware.len) > USB_MAX_FW_LENGTH) {
		error ("Invalid firmware file.\n");
		return -EIO;
	}
	if (NULL == (ptr = buf = (char *) hmalloc (USB_CFG_BLK_SIZE))) {
		error ("Cannot build firmware buffer.\n");
		return -ENOMEM;
	}
	src = ware->firmware.data;
	while (todo > 0) {
		len = (todo > USB_CFG_BLK_SIZE) ? USB_CFG_BLK_SIZE : todo;
		if (ware->firmware.user) {
			if (copy_from_user (buf, src, len)) {
				error ("Error copying firmware data.\n");
				break;
			}
		} else {
			memcpy (buf, src, len);
		}
		if (0 != (res = emit (ctx, buf, len, &olen))) {
			log (
				"Error in %d-byte block #%d.\n",
				USB_CFG_BLK_SIZE,
				count
			);
			break;
		}
		assert (len == olen);
		src  += olen;
		todo -= olen;
#if !defined (NDEBUG)
		++count;
#endif
	}
	hfree (buf);
	if ((0 == todo) && (0 == res)) {
		res = emit (ctx, NULL, 0, &olen);
	} else {
		return -EIO;
	}
	res = select_config (ctx->dev, RESET_CONFIG);
	if (0 == res) {
		res = select_config (ctx->dev, RUNNING_CONFIG);
	}
	if (0 == res) {
		return fdslusb_stack_init ();
	}
	error ("Firmware does not respond.\n");
	assert (ctx->ctr != NULL);
	assert (ctx->ctr->reseted != NULL);
	(*ctx->ctr->reseted) (ctx->ctr);
	return -EBUSY;
} /* fdslusb_load_ware */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int __entry fdslusb_add_card (struct capi_driver * drv, capicardparams * args) {
	
	UNUSED_ARG (drv);
	UNUSED_ARG (args);
	init_waitqueue_head (&wait);
	atomic_set (&resetting_ctrl, 0);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	tq_dpc.next = 
	tq_tx_dpc.next = 
	tq_rx_dpc.next = NULL;
#else
	tq_dpc.list.next = 
	tq_tx_dpc.list.next = 
	tq_rx_dpc.list.next = NULL;
#endif
	tq_dpc.sync = 
	tq_tx_dpc.sync =
	tq_rx_dpc.sync = 0;
	tq_dpc.data = 
	tq_tx_dpc.data =
	tq_rx_dpc.data = fdslusb_capi_ctx;
	tq_dpc.routine = &dpc;
	tq_tx_dpc.routine = &tx_dpc;
	tq_rx_dpc.routine = &rx_dpc;
						
	return fdslusb_kcapi_init ();
} /* fdslusb_add_card */ 

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int remove_appls (dev_context_p card) {
	appl_t *	appp;
	int		n = 0;
	
	if (0 != card->count) {
		log ("Removing registered applications.\n");
		assert (card->appls != NULL);
		if (card->appls != NULL) {
			appp = first_appl (card->appls);
			while (appp != NULL) {
				++n;
				fdslusb_release_appl (
					fdslusb_capi_ctrl[1],
					appp->id
				);
				appp = next_appl (card->appls, appp);
			}
		}
		kick_scheduler ();
	} 
	return n;
} /* remove_appls */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry fdslusb_reset_ctrl (struct capi_ctr * ctrl) {

	DECLARE_WAIT_QUEUE_HEAD(appl_wait);
	
	UNUSED_ARG (ctrl);
	atomic_set (&resetting_ctrl, 1);
	if (remove_appls (fdslusb_capi_ctx)) {
		init_waitqueue_head (&appl_wait);
		interruptible_sleep_on_timeout (&appl_wait, STACK_DELAY);
	}
	if (started) {
		stop (fdslusb_capi_ctx);
	}
	if (fdslusb_capi_ctrl[0] != NULL) {
		assert (fdslusb_capi_ctrl[0]->reseted != NULL);
		(* fdslusb_capi_ctrl[0]->reseted) (fdslusb_capi_ctrl[0]);
	}
	assert (fdslusb_capi_ctrl[1]->reseted != NULL);
	(*fdslusb_capi_ctrl[1]->reseted) (fdslusb_capi_ctrl[1]);
	atomic_set (&resetting_ctrl, 0);
} /* fdslusb_reset_ctrl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry fdslusb_remove_ctrl (struct capi_ctr * ctrl) {

	UNUSED_ARG (ctrl);
	assert (fdslusb_capi_ctx != NULL);
	assert (fdslusb_capi_ctrl[1] != NULL);
	enter_critical ();
	usb_unlink_urb (fdslusb_capi_ctx->tx_urb);
	unlink_urbs (fdslusb_capi_ctx);
	leave_critical ();
	if (fdslusb_context[0].ctrl != NULL) {
		(* fdslusb_drv_interface->detach_ctr) (fdslusb_capi_ctrl[0]);
	}
	(* fdslusb_drv_interface->detach_ctr) (fdslusb_capi_ctrl[1]);
	init_ctrl ();
} /* fdslusb_remove_ctrl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry fdslusb_register_appl (
	struct capi_ctr *	ctrl, 
	u16			appl, 
	capi_register_params *	args
) {
	dev_context_p		card;
	appl_t *		appp;
	void *			ptr;
	unsigned		nc;

	mlog ("REGISTER(appl:%u)\n", appl);
	assert (ctrl != NULL);
	assert (ctrl->driverdata != NULL);
	assert (args);
	card = GET_CARD (ctrl); 
	if ((int) args->level3cnt < 0) {
		nc = nbchans (ctrl) * -((int) args->level3cnt);
	} else {
		nc = args->level3cnt;
	}
	if (0 == nc) {
		nc = nbchans (ctrl);
	}
	if (ctrl->cnr == card->ctrl2) {
		appp = create_appl (
				card->appls, 
				appl, 
				nc, 
				args->datablkcnt, 
				args->datablklen
				);
		if (NULL == appp) {
			log ("Unable to create application record.\n");
			return;
		}
		ptr = hcalloc (card->length);
		if (NULL == ptr) {
			error ("Not enough memory for application data.\n");
			remove_appl (card->appls, appp);
			return;
		} else {
			card->count++;
			appp->data = ptr;
			appp->ctrl = ctrl;
			(*card->reg_func) (ptr, appl);
		}
	}
	assert (ctrl->appl_registered != NULL);
	(*ctrl->appl_registered) (ctrl, appl);
} /* fdslusb_register_appl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry fdslusb_release_appl (struct capi_ctr * ctrl, u16 appl) {
	dev_context_p		card;
	struct sk_buff *	skb;
	appl_t *		appp;

	mlog ("RELEASE(appl:%u)\n", appl);
	assert (ctrl != NULL);
	assert (ctrl->driverdata != NULL);
	card = GET_CARD (ctrl);
	if (ctrl->cnr == card->ctrl2) {
		appp = search_appl (card->appls, appl);
		if (NULL == appp) {
			log ("Attempt to release unknown application (id #%u)\n", appl);
			return;
		}
		skb = make_0xfe_request (appl);
		handle_message (card->appls, appp, skb);
		kick_scheduler ();
		appp->dying = TRUE;
	}
} /* fdslusb_release_appl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry fdslusb_send_msg (struct capi_ctr * ctrl, struct sk_buff * skb) {
	dev_context_p	card;
	unsigned char *	byte;
	unsigned	appl;
	appl_t *	appp;

	assert (ctrl != NULL);
	assert (ctrl->driverdata != NULL);
	assert (skb);
	card = GET_CARD (ctrl);
	byte = skb->data;
	appl = byte[2] + 256 * byte[3];
	mlog (
		"MESSAGE(appl:%u,cmd:%02x,subcmd:%02x)\n", 
		appl,
		byte[4], 
		byte[5]
	);
	appp = search_appl (card->appls, appl);
	if ((NULL == appp) || appp->dying) {
		log ("Message for unknown application (id %u).\n", appl);
		return;
	}
	handle_message (card->appls, appp, skb);
	kick_scheduler ();
} /* fdslusb_send_msg */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
char * __entry fdslusb_proc_info (struct capi_ctr * ctrl) {
	dev_context_p	card;
	per_ctrl_p	pcp;
	static char	text[80];

	assert (ctrl != NULL);
	assert (ctrl->driverdata != NULL);
	card = GET_CARD (ctrl);
	pcp = GET_CTRL (ctrl);
	SNPRINTF (
		text, 
		sizeof (text),
		"%s %s %d",
		pcp->version ? pcp->string[1] : "A1",
		pcp->version ? pcp->string[0] : "-",
		card->dev->devnum
	);
	return text;
} /* fdslusb_proc_info */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int __entry fdslusb_ctr_info (
	char *			page, 
	char **			start, 
	off_t			ofs,
	int			count, 
	int *			eof, 
	struct capi_ctr *	ctrl
) {
	dev_context_p		card;
	per_ctrl_p		pcp;
	char *			temp;
	unsigned char		flag;
	int			len = 0;

	assert (ctrl != NULL);
	assert (ctrl->driverdata != NULL);
	card = GET_CARD (ctrl);
	pcp = GET_CTRL (ctrl);
	pprintf (page, &len, "%-16s %s\n", "name", SHORT_LOGO);
	pprintf (page, &len, "%-16s %d\n", "dev", card->dev->devnum);
	temp = pcp->version ? pcp->string[1] : "A1";
	pprintf (page, &len, "%-16s %s\n", "type", temp);
	temp = pcp->version ? pcp->string[0] : "-";
	pprintf (page, &len, "%-16s %s\n", "ver_driver", temp);
	pprintf (page, &len, "%-16s %s\n", "ver_cardtype", SHORT_LOGO);

	flag = ((unsigned char *) (ctrl->profile.manu))[3];
	if (flag) {
		pprintf (page, &len, "%-16s%s%s%s%s%s%s%s\n", "protocol",
			(flag & 0x01) ? " DSS1" : "",
			(flag & 0x02) ? " CT1" : "",
			(flag & 0x04) ? " VN3" : "",
			(flag & 0x08) ? " NI1" : "",
			(flag & 0x10) ? " AUSTEL" : "",
			(flag & 0x20) ? " ESS" : "",
			(flag & 0x40) ? " 1TR6" : ""
		);
	}
	flag = ((unsigned char *) (ctrl->profile.manu))[5];
	if (flag) {
		pprintf (page, &len, "%-16s%s%s%s%s\n", "linetype",
			(flag & 0x01) ? " point to point" : "",
			(flag & 0x02) ? " point to multipoint" : "",
			(flag & 0x08) ? " leased line without D-channel" : "",
			(flag & 0x04) ? " leased line with D-channel" : ""
		);
	}
	if (len < ofs) {
		return 0;
	}
	*eof = 1;
	*start = page - ofs;
	return ((count < len - ofs) ? count : len - ofs);
} /* fdslusb_ctr_info */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int __entry fdslusb_drv_info (
	char *			page, 
	char **			start, 
	off_t			ofs, 
	int			count, 
	int *			eof, 
	struct capi_driver *	drv
) {
	int			len = 0;

	UNUSED_ARG (drv);
	pprintf (page, &len, "%-16s %s\n", "name", TARGET);
	pprintf (page, &len, "%-16s %s\n", "revision", fdslusb_capi_interface.revision);
	if (len < ofs) {
		return 0;
	}
	*eof = 1;
	*start = page - ofs;
	return ((count < len - ofs) ? count : len - ofs);
} /* fdslusb_drv_info */
 
/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void * data_by_id (unsigned appl_id) {
	appl_t * appp;

	appp = search_appl (fdslusb_capi_ctx->appls, appl_id);
	return (appp != NULL) ? appp->data : NULL;
} /* data_by_id */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
struct capi_ctr * card_by_id (unsigned appl_id) {
	appl_t * appp;

	appp = search_appl (fdslusb_capi_ctx->appls, appl_id);
	return (appp != NULL) ? appp->ctrl : NULL;
} /* card_by_id */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void * first_data (int * res) {
	appl_t * appp;

	assert (res);
	appp = first_appl (fdslusb_capi_ctx->appls);
	*res = (appp != NULL) ? 0 : -1;
	return (appp != NULL) ? appp->data  : NULL;
} /* first_data */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void * next_data (int * res) {
	appl_t * appp;

	assert (res);
	if (NULL != (appp = get_appl (fdslusb_capi_ctx->appls, *res))) {
		appp = next_appl (fdslusb_capi_ctx->appls, appp);
	}
	*res = (appp != NULL) ? 1 + *res : -1;
	return (appp != NULL) ? appp->data  : NULL;
} /* next_data */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int appl_profile (unsigned appl_id, unsigned * bs, unsigned * bc) {
	appl_t * appp;

	appp = search_appl (fdslusb_capi_ctx->appls, appl_id);
	if (NULL == appp) {
		return 0;
	}
	if (bs) *bs = appp->blk_size;
	if (bc) *bc = appp->blk_count;
	return 1;
} /* appl_profile */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int msg2stack (unsigned char * msg) {
	unsigned		mlen;
	unsigned		appl;
	__u32			ncci;
	unsigned		hand;
	unsigned char *		mptr;
	unsigned char *		dptr;
	struct sk_buff *	skb;
	unsigned		temp;
	int			res = 0;

	assert (msg);
	if (!queue_is_empty (fdslusb_capi_ctx->queue)) {
		skb = queue_peek (fdslusb_capi_ctx->queue);
		res = 1;
		assert (skb);
		mptr = (unsigned char *) skb->data;
		mlen = mptr[0] + 256 * mptr[1]; 
		appl = mptr[2] + 256 * mptr[3];
		mlog (
			"PUT_MESSAGE(appl:%u,cmd:%02X,subcmd:%02X)\n", 
			appl, 
			mptr[4], 
			mptr[5]
		);
		if ((0x86 == mptr[4]) && (0x80 == mptr[5])) { /* DATA_B3_REQ */
			ncci = mptr[8] + 256 * (mptr[9] + 256 * 
						(mptr[10] + 256 * mptr[11]));
			hand = mptr[18] + 256 * mptr[19];
			temp = (unsigned) (mptr + mlen);
			dptr = mptr + 12;
			*dptr++ = temp & 0xFF;	temp >>= 8;
			*dptr++ = temp & 0xFF;	temp >>= 8;
			*dptr++ = temp & 0xFF;	temp >>= 8;
			*dptr++ = temp & 0xFF;
			queue_park (fdslusb_capi_ctx->queue, appl, ncci, hand);
			memcpy (msg, mptr, mlen);
		} else {
			queue_drop (fdslusb_capi_ctx->queue);
			memcpy (msg, mptr, mlen);
		}
	} 
	return res;
} /* msg2stack */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void msg2capi (unsigned char * msg) {
	unsigned		mlen; 
	unsigned		appl;
	unsigned		nctr;
	__u32			ncci;
	unsigned		hand;
	unsigned		dlen;
	unsigned char *		dptr;
	unsigned		temp;
	struct sk_buff *	skb;
	struct capi_ctr *	ctrl;

	assert (msg);
	mlen = msg[0] + 256 * msg[1];
	appl = msg[2] + 256 * msg[3];
	nctr = msg[8] & 127;
	mlog (
		"GET_MESSAGE(ctrl:%u,appl:%u,cmd:%02X,subcmd:%02X)\n", 
		nctr,
		appl, 
		msg[4], 
		msg[5]
	);
	if ((0x86 == msg[4]) && (0x81 == msg[5])) { /* DATA_B3_CONF */
		hand = msg[12] + 256 * msg[13];
		ncci = msg[8] + 256 * (msg[9] + 256 * 
						(msg[10] + 256 * msg[11]));
		queue_conf (fdslusb_capi_ctx->queue, appl, ncci, hand);
	}
	if ((0x86 == msg[4]) && (0x82 == msg[5])) { /* DATA_B3_IND */
		dlen = msg[16] + 256 * msg[17];
		skb = alloc_skb (
			mlen + dlen + ((mlen < 30) ? (30 - mlen) : 0), 
			GFP_ATOMIC
			);
		if (NULL == skb) {
			error ("Unable to build CAPI message skb." 
							" Message lost.\n");
			return;
		}
		/* Messages are expected to come with 32 bit data pointers. 
		 * The kernel CAPI works with extended (64 bit ready) message 
		 * formats so that the incoming message needs to be fixed, 
		 * i.e. the length gets adjusted and the required 64 bit data 
		 * pointer is added.
		 */
		temp = msg[12] + 256 * (msg[13] + 256 * 
						(msg[14] + 256 * msg[15]));
		dptr = (unsigned char *) temp;
		if (mlen < 30) {
			msg[0] = 30;
			ncci   = 0;			
			memcpy (skb_put (skb, mlen), msg, mlen);
			memcpy (skb_put (skb, 4), &ncci, 4);
			memcpy (skb_put (skb, 4), &ncci, 4);
		} else {
			memcpy (skb_put (skb, mlen), msg, mlen);
		}
		memcpy (skb_put (skb, dlen), dptr, dlen); 
	} else {
		if (NULL == (skb = alloc_skb (mlen, GFP_ATOMIC))) {
			error ("Unable to build CAPI message skb." 
							" Message lost.\n");
			return;
		}
		memcpy (skb_put (skb, mlen), msg, mlen);
	}
	ctrl = get_capi_ctr (nctr);
	info (ctrl != NULL);
	if (ctrl != NULL) {
		assert (ctrl->cnr == (int) nctr);
		assert (ctrl->handle_capimsg != NULL);
		(*ctrl->handle_capimsg) (ctrl, appl, skb);
	}
} /* msg2capi */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void new_ncci (
	unsigned		appl_id, 
	__u32			ncci,
	unsigned		winsize, 
	unsigned		blksize
) {
	appl_t *		appp;
	ncci_t *		nccip;
	struct capi_ctr *	ctrl;
	unsigned		nctr;

	nctr = ncci & 127;
	mlog ("NEW NCCI(ctrl:%u,appl:%u,ncci:0x%08X)\n", nctr, appl_id, ncci);
	appp = search_appl (fdslusb_capi_ctx->appls, appl_id);
	if (NULL == appp) {
		lprintf (KERN_ERR, invalid_msg, appl_id);
		return;
	}
	nccip = create_ncci (
			fdslusb_capi_ctx->appls, 
			appp, 
			(NCCI_t) ncci, 
			winsize, 
			blksize
			);
	if (NULL == nccip) {
		log ("Cannot handle new NCCI...\n");
		return;
	}
	ctrl = get_capi_ctr (nctr);
	assert (ctrl != NULL);
	assert (ctrl->cnr == (int) nctr);
	assert (ctrl->new_ncci != NULL);
	(*ctrl->new_ncci) (ctrl, appl_id, ncci, winsize);
} /* new_ncci */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void free_ncci (unsigned appl_id, __u32 ncci) {
	appl_t *		appp;
	ncci_t *		nccip;
	struct capi_ctr *	ctrl;
	unsigned		nctr;

	appp = search_appl (fdslusb_capi_ctx->appls, appl_id);
	if (NULL == appp) {
		lprintf (KERN_ERR, invalid_msg, appl_id);
		return;
	}
	assert (appp->ctrl);
	if (0xFFFFFFFF == ncci) {			/* 2nd phase RELEASE */
		mlog ("FREE APPL(appl:%u)\n", appl_id);
		assert (appp->dying == TRUE);
		fdslusb_capi_ctx->count--;
		(*fdslusb_capi_ctx->rel_func) (appp->data);
		remove_appl (fdslusb_capi_ctx->appls, appp);
		appl_released_notify (appl_id);
		if (atomic_read (&resetting_ctrl)) {
			return;
		}
		if (fdslusb_context[0].ctrl != NULL) {
			assert (fdslusb_capi_ctrl[0]->appl_released != NULL);
			(*fdslusb_capi_ctrl[0]->appl_released) (
				fdslusb_capi_ctrl[0], 
				appl_id
			);
		}
		assert (fdslusb_capi_ctrl[1]->appl_released != NULL);
		(*fdslusb_capi_ctrl[1]->appl_released) (
			fdslusb_capi_ctrl[1], 
			appl_id
		);
	} else if (NULL != (nccip = locate_ncci (appp, ncci))) {
		nctr = ncci & 127;
		ctrl = get_capi_ctr (nctr);
		mlog ("FREE NCCI(ctrl:%u,appl:%u,ncci:0x%08X)\n", nctr, appl_id, ncci);	
		assert (ctrl != NULL);
		assert (ctrl->cnr == (int) nctr);
		assert (ctrl->free_ncci != NULL);
		(*ctrl->free_ncci) (ctrl, appl_id, ncci);
		remove_ncci (fdslusb_capi_ctx->appls, appp, nccip);
	} else {
		lprintf (KERN_ERR, "Attempt to free unknown NCCI.\n");
	}
} /* free_ncci */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
unsigned char * data_block (unsigned appl_id, __u32 ncci, unsigned handle) {
	appl_t *	appp;
	char *		buf;
 
	appp = search_appl (fdslusb_capi_ctx->appls, appl_id);
	if (NULL == appp) {
		lprintf (KERN_ERR, invalid_msg, appl_id);
		return NULL;
	}
	buf = ncci_data_buffer (fdslusb_capi_ctx->appls, appp, ncci, handle);
	return buf;
} /* data_block */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
void init (
	unsigned len, 
	void  (* reg) (void *, unsigned),
	void  (* rel) (void *),
	void  (* dwn) (void)
) {
	assert (reg);
	assert (rel);
	assert (dwn);

	fdslusb_capi_ctx->length   = len;
	fdslusb_capi_ctx->reg_func = reg;
	fdslusb_capi_ctx->rel_func = rel;
	fdslusb_capi_ctx->dwn_func = dwn;
} /* init */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void wakeup_control (unsigned enable_flag) { 
	
	enable_flag = 0; 
} /* wakeup_control */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void version_callback (char * vp) { 
	
	assert (nvers < 2);
	if ((0 == nvers) && (NULL == fdslusb_context[0].ctrl)) {
		++nvers;
		log ("Version string #0 rejected.\n");
		return;
	}
	log ("Version string #%d:\n", nvers);
	scan_version (fdslusb_capi_ctrl[nvers], vp);
	copy_version (fdslusb_capi_ctrl[nvers]);
	++nvers;
} /* version_callback */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static unsigned	scheduler_suspend (unsigned long timeout) { 
	
	timeout = 0; 
	return 0; 
} /* scheduler_suspend */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static unsigned	scheduler_resume (void) { 
	
	return 0; 
} /* scheduler_resume */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static unsigned	controller_remove (void) { 

	assert (0);
	return 0; 
} /* controller_remove */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static unsigned	controller_add (void) { 
	
	assert (0);
	return 0; 
} /* controller_add */
	
/*---------------------------------------------------------------------------*\
\*-M-------------------------------------------------------------------------*/
static void clear_ubi (dev_context_p pdc) {
	int	ix;

	UNUSED_ARG (pdc);
	for (ix = 0; ix <= MAX_URB_COUNT; ix++) {
		ubi[ix].buf = NULL;
		ubi[ix].urb = NULL;
		ubi[ix].idx = MAX_URB_COUNT;
	}
	ubi_n    = -1;
#if !defined (NDEBUG)
	ubi_nmin =
#endif
	ubi_free = 
	ubi_in   = 
	ubi_out  = MAX_URB_COUNT;
} /* clear_ubi */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int init_urbs (dev_context_p pdc) {
	int	ix = MAX_URB_COUNT - 1;

	assert (!atomic_read (&pdc->is_open));
	do {	
		ubi[ix].idx = 1 + ix;
		if (NULL == (ubi[ix].buf = hmalloc (MAX_TRANSFER_SIZE))) {
			log (
				"Could not alloc all %d buffers.\n", 
				MAX_URB_COUNT
			);
			++ix;
	free_buffers:
			while (ix < MAX_URB_COUNT) {
				hfree (ubi[ix++].buf);
			}
			clear_ubi (pdc);
			return 0;
		}
	} while (--ix >= 0);
	
	do {
		++ix;
		if (NULL == (ubi[ix].urb = usb_alloc_urb (0))) {
			log (
				"Could not alloc all %d URBs.\n", 
				MAX_URB_COUNT
			);
			--ix;
			while (ix >= 0) {
				usb_free_urb (ubi[ix--].urb);
			}
			ix = 0;
			goto free_buffers;
		}
		ubi[ix].urb->context = (void *) ix;
	} while (ix < MAX_URB_COUNT);
	assert (ix == MAX_URB_COUNT);
	ubi[ix].urb = NULL;
	ubi[ix].buf = NULL;
	ubi[ix].idx = MAX_URB_COUNT;
	
	log ("Allocated %d URBs/buffers.\n", ix);
	ubi_free = 0;
#if !defined (NDEBUG)
	ubi_nmin =
#endif
	ubi_n    =
	ubi_in   = 
	ubi_out  = MAX_URB_COUNT;
	return 1;
} /* init_urbs */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void free_urbs (dev_context_p pdc) {
	int	ix;
	
	assert (!atomic_read (&pdc->is_open));
	info (-1 != ubi_n);
	if (-1 == ubi_n) 
		return;
	for (ix = 0; ix < MAX_URB_COUNT; ix++) {
		assert (ubi[ix].buf != NULL);
		assert (ubi[ix].urb != NULL);
		hfree (ubi[ix].buf);		
		usb_free_urb (ubi[ix].urb);	
	}
	log ("Freed %d URBs/buffers.\n", ix);
	clear_ubi (pdc);
} /* free_urbs */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void unlink_urbs (dev_context_p pdc) {
	int	ix;

	UNUSED_ARG (pdc);
	info (-1 != ubi_n);
	if (-1 == ubi_n) 
		return;
	for (ix = 0; ix < MAX_URB_COUNT; ix++) {
		if (ubi[ix].idx == URB_SUBMITTED) {
			usb_unlink_urb (ubi[ix].urb);
			ubi[ix].idx = URB_UNLINKED;
		}
	}
} /* unlink_urbs */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int claim_urb (
	dev_context_p	pdc, 
	struct urb **	ppurb, 
	char **		ppbuf
) {
	int		ix;
	unsigned long	flags;
	
	UNUSED_ARG (pdc);
	assert (ppurb != NULL);
	assert (ppbuf != NULL);
	spin_lock_irqsave (&ullock, flags);
	ix = ubi_free;
	assert (ix >= 0);
	assert (ix <= MAX_URB_COUNT);
	ubi_free = ubi[ix].idx;
	if (ix != MAX_URB_COUNT) {
		ubi[ix].idx = URB_SUBMITTED;	/* in spe */
		--ubi_n;
#if !defined (NDEBUG)
		if (ubi_n < ubi_nmin) {
			ubi_nmin = ubi_n;
			log ("New minimum: %d URBs left.\n", ubi_n);
		}
#endif
	}
	spin_unlock_irqrestore (&ullock, flags);
	*ppurb = ubi[ix].urb;
	*ppbuf = ubi[ix].buf;
	return (ix != MAX_URB_COUNT);
} /* claim_urb */
	
/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void release_urb (dev_context_p pdc, struct urb * purb) {
	int	ix;
	
	UNUSED_ARG (pdc);
	assert (purb != NULL);
	assert (in_interrupt ());
	ix = (int) purb->context;
	assert (ix >= 0);
	assert (ix < MAX_URB_COUNT);
	ubi[ix].idx = ubi_free;
	ubi_free = ix;
	++ubi_n;
} /* release_urb */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void enlist_urb (dev_context_p pdc, struct urb * purb) {
	int	ix;

	assert (pdc != NULL);
	assert (purb != NULL);
	assert (in_interrupt ());
	ix = (int) purb->context;
	assert (ix >= 0);
	assert (ix < MAX_URB_COUNT);
	ubi[ix].idx = MAX_URB_COUNT;
	if (ubi_in != MAX_URB_COUNT) {
		ubi[ubi_in].idx = ix;
	} else {
		ubi_out = ix;
	}
	ubi_in = ix;
	atomic_inc (&pdc->rx_count);
} /* enlist_urb */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int unlist_urb (
	dev_context_p	pdc,
	struct urb **	ppurb,
	char **		ppbuf
) {
	int		ix;

	assert (pdc != NULL);
	assert (ppurb != NULL);
	assert (ppbuf != NULL);
	assert (in_interrupt ());
	ix = ubi_out;
	assert (ix >= 0);
	assert (ix <= MAX_URB_COUNT);
	ubi_out = ubi[ix].idx;
	if (ubi_out == MAX_URB_COUNT) {
		ubi_in = MAX_URB_COUNT;
	}
	atomic_dec (&pdc->rx_count);
	*ppurb = ubi[ix].urb;
	*ppbuf = ubi[ix].buf;
	return (ix != MAX_URB_COUNT);
} /* unlist_urb */

/*---------------------------------------------------------------------------*\
\*-U-------------------------------------------------------------------------*/
static void issue_hard_error (dev_context_p pdc, const char * msg, int arg) {

	lprintf (KERN_INFO, msg, arg);
	enter_critical ();
	if (!hard_error_issued) {
		hard_error_issued = TRUE;
		(*pdc->ev_handler) (USB_EVENT_HARD_ERROR);
	} else {
		lprintf (KERN_INFO, "Extra events ignored.\n");
	}
	leave_critical ();
} /* issue_hard_error */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void tx_dpc (void * data) {
	static void	usb_write (dev_context_p, unsigned long);
	dev_context_p	pdc;
	struct urb *	purb;
	
	assert (in_interrupt ());
	assert (data != NULL);
	pdc = (dev_context_p) data;
	purb = pdc->tx_urb;
	assert (atomic_read (&pdc->tx_pending));
	if (!atomic_read (&pdc->is_open)) {
		atomic_set (&pdc->tx_pending, 0);
		return;
	}
	if (0 != purb->status) {
		atomic_set (&pdc->tx_pending, 0);
		issue_hard_error (pdc, "Tx URB status: %d\n", purb->status);
		return;
	}
	if ((purb->actual_length > 0) && ((purb->actual_length % 64) == 0)) {
		log (
			"Inserting zero-length data block! (Wrote %d bytes)\n",
			purb->actual_length
		);
		usb_write (pdc, 0);
	} else {
		atomic_set (&pdc->tx_pending, 0);
		usb_TxStart ();	
		kick_scheduler ();
	}
} /* tx_dpc */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void usb_write_completion (struct urb * purb) {

	UNUSED_ARG (purb);
	assert (in_interrupt ());
	kick_queue (&tq_tx_dpc);
} /* usb_write_completion */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void usb_write (dev_context_p pdc, unsigned long len) {
	int	res;

	assert (pdc);
	if (!atomic_read (&pdc->is_open)) {
		log ("Attempt to write on closed device.");
		return;
	}
	assert (atomic_read (&pdc->tx_pending));
	usb_fill_bulk_urb (
		pdc->tx_urb,
		pdc->dev,
		usb_sndbulkpipe (pdc->dev, pdc->epwrite->bEndpointAddress),
		pdc->tx_buffer,
		len,
		usb_write_completion,
		NULL
		);
	if (0 != (res = usb_submit_urb (pdc->tx_urb))) {
		atomic_set (&pdc->tx_pending, 0);
		issue_hard_error (pdc, "Tx URB submission failed, code %d.\n", res);
	}
} /* usb_write */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void rx_dpc (void * data) {
	dev_context_p	pdc;
	struct urb *	purb;
	char *		pbuf;

	assert (in_interrupt ());
	assert (data != NULL);
	pdc = (dev_context_p) data;
	if (!atomic_read (&pdc->is_open)) {
		return;
	}
	if (!unlist_urb (pdc, &purb, &pbuf)) {
		log ("Completion w/o URB!\n");
		return;
	}
	if (0 != purb->status) {
		issue_hard_error (pdc, "Rx URB status: %d\n", purb->status);
		return;
	}
	usb_RxStart ();
	if (0 != purb->actual_length) {
		enter_critical ();
		(*pdc->rx_avail) (pbuf, purb->actual_length);
		leave_critical ();
		kick_scheduler ();
	}
	release_urb (pdc, purb);
} /* rx_dpc */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void usb_read_completion (struct urb * purb) {

	assert (in_interrupt ());
	assert (purb != NULL);
	atomic_dec (&fdslusb_capi_ctx->rx_pending);
	enlist_urb (fdslusb_capi_ctx, purb);
	kick_queue (&tq_rx_dpc);
} /* usb_read_completion */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void usb_read (dev_context_p pdc) {
	struct urb *	purb;
	char *		pbuf;
	int		res;

	assert (pdc);
	if (!atomic_read (&pdc->is_open)) {
		log ("Attempt to read from closed device.\n");
		return;
	}
	if (!claim_urb (pdc, &purb, &pbuf)) {
		log ("No URB/buffer available.\n");
		return;
	}
	atomic_inc (&pdc->rx_pending);
	usb_fill_bulk_urb (
		purb,
		pdc->dev,
		usb_rcvbulkpipe (pdc->dev, pdc->epread->bEndpointAddress),
		pbuf,
		MAX_TRANSFER_SIZE,
		usb_read_completion,
		purb->context
		);
	purb->transfer_flags |= USB_QUEUE_BULK;
	if (0 != (res = usb_submit_urb (purb))) {
		atomic_set (&pdc->rx_pending, 0);
		issue_hard_error (pdc, "Rx URB submission failed, code %d.\n", res);
	}
} /* usb_read */

/*---------------------------------------------------------------------------*\
\*-S-------------------------------------------------------------------------*/
static void scheduler_control (unsigned f) {
	int	flag = (int) f;
	int	changed;

	enter_critical ();
	changed = (atomic_read (&scheduler_enabled) != flag);
	leave_critical ();
	if (changed) {
		if (flag) {
			kick_scheduler ();
		} else {
			atomic_set (&scheduler_enabled, 0);
			wake_up_interruptible (&wait);
		}
	}
} /* scheduler_control */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int make_thread (void) {

	if (-1 != thread_pid) {
		log ("Thread[%d] is running.\n", thread_pid);
		return -1;
	}
	atomic_set (&thread_run_flag, 1);
	atomic_set (&thread_capi_run_flag, 0);
	init_waitqueue_head (&capi_wait);
	thread_pid = kernel_thread (scheduler, NULL, 0);
	log ("Thread[%d] started.\n", thread_pid);
	return thread_pid;
} /* make_thread */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void kill_thread (void) {

	if (-1 == thread_pid) {
		log ("Thread has not been started.\n");
		return;
	}
	atomic_set (&thread_run_flag, 0);
	if (!atomic_read (&thread_capi_run_flag)) {
		wake_up_interruptible (&capi_wait);
	} else {
		wake_up_interruptible (&wait);
	}
	log ("Thread signalled, waiting for termination...\n");
	down (&hotplug);
	log ("Thread[%d] terminated.\n", thread_pid);
	thread_pid = -1;
} /* kill_thread */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void enable_thread (void) {

	assert (atomic_read (&thread_run_flag));
	info (!atomic_read (&thread_capi_run_flag));
	if (!atomic_read (&thread_capi_run_flag)) {
		atomic_set (&thread_capi_run_flag, 1);
		wake_up_interruptible (&capi_wait);
		log ("Thread enabled.\n");
	}
} /* enable_thread */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void disable_thread (void) {

	assert (atomic_read (&thread_run_flag));
	info (atomic_read (&thread_capi_run_flag));
	if (atomic_read (&thread_capi_run_flag)) {
		atomic_set (&thread_capi_run_flag, 0);
		wake_up_interruptible (&wait);
		log ("Thread disabled.\n");
	}
} /* disable_thread */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void dpc (void * data) {

	UNUSED_ARG (data);
	if (!in_critical ()) {
		run_task_queue (&tq_private);
	}
} /* dpc */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int scheduler (void * arg) {

	UNUSED_ARG (arg);
	SNPRINTF (current->comm, 16, "%s_thread", TARGET);	/* See: sched.h */
	log ("Starting scheduler thread '%s'...\n", current->comm);
	while (atomic_read (&thread_run_flag)) {
		if (!atomic_read (&thread_capi_run_flag)) {
			interruptible_sleep_on (&capi_wait);
		} else {
			if (atomic_read (&scheduler_enabled)) {
				interruptible_sleep_on_timeout (&wait, TEN_MSECS);
			} else {
				interruptible_sleep_on (&wait);
			}
		}
		if (!atomic_read (&thread_run_flag)) {
			info (atomic_read (&thread_run_flag));
			break;
		}
		if (!atomic_read (&thread_capi_run_flag)) {
			info (atomic_read (&thread_capi_run_flag));
			continue;
		}
		if (!atomic_read (&scheduler_enabled)) {
			info (atomic_read (&scheduler_enabled));
			continue;
		}
		if (fdslusb_capi_lib == NULL) {
			log ("Library lost! Hot unplug?\n");
			break;
		}
		os_timer_poll ();
		if ((*fdslusb_capi_lib->cm_schedule) ()) {
			scheduler_control (TRUE);
		}
	}
	log ("Scheduler thread stopped.\n");
	up (&hotplug);
	return 0;
} /* scheduler */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void kick_scheduler (void) {

	atomic_set (&scheduler_enabled, 1);
	wake_up_interruptible (&wait);
} /* kick_scheduler */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void stop_base (void) {
	dev_context_p	pdc = fdslusb_capi_ctx;

	if (NULL == pdc) {
		return;
	}
	atomic_set (&pdc->is_open, 0);
	atomic_set (&pdc->rx_pending, 0);
	atomic_set (&pdc->tx_pending, 0);
	log ("USB layer stopped.\n");
} /* stop_base */
		
/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void usb_Open (
	unsigned			max_buffer_size,
	get_next_tx_buffer_func_p	get_next_tx_buffer,
	event_handler_func_p		event_handler,
	new_rx_buffer_avail_func_p	new_rx_buffer_avail,
	complete_func_p			open_complete,
	unsigned			ref_data
) {
	dev_context_p			pdc = fdslusb_capi_ctx;

	atomic_set (&pdc->is_open, 1);
	atomic_set (&pdc->tx_pending, 0);
	atomic_set (&pdc->rx_pending, 0);
	atomic_set (&pdc->rx_count, 0);
	pdc->max_buf_size = max_buffer_size;
	pdc->tx_get_next  = get_next_tx_buffer;
	pdc->rx_avail     = new_rx_buffer_avail;
	pdc->ev_handler   = event_handler;
	hard_error_issued = FALSE;
	assert (open_complete);
	(*open_complete) (0, ref_data);
	usb_RxStart ();
} /* usb_Open */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void usb_Close (complete_func_p close_complete, unsigned ref_data) {

	stop_base ();
	assert (close_complete);
	(*close_complete) (0, ref_data);
	if (hard_error_issued) {
		start_closing_task (NULL, NULL);
	}
} /* usb_Close */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void usb_RxStart (void) {
	dev_context_p	pdc;

	pdc = fdslusb_capi_ctx;
	assert (pdc);
	if (atomic_read (&pdc->rx_count) > 0) {
	 	kick_queue (&tq_rx_dpc);
	} else {
	 	usb_read (pdc);
	}
} /* usb_RxStart */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void usb_TxStart (void) {
	dev_context_p	pdc = fdslusb_capi_ctx;
	unsigned long	len;

	assert (pdc);
	if (atomic_xchg (&pdc->tx_pending, 1)) {
		return;
	}
	enter_critical ();
	assert (pdc->tx_get_next);
	len = (*pdc->tx_get_next) (pdc->tx_buffer);
	assert (pdc->max_buf_size);
	assert (len <= pdc->max_buf_size);
	leave_critical ();
	if (0 == len) {
		atomic_set (&pdc->tx_pending, 0);
		return;
	}
	info (atomic_read (&pdc->is_open));
	if (atomic_read (&pdc->is_open)) {
		assert (atomic_read (&pdc->tx_pending));
		usb_write (pdc, len);
	}
} /* usb_TxStart */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void usb_GetDeviceInfo (device_info_p device_info) {

	assert (device_info);
	device_info->product_id = fdslusb_capi_ctx->dev->descriptor.idProduct;
	device_info->release_number = fdslusb_capi_ctx->dev->descriptor.bcdDevice;
	device_info->self_powered = 0;
} /* usb_GetDeviceInfo */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/

