/* 
 * main.c
 * Copyright (C) 2003, AVM GmbH. All rights reserved.
 * 
 * This Software is  free software. You can redistribute and/or
 * modify such free software under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * The free software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this Software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA, or see
 * http://www.opensource.org/licenses/lgpl-license.html
 * 
 * Contact: AVM GmbH, Alt-Moabit 95, 10559 Berlin, Germany, email: info@avm.de
 */

#ifndef EXPORT_SYMTAB
# define EXPORT_SYMTAB
#endif

#include <stdarg.h>
#include <asm/uaccess.h>
#include <linux/usb.h>
#include <linux/config.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/string.h>
#include <linux/skbuff.h>
#include <linux/errno.h>
#include <linux/capi.h>
#include <linux/ctype.h>
#include "defs.h"
#include "capilli.h"
#include "lib.h"
#include "driver.h" 
#include "tools.h"

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#define	INIT_FAILED		-1
#define	INIT_SUCCESS		0

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static char *	REVCONST = "$Revision: $";
char		REVISION[32];

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)

#if defined (__fcdslslusb__)
static struct usb_device_id fcslusb_id_table[] __initdata = {
	
	{ USB_DEVICE(VENDOR_ID_AVM, USB_PRODUCT_ID_FCDSLSLUSB) },
	{ /* Terminating entry */ }
} ;
#elif defined (__fcdslusb2__)
static struct usb_device_id fcslusb_id_table[] __initdata = {
	
	{ USB_DEVICE(VENDOR_ID_AVM, USB_PRODUCT_ID_FCDSLUSB2) },
	{ /* Terminating entry */ }
} ;
#else
#error Card specifier missing!
#endif

MODULE_DEVICE_TABLE (usb, fcslusb_id_table);

#else
#define	__exit
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
short int	VCC = 0x01;
short int	VPI = 0x01;
short int	VCI = 0x20;

MODULE_PARM (VCC, "h");
MODULE_PARM (VPI, "h");
MODULE_PARM (VCI, "h");

MODULE_PARM_DESC (VCC, "VCC - Virtual Channel Connection");
MODULE_PARM_DESC (VPI, "VPI - Virtual Path Identifier");
MODULE_PARM_DESC (VCI, "VCI - Virtual Channel Identifier");

#if defined (MODULE_LICENSE)
MODULE_LICENSE ("Proprietary");
#endif
#if defined (MODULE_DESCRIPTION)
MODULE_DESCRIPTION ("CAPI4Linux: Driver for " PRODUCT_LOGO);
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#ifndef NDEBUG
static void base_address (void) {

	log ("Base address: %p\n", base_address);
	log ("Compile time: %s\n", __TIME__);
} /* base_address */
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void fcslusb_disconnect (struct usb_device *, void *);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
static void * fcslusb_probe (struct usb_device *, unsigned int);
#else
static void * fcslusb_probe (struct usb_device *, unsigned int, const struct usb_device_id *);
#endif

static struct usb_driver fcslusb_driver = {

        name:           TARGET,
        probe:          fcslusb_probe,
        disconnect:     fcslusb_disconnect,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
        id_table:       fcslusb_id_table,
#endif
} ;

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void inc_use_count (void) { MOD_INC_USE_COUNT; }
void dec_use_count (void) { MOD_DEC_USE_COUNT; }

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
static void * fcslusb_probe (struct usb_device * dev, unsigned int ifnum) 
#else
static void * fcslusb_probe (
		struct usb_device * 		dev, 
		unsigned int 			ifnum,
		const struct usb_device_id *	devid
)
#endif
{
	card_p					pdc;
	struct usb_interface_descriptor *	iface;
	struct usb_config_descriptor *		cfgd;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
	UNUSED_ARG (devid);
#endif

#if defined (__fcdslslusb__)
	if ((VENDOR_ID_AVM != dev->descriptor.idVendor)
	||  (USB_PRODUCT_ID_FCDSLSLUSB != dev->descriptor.idProduct))
#elif defined (_fcdslusb2__)
	if ((VENDOR_ID_AVM != dev->descriptor.idVendor)
	||  (USB_PRODUCT_ID_FCDSLUSB2  != dev->descriptor.idProduct))
#endif
	{
	exit1:
		log ("Probe function failed!\n");
		return NULL;
	}

	MOD_INC_USE_COUNT;
	log ("Probe for device #%d, interface %d.\n", dev->devnum, ifnum);
	if (NULL == (pdc = (card_p) hmalloc (sizeof (card_t)))) {
		error ("Could not allocate device context.\n");
	exit2:
		MOD_DEC_USE_COUNT;
		goto exit1;
	}
	fcslusb_capi_card = pdc;
	memset (pdc, 0, sizeof (card_t));
	pdc->dev = dev;

	cfgd = dev->actconfig;
	assert (cfgd != NULL);
	if (usb_interface_claimed (cfgd->interface + ifnum)) {
		error ("Interface already claimed.\n");
	exit3:
		hfree (pdc);
		goto exit2;
	} 
	iface = cfgd->interface[ifnum].altsetting + 0;
	assert (iface != NULL);

	pdc->epwrite = iface->endpoint + 0;
        pdc->epread  = iface->endpoint + 1;
	if (usb_set_configuration (dev, cfgd->bConfigurationValue)) {
		error ("Could not set configuration.\n");
	exit4:
		goto exit3;
	}
	if (usb_set_interface (dev, 0, 0)) {
		error ("Could not set interface.\n");
	exit5:
		goto exit4;
	}
	card_config = cfgd->bConfigurationValue;
	log ("Device is in config #%d.\n", card_config);
	//
	// Usually we find the device configured. Why?!?  
	// 
	if (NULL == (fcslusb_capi_lib = link_library (&fcslusb_capi_card))) {
		error ("Linking to library failed.\n");
	exit6:
		goto exit5;
	}
	init_closing_task ();
	make_thread ();
	if (0 != fcslusb_add_card (&fcslusb_capi_driver, NULL)) {
		kill_thread ();
		free_library ();
		goto exit6;
	}
	return pdc;
} /* fcslusb_probe */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void close_complete (void * ptr) {
	card_p	pdc = (card_p) ptr;

	log ("Disconnect completed.\n");
	assert (ptr != NULL);
	free_library ();
	fcslusb_capi_card = NULL;
	fcslusb_capi_lib = NULL;
	hfree (pdc);
	MOD_DEC_USE_COUNT;
} /* close_complete */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void fcslusb_disconnect (struct usb_device * dev, void * ptr) {
	card_p	pdc = (card_p) ptr;

	UNUSED_ARG (dev);
	assert (ptr != NULL);
	log ("Disconnect for device #%d.\n", dev->devnum);
	if (NULL != fcslusb_capi_controller[1]) {	
		/* In case of sudden removal... */
		start_closing_task (&close_complete, pdc);
	} else {
		close_complete (pdc);
	}
} /* fcslusb_disconnect */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int __init fcslusb_init (void) {
	int	urr, res = INIT_FAILED;
	char *	tmp;
	
	EXPORT_NO_SYMBOLS;
#ifndef NDEBUG
	base_address ();
#endif
	MOD_INC_USE_COUNT;
	if ((NULL != (tmp = strchr (REVCONST, ':'))) && isdigit (*(tmp + 2))) {
		strncpy (REVISION, tmp + 1, sizeof (REVISION));
		tmp = strchr (REVISION, '$');
		*tmp = 0;
	} else {
		strcpy (REVISION, REV_DEFAULT);
	}
	strncpy (fcslusb_capi_driver.revision, REVISION, sizeof (fcslusb_capi_driver.revision));
	msg ("%s, revision %s\n", DRIVER_LOGO, REVISION);
        msg ("(%s built on %s at %s)\n", TARGET, __DATE__, __TIME__);

	/*-------------------------------------------------------------------*\
	 * 64 bit CAPI is not supported yet.
	\*-------------------------------------------------------------------*/
	if (sizeof (char *) > 4) {
		error ("Cannot deal with 64 bit CAPI messages!\n");
		goto exit;
	}

        msg ("Loading...\n");
	if (NULL == (fcslusb_capi_interface = attach_capi_driver (&fcslusb_capi_driver))) {
		error ("Could not attach the driver.\n");
		goto exit;
	}
	if (0 > (urr = usb_register (&fcslusb_driver))) {
		detach_capi_driver (&fcslusb_capi_driver);
		error ("Function usb_register returned %d.\n", urr);
		goto exit;
	}
	res = INIT_SUCCESS;
exit:
	MOD_DEC_USE_COUNT;
	msg ("%soaded.\n", (res == INIT_SUCCESS) ? "L" : "Not l");
	return res;
} /* fcslusb_init */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void __exit fcslusb_exit (void) {

	msg ("Removing...\n");
	kill_thread ();
	detach_capi_driver (&fcslusb_capi_driver);
	usb_deregister (&fcslusb_driver);
#ifndef NDEBUG
	if (hallocated() != 0) {
		error ("%u bytes leaked.\n", hallocated());
	}
#endif
	msg ("Removed.\n");
} /* fcslusb_exit  */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
module_init (fcslusb_init);
module_exit (fcslusb_exit);

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/

