/* 
 * driver.c
 * Copyright (C) 2002, AVM GmbH. All rights reserved.
 * 
 * This Software is  free software. You can redistribute and/or
 * modify such free software under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * The free software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this Software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA, or see
 * http://www.opensource.org/licenses/lgpl-license.html
 * 
 * Contact: AVM GmbH, Alt-Moabit 95, 10559 Berlin, Germany, email: info@avm.de
 */

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/atomic.h>
#include <asm/uaccess.h>
#include <asm/system.h>
#include <linux/version.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/tqueue.h>
#include <linux/skbuff.h>
#include <linux/kernel.h>
#include <linux/spinlock.h>
#include <linux/usb.h>
#include <linux/delay.h>
#include <linux/ctype.h>
#include <linux/string.h>
#include <linux/capi.h>
#include <linux/kernelcapi.h>
#include <stdarg.h>
#include "defs.h"
#include "libstub.h"
#include "ca.h"
#include "capilli.h"
#include "tables.h"
#include "queue.h"
#include "buffers.h"
#include "lib.h"
#include "main.h"
#include "tools.h"
#include "fw.h"
#include "devif.h"
#include "driver.h"

#if defined (LOG_MESSAGES)
# define mlog		log
#else
# define mlog(f,a...)
#endif

#define	__entry

#ifndef HZ
# error HZ is not defined...
#endif
#define	KILOBYTE		1024
#define	MEGABYTE		(1024*KILOBYTE)

#define	DEBUG_MEM_SIZE		MEGABYTE
#define	DEBUG_TIMEOUT		100
#define	RESET_DELAY		10

#define	TEN_MSECS		(HZ / 100)
#define	STACK_DELAY		(HZ / 2)
#define	USB_CTRL_TIMEOUT	3 * HZ
#define	USB_CFG_BLK_SIZE	777
#define	USB_BULK_TIMEOUT	3 * HZ
#define	USB_MAX_FW_LENGTH	393216

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,20)
#define	DEVREQUEST		devrequest
#define	DR_REQUEST_TYPE		requesttype
#define	DR_REQUEST		request
#define	DR_VALUE		value
#define	DR_INDEX		index
#define	DR_LENGTH		length
#else
#define	DEVREQUEST		struct usb_ctrlrequest
#define	DR_REQUEST_TYPE		bRequestType
#define	DR_REQUEST		bRequest
#define	DR_VALUE		wValue
#define	DR_INDEX		wIndex
#define	DR_LENGTH		wLength
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#define	DBG_FORMAT_LEN		1024
#define	DBG_MEM_SIZE		(1024 * 1024)
#define	DBG_TIMEOUT		100
#define	DBG_MIN_SIZE		1024

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
card_p				fcslusb_capi_card	= NULL;
lib_callback_t *		fcslusb_capi_lib	= NULL;
struct capi_driver_interface *	fcslusb_capi_interface	= NULL;

static int			load_flag		= FALSE;
static int			init_flag		= FALSE;
static int			nvers			= 0;
atomic_t			crit_count		= ATOMIC_INIT(0);
spinlock_t			stack_lock		= SPIN_LOCK_UNLOCKED;
static atomic_t			scheduler_enabled	= ATOMIC_INIT (1);
struct capi_ctr *		fcslusb_capi_controller[2]	= { NULL, NULL } ;
static bundle_t			ctrl_context[2];
static per_ctrl_t		ctrl_params[2];
static unsigned long		qt_flags;
static spinlock_t		qt_lock			= SPIN_LOCK_UNLOCKED;
static atomic_t			thread_flag		= ATOMIC_INIT (0);
static atomic_t			thread_capi_flag	= ATOMIC_INIT (0);
static int			thread_pid		= -1;
static atomic_t			tx_flag			= ATOMIC_INIT (0);
static atomic_t			rx_flag			= ATOMIC_INIT (0);
static char *			firmware_ptr		= NULL;
static unsigned			firmware_len		= 0;
static int			hard_error_issued	= FALSE;
int				card_config;

#if !defined (NDEBUG)
static atomic_t			in_scheduler		= ATOMIC_INIT (0);
#endif

static void			tx_task (unsigned long data);
static void			rx_task (unsigned long data);
static void			tx_handler (card_p pdc);
static void			rx_handler (card_p pdc);
static void			kick_scheduler (void);
static int			scheduler (void *);
static void			enable_thread (void);
static void			disable_thread (void);
int				make_thread (void);
void				kill_thread (void);
static void			scheduler_control (unsigned);
static void			wakeup_control (unsigned);
static void			version_callback (char *);
static void			kill_version (struct capi_ctr *);
static unsigned			scheduler_suspend (unsigned long);
static unsigned			scheduler_resume (void);
static unsigned			controller_remove (void);
static unsigned			controller_add (void);

static DECLARE_TASKLET(tx_tasklet, tx_task, 0);
static DECLARE_TASKLET(rx_tasklet, rx_task, 0);
static DECLARE_WAIT_QUEUE_HEAD(wait);
static DECLARE_WAIT_QUEUE_HEAD(capi_wait);
static DECLARE_MUTEX_LOCKED(hotplug);
static DECLARE_MUTEX_LOCKED(config);

static DECLARE_WAIT_QUEUE_HEAD(dbg_wait);
static DECLARE_MUTEX_LOCKED(thread_sync);

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static atomic_t			resetting_ctrl		= ATOMIC_INIT (0);

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static struct tq_struct		tq_close;
static atomic_t			closing_task_running	= ATOMIC_INIT (0);
static void 		     (*	close_func1) (void *)	= NULL;
static void *			close_data1		= NULL;
static void 		     (*	close_func2) (void *)	= NULL;
static void *			close_data2		= NULL;

static DECLARE_MUTEX_LOCKED(notify);

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
struct capi_driver		fcslusb_capi_driver	= {

	TARGET,
	REV_DEFAULT,			
	fcslusb_load_ware,
	fcslusb_reset_ctrl,
	fcslusb_remove_ctrl,
	fcslusb_register_appl,
	fcslusb_release_appl,
	fcslusb_send_msg,
	fcslusb_proc_info,
	fcslusb_ctr_info,
	fcslusb_drv_info,
	NULL,		/* fcslusb_add_card */
	NULL,
	NULL,
	0,
	NULL,
	""
} ;

static functions_t		ca_funcs = {

	7,
	scheduler_control,
	wakeup_control,
	version_callback,
	scheduler_suspend,
	scheduler_resume,
	controller_remove,
	controller_add
} ;

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static char *	invalid_msg   = "Invalid application id #%d.\n";

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static inline void kill_versions (void) {
	
	if (ctrl_context[0].ctrl != NULL) {
		kill_version (fcslusb_capi_controller[0]);
	}
	kill_version (fcslusb_capi_controller[1]);
} /* kill_versions */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int make_ctx (void ** pctx) {
	urb_context_p	uctx;
	
	assert (pctx != NULL);
	info (*pctx == NULL);
	if (NULL == (uctx = (urb_context_p) hcalloc (sizeof (urb_context_t)))) {
		error ("Error: Could not allocate transfer context!\n");
	}
	*pctx = (void *) uctx;
	return (uctx != NULL);
} /* make_ctx */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int kill_ctx (void ** pctx) {
	urb_context_p	uctx;

	assert (pctx != NULL);
	assert (*pctx != NULL);
	uctx = (urb_context_p) *pctx;
	hfree (uctx);
	*pctx = NULL;
	return TRUE;
} /* kill_ctx */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int start_stack (card_p cp) {
	int		res = 0;
	unsigned	tn, ts, rn, rs;

	/* allocating buffers */
	if (!dif_buffer_params (&tn, &ts, &rn, &rs)) {
		error ("Error: Could not get buffer geometry!\n");
		return 0;
	}
	if (NULL == (cp->tx_pool = init_urbs (tn, ts))) {
		return 0;
	}
	if (NULL == (cp->rx_pool = init_urbs (rn, rs))) {
		remove_urbs (&cp->tx_pool, kill_ctx);
		return 0;
	}
	if (build_urb_contexts (cp->tx_pool, make_ctx, kill_ctx)
	&&  build_urb_contexts (cp->rx_pool, make_ctx, kill_ctx)) {
		log ("URB pool and context has been allocated!\n");
	} else {
		remove_urbs (&cp->tx_pool, kill_ctx);
		remove_urbs (&cp->rx_pool, kill_ctx);
		return 0;
	}
	/* preparing the stack */
	cp->count = 0;
	table_init (&cp->appls);
	queue_init (&cp->queue);

	assert (fcslusb_capi_lib != NULL);
	assert (fcslusb_capi_lib->cm_register_ca_functions != NULL);
	assert (fcslusb_capi_lib->cm_start != NULL);
	assert (fcslusb_capi_lib->cm_init != NULL);
	assert (fcslusb_capi_lib->cm_exit != NULL);
	(*fcslusb_capi_lib->cm_register_ca_functions) (&ca_funcs);
	if ((*fcslusb_capi_lib->cm_start) ()) {
		error ("Starting the stack failed...\n");
	early_exit:
		remove_urbs (&cp->rx_pool, kill_ctx);
		remove_urbs (&cp->tx_pool, kill_ctx);
		queue_exit (&cp->queue);
		table_exit (&cp->appls);
		return FALSE;
	}
	(void) (*fcslusb_capi_lib->cm_init) (0, 0);
	assert (nvers == 2);
	
	/* debug buffer */
	assert (fcslusb_capi_lib->cc_debugging_needed != NULL);
	if ((*fcslusb_capi_lib->cc_debugging_needed) ()) {
		log ("Debugging enabled.\n");
	}

	assert (fcslusb_capi_lib->cc_run != NULL);
	res = (*fcslusb_capi_lib->cc_run) ();
	if (res == FALSE) {	/* Failure */
		error ("Firmware does not respond!\n");
	reset_exit:
		(*fcslusb_capi_lib->cm_exit) ();
		goto early_exit;
	}
	assert (fcslusb_capi_lib->cc_compress_code != NULL);
	(*fcslusb_capi_lib->cc_compress_code) ();
	
	/* start the stack */
	enter_critical ();
	if ((*fcslusb_capi_lib->cm_activate) ()) {
		error ("Activation of the card failed.\n");
		leave_critical ();
		goto reset_exit;
	}
	leave_critical ();
	make_thread ();
	enable_thread ();
	return (init_flag = TRUE);
} /* start_stack */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void stop_stack (card_p card) {

	if (!init_flag) {
		log ("Stack not initialized.\n");
		return;
	}
	
	init_flag = FALSE;
	(*fcslusb_capi_lib->cm_exit) ();
	if (thread_pid != -1) {
	     /* disable_thread (); */
		kill_thread ();
	}
	remove_urbs (&card->tx_pool, kill_ctx);
	remove_urbs (&card->rx_pool, kill_ctx);
	queue_exit (&card->queue);
	table_exit (&card->appls);
	kill_versions ();
	nvers = 0;
} /* stop_stack */

/*-C-------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void appl_released_notify (int appl) {

	UNUSED_ARG (appl);
	if (atomic_read (&closing_task_running)) {
		log ("Note: appl %d released.\n", appl);
		up (&notify);
	}
} /* appl_released_notify */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void closing_task (void * ctx) {
	appl_t *	appp;
	card_p		card = ctx;

	DECLARE_WAIT_QUEUE_HEAD(close_wait);

#define CALLBACK(ix)	if (close_func##ix != NULL) { \
				log ("Callback[%p]\n", close_func##ix); \
				(* (close_func##ix)) (close_data##ix); \
				close_func##ix = NULL; \
			}

	log ("Closing task started.\n");
	assert (ctx != NULL);
	atomic_set (&closing_task_running, 1);
	atomic_set (&resetting_ctrl, 1);
	if (0 != card->count) {
		assert (card->appls != NULL);
		if (card->appls != NULL) {
			appp = first_appl (card->appls);
			while (appp != NULL) {
				log ("Releasing appl #%d...\n", appp->id);
				fcslusb_release_appl (
					fcslusb_capi_controller[1],
					appp->id
				);
				kick_scheduler ();
				down (&notify);
				appp = next_appl (card->appls, appp);
			}
		}
	}
	if (atomic_read (&thread_capi_flag)) {
		init_waitqueue_head (&close_wait);
		kick_scheduler ();
		log ("Waiting...\n");
		interruptible_sleep_on_timeout (&close_wait, STACK_DELAY);
		log ("... completed!\n");
	}
	if (init_flag) {
		stop_stack (card);
	}
	atomic_set (&resetting_ctrl, 0);
	fcslusb_reset_ctrl (fcslusb_capi_controller[1]);
	fcslusb_remove_ctrl (fcslusb_capi_controller[1]);
	CALLBACK (1);
	CALLBACK (2);
	atomic_set (&closing_task_running, 0);
	log ("Closing task terminated.\n");

#undef CALLBACK

} /* closing_task */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void init_closing_task (void) {

	atomic_set (&closing_task_running, 0);
} /* init_closing_task */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void start_closing_task (void (* func) (void *), void * data) {
	unsigned long		flags;
	static spinlock_t	close_lock = SPIN_LOCK_UNLOCKED;

	spin_lock_irqsave (&close_lock, flags);
	assert ((close_func1 == NULL) || (close_func2 == NULL));
	if (close_func1 != NULL) {
		assert (close_func2 == NULL);
		close_func2 = func;
		close_data2 = data;
	} else {
		close_func1 = func;
		close_data1 = data;
	}
	spin_unlock_irqrestore (&close_lock, flags);
	if (atomic_read (&closing_task_running)) {
		log ("Closing task already running.\n");
		return;
	}
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	tq_close.next = NULL;
#else
	tq_close.list.next = NULL;
#endif
	tq_close.sync = 0;
	tq_close.data = fcslusb_capi_card;
	tq_close.routine = &closing_task;
	schedule_task (&tq_close);
	log ("Closing task scheduled.\n");
} /* start_closing_task */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static struct capi_ctr * get_capi_ctr (int ctrl) {
	int	ix;
	
	assert (fcslusb_capi_card);
	assert (fcslusb_capi_card->ctrl2 > 0);
	ix = (fcslusb_capi_card->ctrl2 == ctrl) ? 1 : 0;
	info (NULL != fcslusb_capi_controller[ix]);
	return fcslusb_capi_controller[ix];
} /* get_capi_ctr */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int nbchans (struct capi_ctr * ctrl) {
	per_ctrl_p	C;
	unsigned char *	prf;
	int		temp = 2;
    
	assert (ctrl);
	assert (ctrl->driverdata);
	C = GET_CTRL (ctrl);
	prf = (unsigned char *) C->string[6];
	if (prf != NULL) {
		temp = prf[2] + 256 * prf[3];
	}
	return temp;
} /* nbchans */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static struct sk_buff * make_0xfe_request (unsigned appl) {
	unsigned char    request[8];
	struct sk_buff * skb;

	if (NULL == (skb = alloc_skb (8, GFP_ATOMIC))) {
		lprintf (KERN_ERR, "Unable to allocate message buffer.\n");
	} else {    
		request[0] = 8;
		request[1] = 0;
		request[2] = appl & 0xFF;
		request[3] = (appl >> 8) & 0xFF;
		request[4] = 0xFE;
		request[5] = 0x80;
		memcpy (skb_put (skb, 8), &request, 8);
	}
	return skb;
} /* make_0xfe_request */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void reset_completion (struct urb * purb) {
	DEVREQUEST * req;

	assert (NULL != purb);
	assert (0 == purb->status);
	log ("Completed device reset.\n");
	req = (DEVREQUEST *) purb->context;
	hfree (req);
	usb_free_urb (purb);
} /* reset_completion */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int reset (struct usb_device * dev, void (* cfunc) (struct urb *)) {
	DEVREQUEST *	req;
	struct urb *	purb;
	int		res;

	if (NULL == (req = (DEVREQUEST *) hmalloc (sizeof (DEVREQUEST)))) {
		error ("Not enough memory for device request.\n");
		return -ENOMEM;
	}
	if (NULL == (purb = usb_alloc_urb (0))) {
		error ("Not enough memory for device request URB.\n");
		hfree (req);
		return -ENOMEM;
	}
	req->DR_REQUEST_TYPE = 0;
	req->DR_REQUEST      = USB_REQ_SET_CONFIGURATION;
	req->DR_VALUE        = 0;	/* Little endian required here! */ 
	req->DR_INDEX        = 0;	/* Little endian required here! */
	req->DR_LENGTH       = 0;	/* Little endian required here! */
	usb_fill_control_urb (
		purb,
		dev,
		usb_sndctrlpipe (dev, 0),
		(unsigned char *) req,
		NULL,
		0,
		(cfunc == NULL) ? reset_completion : cfunc,
		req
	);
	if (0 != (res = usb_submit_urb (purb))) {
		hfree (req);
	}
	return res;
} /* reset */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void set_config_completion (struct urb * purb) {
	DEVREQUEST * req;

	assert (NULL != purb);
	assert (0 == purb->status);
	log ("Completed SET_CONFIGURATION(0).\n");
	req = (DEVREQUEST *) purb->context;
	hfree (req);
	usb_free_urb (purb);
	up (&config);
} /* set_config_completion */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int select_config (struct usb_device * dev, int confix) {
	int	cfg, res;
	
	assert (0 == in_interrupt ());
	cfg = (confix == UNDEF_CONFIG) ? 0 : confix;
	if (cfg == card_config) {
		log ("select_config(%d) exited, already configured.\n", cfg);
		return 0;
	}
	log ("select_config(%d)\n", cfg);
	if (0 == cfg) {
		reset (dev, set_config_completion);
		down (&config);
		res = 0;	/* Well... nothing but an assumption! :-) */
	} else  {
		res = usb_set_configuration (dev, cfg);
	}
	if (0 == res) { 
		card_config = cfg;
	} else {
		log ("select_config(%d) returned %d.\n", cfg, res);
	}
	return res;
} /* select_config */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void scan_version (struct capi_ctr * ctrl, const char * ver) {
	int		vlen, i;
	char *		vstr;
	per_ctrl_p	C;

	assert (ctrl);
	assert (ctrl->driverdata);
	C = GET_CTRL (ctrl);

	vlen = (unsigned char) ver[0];
	C->version = vstr = (char *) hmalloc (vlen);
	if (NULL == vstr) {
		log ("Could not allocate version buffer.\n");
		return;
	}
	memcpy (C->version, ver + 1, vlen);
	i = 0;
	for (i = 0; i < 8; i++) {
		C->string[i] = vstr + 1;
		vstr += 1 + *vstr;
	} 
#if defined (NDEBUG)
	lprintf (KERN_INFO, "Stack version %s\n", C->string[0]);
#endif
	log ("Library version:    %s\n", C->string[0]);
	log ("Card type:          %s\n", C->string[1]);
	log ("Capabilities:       %s\n", C->string[4]);
	log ("D-channel protocol: %s\n", C->string[5]);
} /* scan_version */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void copy_version (struct capi_ctr * ctrl) {
	char *		tmp;
	per_ctrl_p	C;
	
	assert (ctrl);
	assert (ctrl->driverdata);
	C = GET_CTRL (ctrl);
	
	if (NULL == (tmp = C->string[3])) {
		lprintf (KERN_ERR, "Do not have version information...\n");
		return;
	}
	strncpy (ctrl->serial, tmp, CAPI_SERIAL_LEN);
#if defined (__fcdslslusb__)
	if (ctrl == fcslusb_capi_controller[0]) {
		memset (&ctrl->profile, 0, sizeof (capi_profile));
	} else {
		memcpy (&ctrl->profile, C->string[6], sizeof (capi_profile));
	}
#else
	memcpy (&ctrl->profile, C->string[6], sizeof (capi_profile));
#endif
	strncpy (ctrl->manu, "AVM-GmbH", CAPI_MANUFACTURER_LEN);
	ctrl->version.majorversion = 2;
	ctrl->version.minorversion = 0;
	tmp = C->string[0];
	ctrl->version.majormanuversion = (((tmp[0] - '0') & 15) << 4)
					+ ((tmp[2] - '0') & 15);
	ctrl->version.minormanuversion = ((tmp[3] - '0') << 4)
					+ (tmp[5] - '0') * 10
					+ ((tmp[6] - '0') & 15);
} /* copy_version */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void kill_version (struct capi_ctr * ctrl) {
	int		i;
	per_ctrl_p	C;
	
	assert (ctrl);
	assert (ctrl->driverdata);
	C = GET_CTRL (ctrl);

	for (i = 0; i < 8; i++) {
		C->string[i] = NULL;
	}
	if (C->version != NULL) {
		hfree (C->version);
		C->version = NULL;
	}
} /* kill_version */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void pprintf (char * page, int * len, const char * fmt, ...) {
	va_list args;

	va_start (args, fmt);
	*len += vsprintf (page + *len, fmt, args);
	va_end (args);
} /* pprintf */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void init_ctrl (void) {

	ctrl_context[0].card = ctrl_context[1].card = NULL;
	ctrl_context[0].ctrl = ctrl_context[1].ctrl = NULL;
	fcslusb_capi_controller[0] = fcslusb_capi_controller[1] = NULL;
} /* init_ctrl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int setup_ctrl (card_p cp, int ix) {
	int			cix;
	struct capi_ctr *	cptr;
	
	assert ((0 < ix) && (ix < 3));
	cptr = (*fcslusb_capi_interface->attach_ctr) (&fcslusb_capi_driver, SHORT_LOGO, NULL);
	if (NULL == cptr) {
		lprintf (KERN_INFO, "Error: Could not attach controller %d.\n", ix);
		return -EBUSY;
	}
	if ((1 == ix) && (cptr->cnr != 1)) {
		cix = ix;
	} else {
		cix = ix - 1;
	}
	log ("Controller %d (%d) --> %d\n", ix, cptr->cnr, cix);
	fcslusb_capi_controller[cix] = cptr;
	ctrl_context[cix].card = cp;
	ctrl_context[cix].ctrl = &ctrl_params[cix];
	fcslusb_capi_controller[cix]->driverdata = (void *) &ctrl_context[cix];
	return 0;
} /* setup_ctrl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int fcslusb_kcapi_init (void) {
	int     res;

	if (NULL != fcslusb_capi_controller[1]) {
		error ("Cannot handle two controllers.\n");
		return -EBUSY;
	}
	init_ctrl ();
#if defined (__fcdslslusb__)
	if (0 != (res = setup_ctrl (fcslusb_capi_card, 2))) {
		init_ctrl ();
		return res;
	}
#else
	if (0 != (res = setup_ctrl (fcslusb_capi_card, 1))) {
		return res;
	}
	if (ctrl_context[0].ctrl != NULL) {
		if (0 != (res = setup_ctrl (fcslusb_capi_card, 2))) {
			(* fcslusb_capi_interface->detach_ctr) (fcslusb_capi_controller[0]);
			init_ctrl ();
			return res;
		}
	}
#endif
	fcslusb_capi_card->ctrl2 = fcslusb_capi_controller[1]->cnr;
	fcslusb_capi_card->ctxp = NULL;
	return 0;
} /* fcslusb_kcapi_init */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int fcslusb_stack_init (void) {

	if (!start_stack (fcslusb_capi_card)) {
		error ("Initialization failed.\n");
		return -EIO;
	}
	if (fcslusb_capi_controller[0] != NULL) {
		assert (fcslusb_capi_controller[0]->ready != NULL);
		(* fcslusb_capi_controller[0]->ready) (fcslusb_capi_controller[0]);
	}
	assert (fcslusb_capi_controller[1]->ready != NULL);
	(*fcslusb_capi_controller[1]->ready) (fcslusb_capi_controller[1]);
	return 0;
} /* fcslusb_stack_init */

/*---------------------------------------------------------------------------*\
\*-F-------------------------------------------------------------------------*/
int __entry fcslusb_load_ware (struct capi_ctr * ctrl, capiloaddata * ware) {
	card_p		cp;
	char *		buffer;
	int		res = 0;

	assert (ctrl != NULL);
	assert (ware != NULL);
	cp = GET_CARD(ctrl);
	if (load_flag) {
		log ("Firmware has already been loaded!\n");
		return 0;
	}

	assert (firmware_ptr == NULL);
	if (USB_MAX_FW_LENGTH < ware->firmware.len) {
		error ("Invalid firmware file.\n");
		return -EIO;
	}
	if (NULL == (buffer = (char *) hmalloc (ware->firmware.len))) {
		error ("Could not allocate firmware buffer.\n");
		return -EIO;
	}
        if (ware->firmware.user) {
		if (copy_from_user (buffer, ware->firmware.data, ware->firmware.len)) {
			error ("Error copying firmware data!\n");
			return -EIO;
		}
	} else {
		memcpy (buffer, ware->firmware.data, ware->firmware.len);
	}
	log ("Loaded %d bytes of firmware.\n", ware->firmware.len);
	firmware_ptr = buffer;
	firmware_len = ware->firmware.len;
	cp->c6205_ctx = fw_init (
				cp, 
				firmware_ptr, 
				firmware_len, 
				&res
			);
	if ((NULL == cp->c6205_ctx) || (res != 0)) {
		error ("Error while processing firmware (%d)!\n", res);
	error1:
		fw_exit (&cp->c6205_ctx);
		hfree (firmware_ptr);
		firmware_ptr = NULL;
		return -EIO;
	}
	assert (fcslusb_capi_lib != NULL);
	assert (fcslusb_capi_lib->cc_link_version != NULL);
	assert ((*fcslusb_capi_lib->cc_link_version) () > 1);
	if (!dif_init (cp)) {
		error ("Error while setting up device structures!\n");
	error2:
		goto error1;
	}

	fw_setup (cp->c6205_ctx);
	if (0 != fcslusb_stack_init ()) {
		error ("Error while starting protocoll stack!\n");
	/* error3: */
		dif_exit ();
		goto error2;
	}
	assert (load_flag == FALSE);
	load_flag = TRUE;
	return 0;
} /* fcslusb_load_ware */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int __entry fcslusb_add_card (struct capi_driver * drv, capicardparams * args) {

	UNUSED_ARG (drv);
	UNUSED_ARG (args);
	assert (fcslusb_capi_card != NULL);
	init_waitqueue_head (&wait);
	atomic_set (&resetting_ctrl, 0);
	return fcslusb_kcapi_init ();
} /* fcslusb_add_card */ 

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int remove_appls (card_p card) {
	appl_t *	appp;
	int		n = 0;

	if (0 != card->count) {
		log ("Removing registered applications.\n");
		assert (card->appls != NULL);
		if (card->appls != NULL) {
			appp = first_appl (card->appls);
			while (appp != NULL) {
				++n;
				fcslusb_release_appl (
					fcslusb_capi_controller[1],
					appp->id
				);
				appp = next_appl (card->appls, appp);
			}
		}
		kick_scheduler ();
	}
	return n;
} /* remove_appls */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry fcslusb_reset_ctrl (struct capi_ctr * ctrl) {

	DECLARE_WAIT_QUEUE_HEAD (appl_wait);
	
	UNUSED_ARG (ctrl);
	log ("Controller reset!\n");
	atomic_set (&resetting_ctrl, 1);

	if (!atomic_read (&closing_task_running)) {
		/* Manual reset? */
		(void) reset (fcslusb_capi_card->dev, NULL);
	}

	if (remove_appls (fcslusb_capi_card)) {
		init_waitqueue_head (&appl_wait);
		interruptible_sleep_on_timeout (&appl_wait, STACK_DELAY);
	}
	if (init_flag) {
		stop_stack (fcslusb_capi_card);
		/* to do: remove marked buffers */
	}
	if (ctrl_context[0].ctrl != NULL) {
		assert (fcslusb_capi_controller[0]->reseted);
		(*fcslusb_capi_controller[0]->reseted) (fcslusb_capi_controller[0]);
	}
	assert (fcslusb_capi_controller[1]->reseted);
	(*fcslusb_capi_controller[1]->reseted) (fcslusb_capi_controller[1]);

	dif_exit ();
	fw_exit (&fcslusb_capi_card->c6205_ctx);
	if (firmware_ptr != NULL) {
		hfree (firmware_ptr);
		firmware_ptr = NULL;
		firmware_len = 0;
	}
	load_flag = FALSE;
} /* fcslusb_reset_ctrl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry fcslusb_remove_ctrl (struct capi_ctr * ctrl) {

	UNUSED_ARG (ctrl);
	assert (fcslusb_capi_card != NULL);
	assert (fcslusb_capi_controller[1] != NULL);
	if (ctrl_context[0].ctrl != NULL) {
		(* fcslusb_capi_interface->detach_ctr) (fcslusb_capi_controller[0]);
	}
	(* fcslusb_capi_interface->detach_ctr) (fcslusb_capi_controller[1]);
	init_ctrl ();

#undef ASSIGN
	
} /* fdslusb_remove_ctrl */

/*---------------------------------------------------------------------------*\
\*-L-------------------------------------------------------------------------*/
void lock (void) {
	unsigned long	local_flags;
	
	info (!spin_is_locked (&qt_lock));
	spin_lock_irqsave (&qt_lock, local_flags);
	qt_flags = local_flags;
} /* lock */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void unlock (void) {
	unsigned long	local_flags = qt_flags;

	spin_unlock_irqrestore (&qt_lock, local_flags);
} /* unlock */

/*---------------------------------------------------------------------------*\
\*-C-------------------------------------------------------------------------*/
static inline void check (void) {

        if (atomic_read (&rx_flag) > 0) {
		rx_handler (fcslusb_capi_card);
	}
	if (atomic_read (&tx_flag) > 0) {
		tx_handler (fcslusb_capi_card);
	}
} /* check */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void enter_critical (void) {
	
	if (!atomic_read (&crit_count) && !in_interrupt ()) {
		check ();
	}
	atomic_inc (&crit_count);
} /* enter_critical */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void leave_critical (void) {
	
	assert (in_critical ());
	atomic_dec (&crit_count);
	if (!atomic_read (&crit_count) && !in_interrupt ()) {
		check ();
	}
} /* leave_critical */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry fcslusb_register_appl (
	struct capi_ctr *	ctrl, 
	u16			appl, 
	capi_register_params *	args
) {
	card_t *		card;
	appl_t *		appp;
	void *			ptr;
	unsigned		nc;

	mlog ("REGISTER(ctrl:%d,appl:%u)\n", ctrl->cnr, appl);
	assert (ctrl);
	assert (ctrl->driverdata);
	assert (args);
	card = GET_CARD (ctrl);
	if ((int) args->level3cnt < 0) {
		nc = nbchans (ctrl) * -((int) args->level3cnt);
	} else {
		nc = args->level3cnt;
	}
	if (0 == nc) {
		nc = nbchans (ctrl);
	}
	if (ctrl->cnr == card->ctrl2) {	
		appp = create_appl (
				card->appls, 
				appl, 
				nc, 
				args->datablkcnt, 
				args->datablklen
				);
		if (NULL == appp) {
			log ("Unable to create application record.\n");
			return;
		}
		ptr = hcalloc (card->length);		
		if (NULL == ptr) {
			lprintf (KERN_ERR, "Not enough memory for application data.\n");
			remove_appl (card->appls, appp);
			return;
		} else {
			inc_use_count ();
			card->count++;
			appp->data = ptr;
			assert (card->reg_func);
			(*card->reg_func) (ptr, appl);
		}
	}
	assert (ctrl->appl_registered);
	(*ctrl->appl_registered) (ctrl, appl);
} /* fcslusb_register_appl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry fcslusb_release_appl (struct capi_ctr * ctrl, u16 appl) {
	card_t *		card;
	struct sk_buff *	skb;
	appl_t *		appp;

	mlog ("RELEASE(ctrl:%d,appl:%u)\n", ctrl->cnr, appl);
	assert (ctrl);
	assert (ctrl->driverdata);
	card =  GET_CARD (ctrl);
	if (ctrl->cnr == card->ctrl2) {	
		if (NULL == (appp = search_appl (card->appls, appl))) {
			log ("Attempt to release unknown application (id #%u)\n", appl);
			return;
		}
		skb = make_0xfe_request (appl);	
		handle_message (card->queue, appp, skb);
		kick_scheduler ();
		appp->dying = TRUE;
	}
} /* fcslusb_release_appl */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void __entry fcslusb_send_msg (struct capi_ctr * ctrl, struct sk_buff * skb) {
	card_t *	card;
	unsigned char *	byte;
	unsigned	appl;
	appl_t *	appp;

	assert (ctrl != NULL);
	assert (ctrl->driverdata != NULL);
	assert (skb != NULL);
	assert (skb->list == NULL);
	card = GET_CARD (ctrl);
	byte = skb->data;
	appl = byte[2] + 256 * byte[3];
	appp = search_appl (card->appls, appl);
	if ((NULL == appp) || appp->dying) {
		log ("Message for unknown application (id %u).\n", appl);
		return;
	}
	handle_message (card->queue, appp, skb);
	kick_scheduler ();
} /* fcslusb_send_msg */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
char * __entry fcslusb_proc_info (struct capi_ctr * ctrl) {
	card_p		cp;
	per_ctrl_p	C;
	static char	text[80];

	assert (ctrl);
	assert (ctrl->driverdata);
	cp = GET_CARD (ctrl);
	C  = GET_CTRL (ctrl);
	SNPRINTF (
		text, 
		sizeof (text),
		"%s %s %d",
		C->version ? C->string[1] : "A1",
		C->version ? C->string[0] : "-",
		cp->dev->devnum
	);
	return text;
} /* fcslusb_proc_info */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int __entry fcslusb_ctr_info (
	char *			page, 
	char **			start, 
	off_t			ofs,
	int			count, 
	int *			eof,
	struct capi_ctr *	ctrl
) {
	card_p			cp;
	per_ctrl_p		C;
	char *			temp;
	unsigned char		flag;
	int			len = 0;

	assert (ctrl);
	assert (ctrl->driverdata);
	cp = GET_CARD (ctrl);
	C  = GET_CTRL (ctrl);
	pprintf (page, &len, "%-16s %s\n", "name", SHORT_LOGO);
	pprintf (page, &len, "%-16s %d\n", "dev", cp->dev->devnum);
	temp = C->version ? C->string[1] : "A1";
	pprintf (page, &len, "%-16s %s\n", "type", temp);
	temp = C->version ? C->string[0] : "-";
	pprintf (page, &len, "%-16s %s\n", "ver_driver", temp);
	pprintf (page, &len, "%-16s %s\n", "ver_cardtype", SHORT_LOGO);

	flag = ((unsigned char *) (ctrl->profile.manu))[3];
	if (flag) {
		pprintf(page, &len, "%-16s%s%s%s%s%s%s%s\n", "protocol",
			(flag & 0x01) ? " DSS1" : "",
			(flag & 0x02) ? " CT1" : "",
			(flag & 0x04) ? " VN3" : "",
			(flag & 0x08) ? " NI1" : "",
			(flag & 0x10) ? " AUSTEL" : "",
			(flag & 0x20) ? " ESS" : "",
			(flag & 0x40) ? " 1TR6" : ""
		);
	}
	flag = ((unsigned char *) (ctrl->profile.manu))[5];
	if (flag) {
		pprintf(page, &len, "%-16s%s%s%s%s\n", "linetype",
			(flag & 0x01) ? " point to point" : "",
			(flag & 0x02) ? " point to multipoint" : "",
			(flag & 0x08) ? " leased line without D-channel" : "",
			(flag & 0x04) ? " leased line with D-channel" : ""
		);
	}
	if (len < ofs) {
		return 0;
	}
	*eof = 1;
	*start = page - ofs;
	return ((count < len - ofs) ? count : len - ofs);
} /* fcslusb_ctr_info */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int __entry fcslusb_drv_info (
	char *			page, 
	char **			start, 
	off_t			ofs,
	int			count, 
	int *			eof, 
	struct capi_driver *	drv
) {
	int			len = 0;

	UNUSED_ARG (drv);
	pprintf (page, &len, "%-16s %s\n", "name", TARGET);
	pprintf (page, &len, "%-16s %s\n", "revision", fcslusb_capi_driver.revision);
	if (len < ofs) {
		return 0;
	}
	*eof = 1;
	*start = page - ofs;
	return ((count < len - ofs) ? count : len - ofs);
} /* fcslusb_drv_info */
 
/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void * data_by_id (unsigned appl_id) {
	appl_t * appp;

	appp = search_appl (fcslusb_capi_card->appls, appl_id);	
	return (appp != NULL) ? appp->data : NULL;
} /* data_by_id */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
unsigned attr_by_id (unsigned appl_id) {
	appl_t * appp;

	appp = search_appl (fcslusb_capi_card->appls, appl_id);	
	return (appp != NULL) ? appp->attr : 0;
} /* attr_by_id */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void * first_data (int * res) {
	appl_t * appp;

	assert (res);
	appp = first_appl (fcslusb_capi_card->appls);
	*res = (appp != NULL) ? 0 : -1;
	return (appp != NULL) ? appp->data  : NULL;
} /* first_data */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void * next_data (int * res) {
	appl_t * appp;

	assert (res);
	if (NULL != (appp = get_appl (fcslusb_capi_card->appls, *res))) {
		appp = next_appl (fcslusb_capi_card->appls, appp);
	}
	*res = (appp != NULL) ? 1 + *res : -1;
	return (appp != NULL) ? appp->data  : NULL;
} /* next_data */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int appl_profile (unsigned appl_id, unsigned * bs, unsigned * bc) {
	appl_t * appp;

	appp = search_appl (fcslusb_capi_card->appls, appl_id);	
	if (NULL == appp) {
		return 0;
	}
	*bs = appp->blk_size; 
	*bc = appp->blk_count; 
	return 1;
} /* appl_profile */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int msg2stack (unsigned char * msg) {
	unsigned		mlen;
	unsigned		appl;
	unsigned		nctr;
	__u32			ncci;
	unsigned		hand;
	unsigned char *		mptr;
	unsigned char *		dptr;
	struct sk_buff *	skb;
	unsigned		temp;
	int			res = 0;

	assert (msg);
	if (!queue_is_empty (fcslusb_capi_card->queue)) {
		res = 1;
		skb = queue_peek (fcslusb_capi_card->queue);	
		assert (skb);
		mptr = (unsigned char *) skb->data;
		mlen = mptr[0] + 256 * mptr[1]; 
		appl = mptr[2] + 256 * mptr[3];
		nctr = mptr[8] & 127;
		
		mlog (
			"PUT_MESSAGE(ctrl:%d,appl:%u,cmd:0x%02X,subcmd:0x%02X)\n", 
			nctr,
			appl, 
			mptr[4], 
			mptr[5]
		);

		if ((0x86 == mptr[4]) && (0x80 == mptr[5])) { /* DATA_B3_REQ */
			ncci = mptr[8] 
				+ 256 * (mptr[9] 
					+ 256 * (mptr[10] 
						+ 256 * mptr[11]
					)	);
			hand = mptr[18] + 256 * mptr[19];
			temp = (unsigned) (mptr + mlen);
			dptr = mptr + 12;
			*dptr++ = temp & 0xFF;	temp >>= 8;
			*dptr++ = temp & 0xFF;	temp >>= 8;
			*dptr++ = temp & 0xFF;	temp >>= 8;
			*dptr++ = temp & 0xFF;
			memcpy (msg, mptr, mlen);
			queue_park (fcslusb_capi_card->queue, appl, ncci, hand);
		} else {
			memcpy (msg, mptr, mlen);
			queue_drop (fcslusb_capi_card->queue);
		}
	}
	return res;
} /* msg2stack */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void msg2capi (unsigned char * msg) {
	unsigned		mlen; 
	unsigned		appl;
	unsigned		nctr;
	__u32			ncci;
	unsigned		hand;
	unsigned		dlen;
	unsigned char *		dptr;
	unsigned		temp;
	struct sk_buff *	skb;
	struct capi_ctr *	ctrl;

	assert (msg);
	mlen = msg[0] + 256 * msg[1];
	appl = msg[2] + 256 * msg[3];
	nctr = msg[8] & 127;

	mlog (
		"GET_MESSAGE(ctrl:%u,appl:%d,cmd:0x%02X,subcmd:0x%02X)\n", 
		nctr,
		appl, 
		msg[4], 
		msg[5]
	);
	if ((0x86 == msg[4]) && (0x81 == msg[5])) {	/* DATA_B3_CONF */
		hand = msg[12] + 256 * msg[13];
		ncci = msg[8] 
			+ 256 * (msg[9] 
				+ 256 * (msg[10] 
					+ 256 * msg[11]
				)	);
		queue_conf (fcslusb_capi_card->queue, appl, ncci, hand);
	}
	if ((0x86 == msg[4]) && (0x82 == msg[5])) {	/* DATA_B3_IND */
		dlen = msg[16] + 256 * msg[17];
		skb = alloc_skb (
				mlen + dlen + ((mlen < 30) ? (30 - mlen) : 0), 
				GFP_ATOMIC
				); 
		if (NULL == skb) {
			lprintf (
				KERN_ERR, 
				"Unable to build CAPI message skb. Message lost.\n"
			);
			return;
		}
		/* Messages are expected to come with 32 bit data pointers. 
		 * The kernel CAPI works with extended (64 bit ready) message 
		 * formats so that the incoming message needs to be fixed, 
		 * i.e. the length gets adjusted and the required 64 bit data 
		 * pointer is added.
		 */
		temp = msg[12] + 256 * (msg[13] 
					+ 256 * (msg[14] + 256 * msg[15]));
		dptr = (unsigned char *) temp;
		if (mlen < 30) {
			msg[0] = 30;
			ncci   = 0;	
			memcpy (skb_put (skb, mlen), msg, mlen);
			memcpy (skb_put (skb, 4), &ncci, 4);	
			memcpy (skb_put (skb, 4), &ncci, 4);
		} else {
			memcpy (skb_put (skb, mlen), msg, mlen);
        	}
		memcpy (skb_put (skb, dlen), dptr, dlen); 
	} else {
		if (NULL == (skb = alloc_skb (mlen, GFP_ATOMIC))) {
			lprintf (
				KERN_ERR, 
				"Unable to build CAPI message skb."
							" Message lost.\n"
			);
			return;
		}
		memcpy (skb_put (skb, mlen), msg, mlen);
	}
	ctrl = get_capi_ctr (nctr);
	info (ctrl != NULL);
	if (ctrl != NULL) {
		assert (ctrl->cnr == (int) nctr);
		assert (ctrl->handle_capimsg != NULL);
		(*ctrl->handle_capimsg) (ctrl, appl, skb);
	}
} /* msg2capi */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void new_ncci (
	unsigned		appl_id, 
	__u32			ncci,
	unsigned		winsize,
	unsigned		blksize
) {
	appl_t *		appp;
	ncci_t *		nccip;
	struct capi_ctr *	ctrl;
	unsigned		nctr;
	
	nctr = ncci & 127;
	mlog ("NEW NCCI(ctrl:%u,appl:%u,ncci:0x%08X)\n", nctr, appl_id, ncci);
	if (NULL == (appp = search_appl (fcslusb_capi_card->appls, appl_id))) {
		lprintf (KERN_ERR, invalid_msg, appl_id);
		return;
	}
	nccip = create_ncci (
			fcslusb_capi_card->appls, 
			appp, 
			(NCCI_t) ncci, 
			winsize, 
			blksize
			);
	if (NULL == nccip) {
		log ("Cannot handle new NCCI...\n");
		return;
	}
	ctrl = get_capi_ctr (nctr);
	assert (ctrl != NULL);	
	assert (ctrl->cnr == (int) nctr);
	assert (ctrl->new_ncci != NULL);
	(*ctrl->new_ncci) (ctrl, appl_id, ncci, winsize);
} /* new_ncci */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void free_ncci (unsigned appl_id, __u32 ncci) {
	appl_t *		appp;
	ncci_t *		nccip;
	struct capi_ctr *	ctrl;
	unsigned		nctr;
	
	appp = search_appl (fcslusb_capi_card->appls, appl_id);
	if (NULL == appp) {
		lprintf (KERN_ERR, invalid_msg, appl_id);
		return;
	}
	if (0xFFFFFFFF == ncci) {		/* 2nd phase RELEASE */
		assert (appp->dying);
		dec_use_count ();
		fcslusb_capi_card->count--;
		mlog ("FREE APPL(appl:%u)\n", appl_id);
		(*fcslusb_capi_card->rel_func) (appp->data);			
		remove_appl (fcslusb_capi_card->appls, appp);		
		appl_released_notify (appl_id);
		if (ctrl_context[0].ctrl != NULL) {
			assert (fcslusb_capi_controller[0]->appl_released);
			(*fcslusb_capi_controller[0]->appl_released) (fcslusb_capi_controller[0], appl_id);
		}
		assert (fcslusb_capi_controller[1]->appl_released);
		(*fcslusb_capi_controller[1]->appl_released) (fcslusb_capi_controller[1], appl_id);
	} else if (NULL != (nccip = locate_ncci (appp, ncci))) {
		nctr = ncci & 127;
		ctrl = get_capi_ctr (nctr);
		assert (ctrl);		
		assert (ctrl->cnr == (int) nctr);
		mlog ("FREE NCCI(ctrl:%u, appl:%u,ncci:0x%08X)\n", nctr,  appl_id, ncci);
		(*ctrl->free_ncci) (ctrl, appl_id, ncci);
		remove_ncci (fcslusb_capi_card->appls, appp, nccip);
	} else {
		lprintf (KERN_ERR, "Attempt to free unknown NCCI.\n");
	}
} /* free_ncci */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
unsigned char * data_block (unsigned appl_id, __u32 ncci, unsigned handle) {
	appl_t * appp;
 
	appp = search_appl (fcslusb_capi_card->appls, appl_id);
	if (NULL == appp) {
		lprintf (KERN_ERR, invalid_msg, appl_id);
		return NULL;
	}
	return ncci_data_buffer (fcslusb_capi_card->appls, appp, ncci, handle);
} /* data_block */

/*-S-------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void kick_scheduler (void) {

	if ((thread_pid != -1) && (NULL != find_task_by_pid (thread_pid))) {
		atomic_set (&scheduler_enabled, 1);
		wake_up_interruptible (&wait);
	} 
} /* kick_scheduler */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void scheduler_control (unsigned ena) {
	int	enabled = (int) ena;
	int	changed;

	enter_critical ();
	changed = (atomic_read (&scheduler_enabled) != enabled);
	leave_critical ();
	if (changed) {
		if (enabled) {
			kick_scheduler ();
		} else {
			atomic_set (&scheduler_enabled, 0);
			wake_up_interruptible (&wait);
		}
	}
} /* scheduler_control */

/*-S-------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int scheduler (void * arg) {

	UNUSED_ARG (arg);
	daemonize ();
	SNPRINTF (current->comm, 16, "%s_thread", TARGET);	/* See: sched.h */
	log ("Starting kicker thread '%s'...\n", current->comm);
	while (atomic_read (&thread_flag)) {
		/* Start/stop logic */
		if (!atomic_read (&thread_capi_flag)) {
			interruptible_sleep_on (&capi_wait);
		} else {
			if (atomic_read (&scheduler_enabled)) {
				interruptible_sleep_on_timeout (&wait, TEN_MSECS);
			} else {
				interruptible_sleep_on (&wait);
			}
		}

		assert (!in_critical ());
		/* Enable/disable logic */
		if (!atomic_read (&thread_flag)) {
			info (atomic_read (&thread_flag));
			break;
		}
		if (!atomic_read (&thread_capi_flag)) {
			info (atomic_read (&thread_capi_flag));
			continue;
		}
                if (!atomic_read (&scheduler_enabled)) {
			info (atomic_read (&scheduler_enabled));
			continue;
		}
		/* Body of thread, invoke scheduler */
		if (spin_trylock (&stack_lock)) {
			info (!atomic_xchg (&in_scheduler, 1));
			os_timer_poll ();
			assert (fcslusb_capi_lib->cm_schedule != NULL);
			if ((*fcslusb_capi_lib->cm_schedule) ()) {
				scheduler_control (TRUE); 
			}
			debug (atomic_set (&in_scheduler, 0));
			spin_unlock (&stack_lock);
		}
	}
	log ("Scheduler thread stopped.\n");
	up (&thread_sync);
	return 0;
} /* scheduler */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int make_thread (void) {

	if (-1 != thread_pid) {
		log ("Thread[%d] already running.\n", thread_pid);
		return thread_pid;
	}
	atomic_set (&thread_flag, 1);
	atomic_set (&thread_capi_flag, 0);
	init_waitqueue_head (&capi_wait);
	thread_pid = kernel_thread (scheduler, NULL, 0);
	log ("Thread[%d] started.\n", thread_pid);
	return thread_pid;
} /* make_thread */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void kill_thread (void) {

	atomic_set (&thread_flag, 0);
	if (NULL == find_task_by_pid (thread_pid)) {
		log ("Thread[%d] has died before!\n", thread_pid);
	} else {
		if (!atomic_read (&thread_capi_flag)) {
			wake_up_interruptible (&capi_wait);
		} else {
			wake_up_interruptible (&wait);
		}
		log ("Thread signalled, waiting for termination...\n");
		down (&thread_sync);
		log ("Thread[%d] terminated.\n", thread_pid);
	}
	thread_pid = -1;
} /* kill_thread */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void enable_thread (void) {

	info (!atomic_read (&thread_capi_flag));
	if (!atomic_read (&thread_capi_flag)) {
		atomic_set (&thread_capi_flag, 1);
		wake_up_interruptible (&capi_wait);
		log ("Thread enabled.\n");
	}
} /* enable_thread */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void disable_thread (void) {

	info (atomic_read (&thread_capi_flag));
	if (atomic_read (&thread_capi_flag)) {
		atomic_set (&thread_capi_flag, 0);
		wake_up_interruptible (&wait);
		log ("Thread disabled.\n");
	}
} /* disable_thread */

/*---------------------------------------------------------------------------*\
\*-T-------------------------------------------------------------------------*/ 
static void tx_handler (card_p pdc) {
	struct urb *		purb;
	void *			ptr;
	urb_context_p		pctx;
	hw_usb_request_p	urp;
	int			status;
	unsigned		alength;
	
	assert (pdc != NULL);
	atomic_dec (&tx_flag);
	if (!unlist_urb (pdc->tx_pool, &purb, &ptr)) {
		log ("TX completion w/o URB!\n");
		return;
	}

	info (!atomic_read (&f_stop_tx_pending));
	atomic_dec (&n_tx_pending);
	
	pctx = (urb_context_p) ptr;
	urp = (hw_usb_request_p) pctx->bdp->context;
	assert (urp != NULL);
	assert (pctx->buffer == (pctx->bdp->buffer + urp->offset));
	assert (atomic_xchg (&urp->in_use, 0));

	// <<<< enter_critical ();
	status = purb->status;
	alength = purb->actual_length;
	release_urb (pdc->tx_pool, purb);
	if (0 != purb->status) {
		log (
			"TX URB(%p,%d) status: %d\n", 
			purb,
			(int) purb->context,
			purb->status
		);
		dif_error (
			USB_EVENT_HARD_ERROR, 
			pctx->bdp,
			urp->offset,
			urp->length,
			urp->context
			);
	} else {
		assert (pctx->callback != NULL);
		(* pctx->callback) (
			pctx->bdp,
			urp->offset,
			purb->actual_length,
			urp->context
		);
	}
	// <<<< leave_critical ();
	kick_scheduler ();
} /* tx_handler */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void rx_handler (card_p pdc) {
	struct urb *		purb;
	void *			ptr;
	urb_context_p		pctx;
	hw_usb_request_p	urp;
	int			status;
	unsigned		alength;
	
	assert (pdc != NULL);
	atomic_dec (&rx_flag);
	if (!unlist_urb (pdc->rx_pool, &purb, &ptr)) {
		log ("Completion w/o URB!\n");
		return;
	}
	info (!atomic_read (&f_stop_rx_pending));
	atomic_dec (&n_rx_pending);
	
	pctx = (urb_context_p) ptr;
	urp = (hw_usb_request_p) pctx->bdp->context;
	assert (urp != NULL);
	assert (pctx->buffer == (pctx->bdp->buffer + urp->offset));
	assert (atomic_xchg (&urp->in_use, 0));

	// <<<< enter_critical ();
	status = purb->status;
	alength = purb->actual_length;
	release_urb (pdc->rx_pool, purb);
	if (0 != status) {
		log (
			"RX URB(%p,%d) status: %d\n", 
			purb,
			(int) purb->context,
			purb->status
		);
		dif_error (
			USB_EVENT_HARD_ERROR, 
			pctx->bdp,
			urp->offset,
			urp->length,
			urp->context
			);
	} else {
		assert (pctx->callback != NULL);
		(* pctx->callback) (
			pctx->bdp,
			urp->offset,
			alength,
			urp->context
		);
	}
	// <<<< leave_critical ();
	kick_scheduler ();
} /* rx_handler */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void tx_task (unsigned long data) {

	UNUSED_ARG (data);
	if (!in_critical () && spin_trylock (&stack_lock)) {
		tx_handler (fcslusb_capi_card);
		spin_unlock (&stack_lock);
	}
} /* tx_task */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void rx_task (unsigned long data) {

	UNUSED_ARG (data);
	if (!in_critical () && spin_trylock (&stack_lock)) {
		rx_handler (fcslusb_capi_card);
		spin_unlock (&stack_lock);
	}
} /* rx_task */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void wakeup_control (unsigned enable_flag) { 
	
	enable_flag = 0; 
} /* wakeup_control */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static void version_callback (char * vp) { 
	
	assert (nvers < 2);
	if ((0 == nvers) && (NULL == ctrl_context[0].ctrl)) {
		++nvers;
		log ("Version string #0 rejected.\n");
		return;
	}
	log ("Version string #%d:\n", nvers);
	scan_version (fcslusb_capi_controller[nvers], vp);
	copy_version (fcslusb_capi_controller[nvers]);
	++nvers;
} /* version_callback */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static unsigned	scheduler_suspend (unsigned long timeout) { 
	
	timeout = 0; 
	return 0; 
} /* scheduler_suspend */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static unsigned	scheduler_resume (void) { 
	
	return 0; 
} /* scheduler_resume */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static unsigned	controller_remove (void) { 

	assert (0);
	return 0; 
} /* controller_remove */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static unsigned	controller_add (void) { 
	
	assert (0);
	return 0; 
} /* controller_add */

/*---------------------------------------------------------------------------*\
\*-U-------------------------------------------------------------------------*/
static void issue_hard_error (card_p pdc, urb_context_p ctx, const char * msg, int arg) {
	hw_usb_request_p	urp;

	UNUSED_ARG(pdc);
	lprintf (KERN_INFO, msg, arg);
	if (!hard_error_issued) {
		hard_error_issued = TRUE;
		urp = (hw_usb_request_p) ctx->bdp->context;
		assert (urp != NULL);
		dif_error (
			USB_EVENT_HARD_ERROR, 
			ctx->bdp, 
			urp->offset,
			urp->length,
			urp->context
		);
	} else {
		lprintf (KERN_INFO, "Extra events ignored.\n");
	}
} /* issue_hard_error */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#if !defined (USE_SYSTEM_ZERO_CODE)
static struct urb * get_zero_urb (card_p pdc) {
	static struct urb	zero_urb;
	static int		zero_urb_ready = 0;
	
	if (!zero_urb_ready) {
		usb_fill_bulk_urb (
			&zero_urb, pdc->dev, 
			usb_sndbulkpipe (pdc->dev, pdc->epwrite->bEndpointAddress),
			0, 0, NULL, NULL
		);
		zero_urb_ready = 1;
	}
	return &zero_urb;
} /* get_zero_urb */
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void usb_write_completion (struct urb * purb) {

	assert (purb != NULL);
	atomic_inc (&tx_flag);
	enlist_urb (fcslusb_capi_card->tx_pool, purb);
	tasklet_schedule (&tx_tasklet);
} /* usb_write_completion */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void usb_read_completion (struct urb * purb) {
	
	assert (purb != NULL);
	atomic_inc (&rx_flag);
	enlist_urb (fcslusb_capi_card->rx_pool, purb);
	tasklet_schedule (&rx_tasklet);
} /* usb_read_completion */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
static int usb_write (card_p pdc, struct urb * purb, urb_context_p pctx, void * buffer, unsigned length) {
	int			res;
#if !defined (USE_SYSTEM_ZERO_CODE)
	static struct urb *	urb0;
	unsigned long		flags;
	static spinlock_t	order_lock = SPIN_LOCK_UNLOCKED;
#endif
	
	assert (pdc != NULL);
	assert (buffer != NULL);
	usb_fill_bulk_urb (
		purb,
		pdc->dev,
		usb_sndbulkpipe (pdc->dev, pdc->epwrite->bEndpointAddress),
		buffer,
		length,
		usb_write_completion,
		purb->context
		);

#if defined (USE_SYSTEM_ZERO_CODE)
	if ((length > 0) && ((length % 64) == 0)) {
		log ("Zero byte URB required!\n");
	}
	purb->transfer_flags |= USB_QUEUE_BULK | USB_ZERO_PACKET;
	res = usb_submit_urb (purb);
#else
	purb->transfer_flags |= USB_QUEUE_BULK;
	if ((length > 0) && ((length % 64) == 0)) {
		log ("Writing zero byte URB!\n");
		urb0 = get_zero_urb (pdc);
		spin_lock_irqsave (&order_lock, flags);
		if (0 != (res = usb_submit_urb (purb))) {
			res = usb_submit_urb (urb0);
		}
		spin_unlock_irqrestore (&order_lock, flags);
	} else {
		spin_lock_irqsave (&order_lock, flags);
		res = usb_submit_urb (purb);
		spin_unlock_irqrestore (&order_lock, flags);
	}
#endif
	if (0 != res) {
		atomic_dec (&n_tx_pending);
		issue_hard_error (pdc, pctx, "TX URB submission failed, code %d.\n", res);
		return FALSE;
	}
	return TRUE;
} /* usb_write */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int usb_read (card_p pdc, struct urb * purb, urb_context_p pctx, void * buffer, unsigned length) {
	int	res;

	assert (pdc != NULL);
	assert (buffer != NULL);
	usb_fill_bulk_urb (
		purb,
		pdc->dev,
		usb_rcvbulkpipe (pdc->dev, pdc->epread->bEndpointAddress),
		buffer,
		length,
		usb_read_completion,
		purb->context
		);
	
	purb->transfer_flags |= USB_QUEUE_BULK;
	if (0 != (res = usb_submit_urb (purb))) {
		atomic_dec (&n_rx_pending);
		issue_hard_error (pdc, pctx, "Rx URB submission failed, code %d.\n", res);
		return FALSE;
	}
	return TRUE;
} /* usb_read */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
int os_usb_submit_rx (
	int			pipe,
	void *			buf,
	unsigned		len,
	hw_completion_func_t	completion, 
	hw_buffer_descriptor_p	dp
) {
	struct urb *		purb;
	void *			ptr;
	urb_context_p		pctx;

	assert (pipe == fcslusb_capi_card->epread->bEndpointAddress);
	assert (fcslusb_capi_card->rx_pool != NULL);
	if (!claim_urb (fcslusb_capi_card->rx_pool, &purb, &ptr)) {
		log ("No RX URB/buffer available.\n");
		return FALSE;
	}
	assert (ptr != NULL);
	pctx = (urb_context_p) ptr;
	pctx->callback = completion;
	pctx->buffer = buf;
	pctx->bdp = dp;

	return usb_read (fcslusb_capi_card, purb, pctx, buf, len);
} /* os_usb_submit_rx */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
int os_usb_submit_tx (
	int			pipe,
	void *			buf,
	unsigned		len,
	hw_completion_func_t	completion, 
	hw_buffer_descriptor_p	dp
) {
	struct urb *		purb;
	void *			ptr;
	urb_context_p		pctx;

	assert (pipe == fcslusb_capi_card->epwrite->bEndpointAddress);
	assert (fcslusb_capi_card->tx_pool != NULL);
	if (!claim_urb (fcslusb_capi_card->tx_pool, &purb, &ptr)) {
		log ("No TX URB/buffer available.\n");
		return FALSE;
	}
	assert (ptr != NULL);
	pctx = (urb_context_p) ptr;
	pctx->callback = completion;
	pctx->buffer = buf;
	pctx->bdp = dp;

	return usb_write (fcslusb_capi_card, purb, pctx, buf, len);
} /* os_usb_submit_tx */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/ 
void init (
	unsigned len, 
	void  (* reg) (void *, unsigned),
	void  (* rel) (void *),
	void  (* dwn) (void)
) {
	assert (reg);
	assert (rel);
	assert (dwn);

	fcslusb_capi_card->length   = len;
	fcslusb_capi_card->reg_func = reg;
	fcslusb_capi_card->rel_func = rel;
	fcslusb_capi_card->dwn_func = dwn;
} /* init */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
